
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
//#define BRISTOL_DBG
/*
 * Need to have basic template for an operator. Will consist of
 *
 *	filterinit()
 *	operate()
 *	reset()
 *	destroy()
 *
 *	destroy() is in the library.
 *
 * Operate will be called when all the inputs have been loaded, and the result
 * will be an output buffer written to the next operator.
 */
#include <math.h>

#include "bristol.h"
#include "filter.h"

/*
 * The name of this operator, IO count, and IO names.
 */
#define OPNAME "DCF"
#define OPDESCRIPTION "Digital Filter One"
#define PCOUNT 4
#define IOCOUNT 3

#define FILTER_IN_IND 0
#define FILTER_MOD_IND 1
#define FILTER_OUT_IND 2

/*
 * Reset any local memory information.
 */
static int destroy(bristolOP *operator)
{
#ifdef BRISTOL_DBG
	printf("reset(%x)\n", operator);
#endif

	/*
	 * Unmalloc anything we added to this structure
	 */
	bristolfree(operator->specs);

	/*
	 * Free any local memory. We should also free ourselves, since we did the
	 * initial allocation.
	 */
	cleanup(operator);
}

#define ROOT2 1.4142135623730950488
/*
 * Reset any local memory information.
 */
static int reset(bristolOP *operator, bristolOPParams *param)
{
	float *a, c;
	double pidsr;

#ifdef BRISTOL_DBG
	printf("reset(%x)\n", operator);
#endif
	param->param[0].float_val = 0.5;
	param->param[1].float_val = 0.0;
	param->param[2].float_val = 0.5;
	param->param[3].int_val = 0;

	a = (float *) bristolmalloc(8 * sizeof(float));
	param->param[0].mem = a;

	pidsr = M_PI / sqrt(M_E);

	c = 1.0 / (float) tan((double) pidsr * 0.5);

	a[1] = 1.0 / (1.0 + ROOT2 * c + c * c);
	a[2] = 2 * a[1];
	a[3] = a[1];
	a[4] = 2.0 * (1.0 - c*c) * a[1];
	a[5] = (1.0 - ROOT2 * c + c * c) * a[1];

	a[6] = 0;
	a[7] = 0;
}

/*
 * Alter an internal parameter of an operator.
 */
static int param(bristolOP *operator, bristolOPParams *param,
	unsigned char index, float value)
{
#ifdef BRISTOL_DBG
	printf("param(%x, %f)\n", operator,	value);
#endif

	switch (index) {
		case 0:
			//param->param[index].float_val =
			//	gainTable[(int) (value * (CONTROLLER_RANGE - 1))].gain;
			if ((param->param[index].float_val = value) < (float) 0.000122)
				param->param[index].float_val = (float) 0.000122;
			break;
		case 1:
			param->param[index].float_val =
				gainTable[CONTROLLER_RANGE - 1
					- (int) (value * (CONTROLLER_RANGE - 1))].gain;
			break;
		case 2:
			param->param[index].float_val = value;
			break;
		case 3:
			if (value > 0)
				param->param[index].int_val = 1;
			else
				param->param[index].int_val = 0;
			break;
	}

printf("filterparam(%x, %x, %i, %f): %f [%f]\n", operator, param, index, value,
param->param[index].float_val, value * (CONTROLLER_RANGE - 1));
	return(0);
}

void butter_filter(float *in, float *out, float *a, int count) /* Filter loop */
{
    float t, y;
    long n = count;

    do {
      t = *in++ - a[4] * a[6] - a[5] * a[7];
      y = t * a[1] + a[2] * a[6] + a[3] * a[7];
      a[7] = a[6];
      a[6] = t;
      *out++ = y;
    } while (--n);
}

/*
 * filter - takes input signal and filters it according to the mod level.
 */
static int operate(register bristolOP *operator, bristolVoice *voice,
	bristolOPParams *param,
	void *lcl)
{
	bristolFILTERlocal *local = lcl;
	register int count;
	register float *ib, *ob, *mb;
	bristolFILTER *specs;
	register float BLim, Res, Mod, gain;
	register float Bout = local->Bout;
	register float oSource = local->oSource;

	register float velocity, output, ccut, cvdelta, cutoff;

	/*
	 * Every operator accesses these variables, the count, and a pointer to
	 * each buffer. We should consider passing them as readymade parameters?
	 *
	 * The Filter now takes normalised inputs, in ranges of 12PO.
	 */
	specs = (bristolFILTER *) operator->specs;
	count = specs->spec.io[FILTER_OUT_IND].samplecount;
	ib = specs->spec.io[FILTER_IN_IND].buf;
	mb = specs->spec.io[FILTER_MOD_IND].buf;
	ob = specs->spec.io[FILTER_OUT_IND].buf;

	/*
	 * This is the code from one of the SLab floating point filter routines.
	 * It has been altered here to have a single lowpass filter, and will be
	 * a starting point for the Bristol filters. There will end up being a
	 * number of filter algorithms.
	 */

	if (param->param[3].int_val) {
		if ((BLim = param->param[0].float_val + voice->dFreq / 128) > 1.0)
			BLim = 1.0;
		cutoff = param->param[0].float_val * voice->dFreq / 4;
		if ((Res = param->param[1].float_val * voice->dFreq / 4) == 0)
			Res = 0.000030;
		if (Res > 1.)
			Res = 1.0;
	} else {
		cutoff = BLim = param->param[0].float_val;
		Res = param->param[1].float_val;
	}
	if (voice->flags & BRISTOL_KEYON)
	{
		/*
		 * Do any relevant note_on initialisation.
		 */
		output = 0;
		velocity = 0;
		oSource = 0.0;
		Bout = 0.0;
	} else {
		output = local->output;
		velocity = local->velocity;
	}

	Mod = param->param[2].float_val;

#ifdef BRISTOL_DBG
	printf("filter(%x, %x, %x)\n", operator, param, local);
#endif

	butter_filter(ib, ob, param->param[0].mem, count);

//printf("BLim %f, gain %f\n", BLim, gain);
	/*
	 * Put back the local variables
	 */
	local->Bout = Bout;
	local->oSource = oSource;
	local->output = output;
	local->velocity = velocity;
}

/*
 * Setup any variables in our OP structure, in our IO structures, and malloc
 * any memory we need.
 */
bristolOP *
filterinit(bristolOP **operator, int index, int samplerate, int samplecount)
{
	bristolFILTER *specs;

#ifdef BRISTOL_DBG
	printf("filterinit(%x(%x), %i, %i, %i)\n",
		operator, *operator, index, samplerate, samplecount);
#endif

	*operator = bristolOPinit(operator, index, samplecount);

	/*
	 * Then the local parameters specific to this operator. These will be
	 * the same for each operator, but must be inited in the local code.
	 */
	(*operator)->operate = operate;
	(*operator)->destroy = destroy;
	(*operator)->reset = reset;
	(*operator)->param = param;

	specs = (bristolFILTER *) bristolmalloc0(sizeof(bristolFILTER));
	(*operator)->specs = (bristolOPSpec *) specs;
	(*operator)->size = sizeof(bristolFILTER);

	/*
	 * These are specific to this operator, and will need to be altered for
	 * each operator.
	 */
	specs->spec.opname = OPNAME;
	specs->spec.description = OPDESCRIPTION;
	specs->spec.pcount = PCOUNT;
	specs->spec.iocount = IOCOUNT;
	specs->spec.localsize = sizeof(bristolFILTERlocal);

	/*
	 * Now fill in the control specs for this filter.
	 */
	specs->spec.param[0].pname = "cutoff";
	specs->spec.param[0].description = "Filter cutoff frequency";
	specs->spec.param[0].type = BRISTOL_FLOAT;
	specs->spec.param[0].low = 0;
	specs->spec.param[0].high = 1;
	specs->spec.param[0].flags = BRISTOL_ROTARY|BRISTOL_SLIDER;

	specs->spec.param[1].pname = "resonance";
	specs->spec.param[1].description = "Filter emphasis";
	specs->spec.param[1].type = BRISTOL_FLOAT;
	specs->spec.param[1].low = 0;
	specs->spec.param[1].high = 1;
	specs->spec.param[1].flags = BRISTOL_ROTARY|BRISTOL_SLIDER;

	specs->spec.param[2].pname = "modulation";
	specs->spec.param[2].description = "Depth of modulation control";
	specs->spec.param[2].type = BRISTOL_FLOAT;
	specs->spec.param[2].low = 0;
	specs->spec.param[2].high = 1;
	specs->spec.param[2].flags = BRISTOL_ROTARY|BRISTOL_SLIDER;

	specs->spec.param[3].pname = "keyboard tracking";
	specs->spec.param[3].description = "Cutoff tracks keyboard";
	specs->spec.param[3].type = BRISTOL_TOGGLE;
	specs->spec.param[3].low = 0;
	specs->spec.param[3].high = 1;
	specs->spec.param[3].flags = BRISTOL_BUTTON;

	/*
	 * Now fill in the dco IO specs.
	 */
	specs->spec.io[0].ioname = "input";
	specs->spec.io[0].description = "Filter Input signal";
	specs->spec.io[0].samplerate = samplerate;
	specs->spec.io[0].samplecount = samplecount;
	specs->spec.io[0].flags = BRISTOL_AC|BRISTOL_INPUT;

	specs->spec.io[1].ioname = "mod";
	specs->spec.io[1].description = "Filter Control Signal";
	specs->spec.io[1].samplerate = samplerate;
	specs->spec.io[1].samplecount = samplecount;
	specs->spec.io[1].flags = BRISTOL_DC|BRISTOL_INPUT|BRISTOL_HIDE;

	specs->spec.io[2].ioname = "output";
	specs->spec.io[2].description = "Filter Output Signal";
	specs->spec.io[2].samplerate = samplerate;
	specs->spec.io[2].samplecount = samplecount;
	specs->spec.io[2].flags = BRISTOL_AC|BRISTOL_OUTPUT;

	return(*operator);
}

