
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * Some structures for device management. These primarily MIDI message control,
 * and eventually some for device management.
 */

#ifndef _BRISTOL_MIDI_API_H
#define _BRISTOL_MIDI_API_H

#include <time.h>

#include "bristolmessages.h"

extern int bristolMidiOpen();
extern int bristolMidiClose();
extern int bristolMidiRead();
extern int bristolMidiWrite();

#define MIDI_CONTROLLER_COUNT 128
#define CONTROLLER_RANGE 16384

/*
 * API Status codes
 */
#define BRISTOL_MIDI_OK 0
#define BRISTOL_MIDI_DEV -1
#define BRISTOL_MIDI_HANDLE -2
#define BRISTOL_MIDI_DEVICE -3
#define BRISTOL_MIDI_DRIVER -4
#define BRISTOL_MIDI_CHANNEL -5

#define BRISTOL_CONN_UNIX 0x10
#define BRISTOL_CONN_TCP 0x20
#define BRISTOL_CONN_MIDI 0x40
#define BRISTOL_CONN_OSSMIDI 0x080 /* As opposed to ALSA */
#define BRISTOL_CONN_SEQ 0x100 /* ALSA Sequencer interface */

#define BRISTOL_CONN_PASSIVE 0x1000 /* listen socket. */
#define BRISTOL_CONN_FORCE 0x2000

/*
 * API flags
 */
#define BRISTOL_RDONLY 1
#define BRISTOL_WRONLY 2
#define BRISTOL_DUPLEX 3

#define BRISTOL_ACCEPT_SOCKET 0x80000000
#define BRISTOL_CONTROL_SOCKET 0x40000000

/*
 * Midi command codes
 */
#define BRISTOL_MIDI_CHANNELS 16
#define BRISTOL_NOTE_ON (1 << (MIDI_NOTE_ON >> 4))
#define BRISTOL_NOTE_OFF (1 << (MIDI_NOTE_OFF >>4))

#define MIDI_COMMAND_MASK 0x0f0
#define MIDI_CHAN_MASK 0x0f

#define MIDI_STATUS_MASK	0x080
#define MIDI_NOTE_ON		0x090
#define MIDI_NOTE_OFF		0x080
#define MIDI_POLY_PRESS		0x0a0
#define MIDI_CONTROL		0x0b0
#define MIDI_PROGRAM		0x0c0
#define MIDI_CHAN_PRESS		0x0d0
#define MIDI_PITCHWHEEL		0x0e0

#define MIDI_ALL_NOTES_OFF	123
#define MIDI_OMNI_ON		124
#define MIDI_OMNI_OFF		125
#define MIDI_POLY_ON		126
#define MIDI_POLY_OFF		127

#define MIDI_SYSTEM		0x0f0
#define MIDI_SYSEX		0x0f8
#define MIDI_EOS		0x0f7

#define BRISTOL_REQ_ON		0x001
#define BRISTOL_REQ_OFF		0x002
#define BRISTOL_REQ_PP		0x004
#define BRISTOL_REQ_CONTROL	0x008
#define BRISTOL_REQ_PROGRAM	0x010
#define BRISTOL_REQ_CHAN	0x020
#define BRISTOL_REQ_PITCH	0x040
#define BRISTOL_REQ_SYSEX	0x080

#define BRISTOL_REQ_ALL		0x0ff
#define BRISTOL_REQ_NSX		0x07f // not sysex messages.......

typedef struct KeyMsg {
	unsigned char key;
	unsigned char velocity;
} keyMsg;

typedef struct ChanPressMsg {
	unsigned char pressure;
} chanPressMsg;

typedef struct PressureMsg {
	unsigned char key;
	unsigned char pressure;
} pressureMsg;

typedef struct PitchMsg {
	unsigned char lsb;
	unsigned char msb;
} pitchMsg;

typedef struct SysexMsg {
	int length;
	unsigned char *data;
} sysexMsg;

typedef struct ControlMsg {
	unsigned char c_id;
	unsigned char c_val;
} controlMsg;

typedef struct ProgMsg {
	unsigned char p_id;
} programMsg;

typedef struct BristolMidiMsg {
	unsigned char midiHandle;
	unsigned char channel;
	unsigned char mychannel;
	unsigned char command;
	time_t timestamp;
	union {
		keyMsg key;
		pressureMsg pressure;
		chanPressMsg channelpress;
		pitchMsg pitch;
		controlMsg controller;
		programMsg program;
		bristolMsg bristol;
		sysexMsg *sysex;
	} params;
} bristolMidiMsg;

#define DEF_TAB_SIZE 128

typedef struct TableEntry {
	float defnote;
	float rate;
	float gain;
} tableEntry;

/*
 * The remainder are actually specific to bristol, and could be moved to a
 * separate header file.
 */
tableEntry defaultTable[DEF_TAB_SIZE];
tableEntry gainTable[CONTROLLER_RANGE];

typedef int (*midiHandler)();

#endif // _BRISTOL_MIDI_API_H

