/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BSCommander is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BSCommander is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/
 
/*------- include files:
-------------------------------------------------------------------*/
#include "DirsDialog.h"
#include "InfoField.h"
#include "Shared.h"
#include "Config.h"
#include <qdir.h>
#include <qlabel.h>
#include <qlayout.h>
#include <algorithm>
#include <qcursor.h>
#include <qgroupbox.h>
#include <qcheckbox.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qapplication.h>
using namespace std;

/*------- local constants:
-------------------------------------------------------------------*/
const QString DirsDialog::ResultCaption   = QT_TR_NOOP( "Result" );
const QString DirsDialog::LeftDirLabel    = QT_TR_NOOP( "Left panel directory" );
const QString DirsDialog::RightDirLabel   = QT_TR_NOOP( "Right panel directory" );
const QString DirsDialog::BlockLeftLabel  = QT_TR_NOOP( "no update left panel" );
const QString DirsDialog::BlockRightLabel = QT_TR_NOOP( "no update right panel" );
const QString DirsDialog::OptionsLabel    = QT_TR_NOOP( "Options" );
const QString DirsDialog::ViewHdrCaption[ ColsNumber ] = {
	QT_TR_NOOP( "File from left panel" ),
	QT_TR_NOOP( "Result" ),
	QT_TR_NOOP( "File from right panel" )
};
const QString DirsDialog::NotEqual        = "!=";
const QString DirsDialog::Equal           = "=";
const QString DirsDialog::L2R             = ">>";
const QString DirsDialog::R2L             = "<<";
const QString DirsDialog::QuestMark       = "?";
const QString DirsDialog::CopyMark        = QT_TR_NOOP( "copied" );
const QString DirsDialog::BlockMark       = QT_TR_NOOP( "blocked" );
const QString DirsDialog::ErrorMark       = QT_TR_NOOP( "error" );
const QString DirsDialog::DirMark         = "DIR";


//*******************************************************************
// DirsDialog                                            CONSTRUCTOR
//*******************************************************************
DirsDialog::DirsDialog( QWidget* const  in_parent,
								const QString&  in_src_dir,
								const QString&  in_dst_dir,
								const Operation in_operation )
: QDialog        ( in_parent )
, d_main_layout  ( new QHBoxLayout( this ))
, d_wrk_layout   ( new QVBoxLayout )
, d_info_grpbox  ( new QGroupBox( 2, Qt::Horizontal, "", this ))
, d_src_dir_lbl  ( new QLabel( tr(LeftDirLabel), d_info_grpbox ))
, d_src_dir_path ( new InfoField( d_info_grpbox ))
, d_dst_dir_lbl  ( new QLabel( tr(RightDirLabel), d_info_grpbox ))
, d_dst_dir_path ( new InfoField( d_info_grpbox ))
, d_blocks_grpbox( new QGroupBox( 2, Qt::Horizontal, tr(OptionsLabel), this ))
, d_result_grpbox( new QGroupBox( 1, Qt::Vertical, tr(ResultCaption), this ))
, d_result_lv    ( new QListView( d_result_grpbox ))
, d_btn_layout   ( new QVBoxLayout )
, d_block_left   ( new QCheckBox( tr(BlockLeftLabel), d_blocks_grpbox ))
, d_block_right  ( new QCheckBox( tr(BlockRightLabel), d_blocks_grpbox ))
, d_operation    ( in_operation )
, d_src_dir      ( in_src_dir )
, d_dst_dir      ( in_dst_dir )
, d_run_btn      ( new QPushButton( tr(Shared::RunBtnLabel), this ))
, d_close_btn    ( new QPushButton( tr(Shared::CloseBtnLabel), this ))
, d_break        ( FALSE )
, d_processing   ( FALSE )
{
	setFont( Config::instance()->lfs_default_font() );
	if( Compare	== d_operation ) d_blocks_grpbox->hide();
	
	d_result_lv->setPaletteBackgroundColor( Config::instance()->lfs_default_bkg_color() );
	d_result_lv->setShowSortIndicator( TRUE );
	d_result_lv->setSelectionMode( QListView::Extended );
	d_result_lv->setAllColumnsShowFocus( TRUE );
	for( int i = 0; i < ColsNumber; ++i ) {
		d_result_lv->addColumn( tr(ViewHdrCaption[i]) );
	}
	d_result_lv->setColumnAlignment ( 0, Qt::AlignLeft );
	d_result_lv->setColumnAlignment ( 1, Qt::AlignHCenter );
	d_result_lv->setColumnAlignment ( 2, Qt::AlignLeft );
	
	d_main_layout->setMargin( Shared::LayoutMargin  );
	d_main_layout->setSpacing( Shared::LayoutSpacing );

	d_wrk_layout->addWidget( d_info_grpbox );
	d_wrk_layout->addWidget( d_blocks_grpbox );
	d_wrk_layout->addWidget( d_result_grpbox );
	d_wrk_layout->setStretchFactor( d_result_grpbox, Shared::OverStretch );
	d_main_layout->addLayout( d_wrk_layout );

	d_btn_layout->addStretch( Shared::OverStretch );
	d_btn_layout->addWidget( d_run_btn );
	d_btn_layout->addWidget( d_close_btn );
	d_main_layout->addLayout( d_btn_layout );

	connect( d_block_left , SIGNAL( toggled( bool ) ), this, SLOT( toggled()       ));
	connect( d_block_right, SIGNAL( toggled( bool ) ), this, SLOT( toggled()       ));
	connect( d_run_btn    , SIGNAL( clicked()       ), this, SLOT( run_clicked()   ));
	connect( d_close_btn  , SIGNAL( clicked()       ), this, SLOT( close_clicked() ));
}
// end of DirsDialog

//*******************************************************************
// show                                            PRIVATE inherited
//*******************************************************************
void DirsDialog::show()
{
	QDialog::show();
	
	const QFontMetrics fm( font() );
	QString dir;

	Shared::clip_path( fm, d_src_dir_path->width(), dir = d_src_dir );
	d_src_dir_path->setText( dir );
	Shared::clip_path( fm, d_dst_dir_path->width(), dir = d_dst_dir );
	d_dst_dir_path->setText( dir );
}
// end of show

//*******************************************************************
// polish                                          PRIVATE inherited
//*******************************************************************
void DirsDialog::polish()
{
	Shared::polish( this, 40, 60 );
}
// end of polish

//*******************************************************************
// set_info_title                                          PROTECTED
//*******************************************************************
void DirsDialog::set_info_title( const QString& in_title )
{
	d_info_grpbox->setTitle( in_title );
}
// end of set_info_title

//*******************************************************************
// add_item                                                PROTECTED
//*******************************************************************
void DirsDialog::add_item(	const QString& in_lft_name,
									const QString& in_mark,
									const QString& in_rgt_name,
									const QString& in_lft_path,
									const QString& in_rgt_path )
{
	new ViewItem(	d_result_lv,
						in_lft_name, in_mark, in_rgt_name,
						in_lft_path, in_rgt_path );
}
// end of add_item

//*******************************************************************
// toggled                                              PRIVATE slot
//*******************************************************************
void DirsDialog::toggled()
{
	d_run_btn->setDisabled( d_block_left->isChecked() && d_block_right->isChecked() );
}
// end of toggled

//*******************************************************************
// close_clicked                                        PRIVATE slot
//*******************************************************************
void DirsDialog::close_clicked()
{
	if( d_processing ) {
		d_break = TRUE;
		Shared::idle();
	}
	else {
		accept();
	}
}
// end of close_clicked

//*******************************************************************
// run_clicked                                          PRIVATE slot
//*******************************************************************
void DirsDialog::run_clicked()
{
	d_run_btn->setEnabled( FALSE );
	d_close_btn->setText( tr(Shared::BreakBtnLabel) );
	d_processing = TRUE;
	Shared::idle();

	QApplication::setOverrideCursor( Qt::WaitCursor );
	//..............................................
	reset();
	scanning( d_src_dir, d_dst_dir );
	adjust();
	//..............................................
	QApplication::restoreOverrideCursor();

	d_run_btn->setEnabled( TRUE );
	d_close_btn->setText( tr(Shared::CloseBtnLabel) );
	d_processing = FALSE;
	d_break = FALSE;
}
// end of run_clicked

//*******************************************************************
// keyPressEvent                                   PRIVATE inherited
//*******************************************************************
void DirsDialog::keyPressEvent( QKeyEvent* e )
{
	if( Qt::Key_Escape == e->key() ) {
		d_break = TRUE;
	}
}
// end of keyPressEvent

//*******************************************************************
// scanning                                                  PRIVATE
//*******************************************************************
void DirsDialog::scanning( const QString& lpath, const QString& rpath )
{
	Shared::idle();
	if( d_break ) return;

	QDir ldir( lpath, QString::null, QDir::Unsorted );
	QDir rdir( rpath, QString::null, QDir::Unsorted );
	
	if( ldir.exists() && ldir.isReadable() ) {
		if( rdir.exists() && rdir.isReadable() ) {
			const QFileInfoList* const lft_items = ldir.entryInfoList( QDir::All, QDir::Unsorted  );
			const QFileInfoList* const rgt_items = rdir.entryInfoList( QDir::All, QDir::Unsorted  );
			if( lft_items && rgt_items ) {
				scan_files( lft_items, lpath, rgt_items, rpath );
				scan_dirs ( lft_items, lpath, rgt_items, rpath );
			}
		}
	}
}
// end of compare

//*******************************************************************
// files                                                     PRIVATE
//-------------------------------------------------------------------
// Przepisanie informacji o plikach ze zwyklej listy plikow do
// posortowanej w/g nazw mapy.
//*******************************************************************
void DirsDialog::files( const QFileInfoList* const in_list, DataMap& in_v )
{
	Shared::idle();
	if( d_break ) return;

	QFileInfoListIterator it( *in_list );
	while( ( FALSE == d_break ) && it.current() ) {
		if( (*it)->isFile() ) {
			in_v.insert(  make_pair( (*it)->fileName(), *it ) );
		}
		Shared::idle();
		++it;
	}
}
// end of files

//*******************************************************************
// dirs                                                      PRIVATE
//-------------------------------------------------------------------
// Przepisanie informacji o katalogach ze zwyklej listy katalogow do
// posortowanej w/g nazw mapy.
//*******************************************************************
void DirsDialog::dirs( const QFileInfoList* const in_list, DataMap& in_v )
{
	Shared::idle();
	if( d_break ) return;

	QFileInfoListIterator it( *in_list );
	while( ( FALSE == d_break ) && it.current() ) {
		if( (*it)->isDir() ) {
			const QString name = (*it)->fileName();
			if( Shared::is_regular_file( name ) ) {
				in_v.insert(  make_pair( name, *it ) );
			}
		}
		Shared::idle();
		++it;
	}
}
// end of dirs

//*******************************************************************
// scan_files                                                PRIVATE
//*******************************************************************
void DirsDialog::scan_files(	const QFileInfoList* const in_lft_items,
										const QString&             in_lft_path,
										const QFileInfoList* const in_rgt_items,
										const QString&             in_rgt_path )
{
	Shared::idle();
	if( d_break ) return;
	
	bool lft_flag;
	bool rgt_flag;
	
	DataMap lft_files;
	DataMap rgt_files;
	files( in_lft_items, lft_files );
	files( in_rgt_items, rgt_files );

	DataMap total( lft_files );
	total.insert( rgt_files.begin(), rgt_files.end() );

	if( FALSE == total.empty() ) {
		DataMap::const_iterator lft_it;
		DataMap::const_iterator rgt_it;
		DataMap::const_iterator it = total.begin();
	
		while( ( FALSE == d_break ) && (it != total.end() )) {
			const QString name = it->first;
			//.............................................
			lft_it = lft_files.find( name );
			lft_flag = ( lft_it != lft_files.end() );
			//.............................................
			rgt_it = rgt_files.find( name );
			rgt_flag = ( rgt_it != rgt_files.end() );
			//.............................................
			emit file_do_it( name, lft_flag, rgt_flag, in_lft_path, in_rgt_path );
			++it;
		}
	}
}
// end of scan_files

//*******************************************************************
// scan_dirs                                                 PRIVATE
//*******************************************************************
void DirsDialog::scan_dirs(	const QFileInfoList* const in_lft_items,
										const QString&             in_lft_dir,
										const QFileInfoList* const in_rgt_items,
										const QString&             in_rgt_dir )
{
	Shared::idle();
	if( d_break ) return;

	bool lft_flag;
	bool rgt_flag;

	DataMap lft_dirs;
	DataMap rgt_dirs;
	dirs( in_lft_items, lft_dirs );
	dirs( in_rgt_items, rgt_dirs );

	DataMap total( lft_dirs );
	total.insert( rgt_dirs.begin(), rgt_dirs.end() );
				
	if( FALSE == total.empty() ) {
		DataMap::const_iterator lft_it;
		DataMap::const_iterator rgt_it;
		DataMap::const_iterator it = total.begin();
		
		while( ( FALSE == d_break ) && ( it != total.end() )) {
			const QString name = it->first;
			//.............................................
			lft_it = lft_dirs.find( name );
			lft_flag = ( lft_it != lft_dirs.end() );
			//.............................................
			rgt_it = rgt_dirs.find( name );
			rgt_flag = ( rgt_it != rgt_dirs.end() );
			//.............................................
			if( lft_flag && rgt_flag ) {
				const QString lft_path = ( "/" == in_lft_dir )	? in_lft_dir + name
																				: in_lft_dir + "/" + name;
				const QString rgt_path = ( "/" == in_rgt_dir )	? in_rgt_dir + name
																				: in_rgt_dir + "/" + name;
				scanning( lft_path, rgt_path );
			}
			else {
				emit dir_do_it( name, lft_flag, rgt_flag, in_lft_dir, in_rgt_dir );
			}
			++it;
		}
	}
}
// end of scan_dirs

//*******************************************************************
// adjust                                                    PRIVATE
//*******************************************************************
void DirsDialog::adjust()
{
	for( int col = 0; col < ColsNumber; ++col ) {
		d_result_lv->adjustColumn( col );
	}
}
// end of adjust

//*******************************************************************
// reset                                                     PRIVATE
//*******************************************************************
void DirsDialog::reset()
{
	QListViewItem* item = d_result_lv->firstChild();
	while ( item ) {
		d_result_lv->takeItem( item );
		item = d_result_lv->firstChild();
   }
}
// end of add_new_item

//###################################################################
//#                                                                 #
//#                        VIEW ITEM CLASS                          #
//#                                                                 #
//###################################################################


//*******************************************************************
// ViewItem::ViewItem
//*******************************************************************
DirsDialog::ViewItem::ViewItem( QListView* const parent,
		const QString& label1, const QString& label2, const QString& label3,
		const QString& in_lft_path, const QString& in_rgt_path )
: QListViewItem( parent, label1, label2, label3 )
, d_lft_path   ( in_lft_path )
, d_rgt_path   ( in_rgt_path )
{}
// end of ViewItem::ViewItem

//*******************************************************************
// paintCell                                       PRIVATE inherited
//*******************************************************************
void DirsDialog::ViewItem::paintCell( QPainter* p, const QColorGroup& cg, int col, int w, int align )
{
	QColorGroup new_cg  = cg;
	const QString mark  = text( 1 );
	
	if( mark.find( L2R ) != -1 ) {
		const QString right = text( 2 );	
		
		if( tr(CopyMark) == right ) {
			new_cg.setColor( QColorGroup::Base, QColor( 210, 210, 255 ) );
		}
		else if( tr(BlockMark) == right ) {
			new_cg.setColor( QColorGroup::Base, QColor( 255, 200, 200 ) );
		}
		else if( tr(ErrorMark) == right ) {
			new_cg.setColor( QColorGroup::Base, QColor( 255, 100, 100 ) );
		}
	}
	else if( mark.find( R2L ) != -1 ) {
		const QString left  = text( 0 );
		
		if( tr(CopyMark) == left ) {
			new_cg.setColor( QColorGroup::Base, QColor( 210, 255, 210 ) );
		}
		else if( tr(BlockMark) == left ) {
			new_cg.setColor( QColorGroup::Base, QColor( 255, 200, 200 ) );
		}
		else if( tr(ErrorMark) == left ) {
			new_cg.setColor( QColorGroup::Base, QColor( 255, 100, 100 ) );
		}
	}

	QListViewItem::paintCell( p, new_cg, col, w, align );
}
// end of paintCell

//*******************************************************************
// compare                                         PRIVATE inherited
//*******************************************************************
int DirsDialog::ViewItem::compare( QListViewItem* in_item, int in_column, bool ) const
{
	bool retval = 0;
	const ViewItem* const item = dynamic_cast<const ViewItem*>( in_item );
	return lft_path().localeAwareCompare( item->lft_path() );
		
	if( 0 == in_column ) {
		QString my_dir, my_name;
		Shared::clip_path( lft_path(), my_dir, my_name );
		QString another_dir, another_name;
		Shared::clip_path( item->lft_path(), another_dir, another_name );
	
		if( my_dir == another_dir ) retval = my_name.localeAwareCompare( another_name );
		else                        retval = my_dir.localeAwareCompare ( another_dir  );
	}
	
	return retval;
}
// end of compare
