# Copyright (C) 2005, 2006 by Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
"""Add functionality for controlling a Patch Queue Manager (pqm).

This adds a few configuration options. All of these
are accessible at the Branch level.

# The email address of the Patch Queue Manager
pqm_email = pqm@pqm.ubuntu.com

# The pqm managed branch to merge these changes into
pqm_branch = http://bazaar-ng.org/bzr/bzr.dev/

# The branch we will be submitting to the pqm
# This is custom because the publically accessible location
# is frequently not this or parent, because they usually
# need write access, and the public location is readonly
# The preferred submission locations are:
#   1) public_branch
#   2) push_location
#   3) parent
#   4) this.base
public_branch = None

# This is the smtp server that should be used for
# outgoing mail. If not set, or set to the empty
# string, then the email will be sent directly to
# the pqm machine
smtp_server = localhost
# You can also use 
# smtp_server = host:port 
"""

import os

from bzrlib.commands import Command, register_command
from bzrlib.option import Option
import bzrlib.errors as errors

from bzrlib.bzrdir import BzrDir


class cmd_pqm_submit(Command):
    """Submit the parent tree to the pqm.

    This acts like:
        $ echo "star-merge $PARENT $TARGET" 
            | gpg --cl 
            | mail pqm@somewhere -s "merge text"

    But it pays attention to who the local committer is 
    (using their e-mail address), and uses the local
    gpg signing configuration. (As well as target pqm
    settings, etc.)

    The reason we use 'parent' instead of the local branch
    is that most likely the local branch is not a public
    branch. And the branch must be available to the pqm.
    """

    takes_args = ['location?']
    takes_options = ['message',
                     Option('dry-run', help='Print request instead of sending'),
                     Option('public-location', type=str,
                         help='Use this url as the public location to the pqm.'),
                    ]

    def run(self, location=None, message=None, public_location=None, dry_run=False):
        from pqm_submit import submit

        if location is None:
            location = '.'
        # If we are simply specifying '.' then
        # we want to open the local branch.
        # all other uses require an exact path
        if location == '.':
            bzrdir = BzrDir.open_containing('.')[0]
        else:
            bzrdir = BzrDir.open(location)

        b = bzrdir.open_branch()
        submit(b, message=message, dry_run=dry_run, public_location=public_location)


register_command(cmd_pqm_submit)


def test_suite():
    from bzrlib.tests import TestSuite, TestLoader
    import test_pqm_submit

    loader = TestLoader()
    return loader.loadTestsFromModule(loader)
