/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2005 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */
package org.freedesktop.cairo;

import java.io.OutputStream;

import org.gnu.glib.Handle;

/**
 * The Surface class represents an Cairo drawing surface. A Cairo object is used
 * to draw on the drawing surface.
 */
public class Surface extends CairoObject {

    Surface(Handle hndl) {
        super(hndl);
    }

    protected void finalize() throws Throwable {
        cairo_surface_destroy(getHandle());
        super.finalize();
    }

    public Surface createSimilar(Content content, int width, int height) {
        return new Surface(cairo_surface_create_similar(getHandle(), content
                .getValue(), width, height));
    }

    public void finish() {
        cairo_surface_finish(getHandle());
    }

    public void setDeviceOffset(double xOffset, double yOffset) {
        cairo_surface_set_device_offset(getHandle(), xOffset, yOffset);
    }

    public void writeToPNG(String filename) {
        cairo_surface_write_to_png(getHandle(), filename);
    }

    public void writeToPNG(OutputStream ostream) {
        cairo_surface_write_to_png_stream(getHandle(), ostream);
    }

    public FontOptions getFontOptions() {
        return new FontOptions(cairo_surface_get_font_options(getHandle()));
    }

    public void flush() {
        cairo_surface_flush(getHandle());
    }

    public void markDirty() {
        cairo_surface_mark_dirty(getHandle());
    }

    public void markDirty(int x, int y, int width, int height) {
        cairo_surface_mark_dirty_rectangle(getHandle(), x, y, width, height);
    }

    /*
     * Native calls
     */
    native static final private Handle cairo_surface_create_similar(
            Handle other, int format, int width, int height);

    native static final private void cairo_surface_destroy(Handle obj);

    native static final private void cairo_surface_finish(Handle obj);

    native static final private int cairo_surface_write_to_png(Handle obj,
            String filename);

    native static final private int cairo_surface_write_to_png_stream(
            Handle obj, OutputStream ostream);

    native static final private void cairo_surface_set_device_offset(
            Handle obj, double xOff, double yOff);

    native static final private Handle cairo_surface_get_font_options(Handle obj);

    native static final private void cairo_surface_flush(Handle obj);

    native static final private void cairo_surface_mark_dirty(Handle obj);

    native static final private void cairo_surface_mark_dirty_rectangle(
            Handle obj, int x, int y, int width, int height);

}
