(* $Id: rpc_client.ml 185 2004-07-13 13:09:04Z gerd $
 * ----------------------------------------------------------------------
 *
 *)

open Rtypes
open Xdr
open Rpc
open Rpc_common
open Rpc_packer
open Unixqueue

exception Message_not_processable


(* The following exceptions are delivered to the callback function: *)

exception Message_lost
  (* got EOF when some procedure calls were not replied or not even sent *)

exception Message_timeout
  (* After all retransmissions, there was still no reply *)

exception Communication_error of exn
  (* an I/O error happened *)

exception Client_is_down
  (* The RPC call cannot be performed because the client has been shut down
   * in the meantime. You can get this exception if you begin a new call,
   * but the connection is closed now.
   *)

exception Keep_call

exception Stop_retransmission_timer of Unixqueue.wait_id


module SessionInt = struct
  type t = int
  let compare = (Pervasives.compare : int -> int -> int)
end

module SessionMap =
  Map.Make(SessionInt)


type call_state =
    Waiting    (* call has not yet been sent *)
  | Pending    (* call has been sent, still no answer *)
  | Done       (* got answer for the call *)

(* Normally, the state of the call is changed from Waiting to Pending to
 * Done.
 * In the case of a retransmission, the call is added to the waiting calls
 * again but its state remains 'Pending' (because the call is still member
 * of the set of pending calls).
 *)


(* The following class types are only preliminary definitions. The type
 * parameter 't is later instantiated with the type of the clients, t.
 *)

class type ['t] pre_auth_session =
object
  method next_credentials : 't -> (string * string * string * string)
  method server_rejects : server_error -> unit
  method server_accepts : string -> string -> unit
end


class type ['t] pre_auth_method =
object
  method name : string
  method new_session : unit -> 't pre_auth_session
end


type call =
      { mutable proc : string;
	mutable xdr_value : xdr_value;      (* the argument of the call *)
	mutable value : packed_value;       (* the argument of the call *)
	mutable get_result : (unit -> xdr_value) -> unit;

	mutable state : call_state;
	mutable retrans_count : int;        (* retransmission counter *)
	mutable xid : int;

	mutable call_timeout : float;
	mutable timeout_id : wait_id option;
	  (* If a timeout handler has been set, this is the corresponding ID *)

	mutable call_auth_session : t pre_auth_session;
	mutable call_auth_method : t pre_auth_method;
	  (* calls store the authentication session and the method. *)
      }

and t =
      { mutable ready : bool;

        mutable trans : Rpc_transport.t;
	mutable prog :  Rpc_program.t;
	mutable prot :  protocol;
        mutable esys :  event_system;
        mutable group : group;

	mutable close : unit -> unit;
	mutable shutdown_connector : t -> connector -> Unix.file_descr -> unit;

	mutable waiting_calls : call Queue.t;
	mutable pending_calls : call SessionMap.t;

	mutable next_xid : int;

	mutable has_output_resource : bool;
	mutable has_input_resource : bool;

	mutable within_callback : bool;

	(* configs: *)
	mutable timeout : float;
        mutable max_retransmissions : int;

	(* authentication: *)
	mutable auth_methods : t pre_auth_method list;     (* methods to try *)
	mutable current_auth_method : t pre_auth_method;
	mutable unused_auth_sessions : t pre_auth_session list;
	  (* The [unused_auth_sessions] are the sessions that were used for
	   * previous calls and that can be reused. These sessions belong to
	   * [current_auth_method].
	   *)

	mutable exception_handler : exn -> unit;
      }

and connector =
    Inet of (string * int)                        (* Hostname, port *)
  | InetNumeric of (int * int * int * int * int)  (* IP addr, port *)
  | Unix of string                                (* path to unix dom sock *)
  | Descriptor of Unix.file_descr
  | Dynamic_descriptor of (unit -> Unix.file_descr)
  | Dynamic_descriptor' of ((unit -> Unix.file_descr) *
			    (t -> Unix.file_descr -> unit))

class type auth_session = [t] pre_auth_session
class type auth_method = [t] pre_auth_method


class auth_none_session : auth_session =
object
  method next_credentials _ = ("AUTH_NONE", "", "AUTH_NONE", "")
  method server_rejects _ = ()
  method server_accepts _ _ = ()
end


class auth_none =
object
  method name = "AUTH_NONE"
  method new_session () = new auth_none_session
end

let auth_none = new auth_none

let debug = ref false

  (*****)

let set_auth_methods cl list =
  match list with
      m :: list' ->
	cl.current_auth_method <- m;
	cl.auth_methods <- list';
	cl.unused_auth_sessions <- []
    | [] ->
	invalid_arg "Rpc_client.set_auth_methods"

  (*****)

let stop_retransmission_timer cl call =
  (* Schedules a special event that will stop the timer in the near future *)
  match call.timeout_id with
      None -> ()
    | Some id ->
	call.timeout_id <- None;
	Unixqueue.add_event
	  cl.esys
	  (Unixqueue.Extra (Stop_retransmission_timer id))
;;

  (*****)

let pass_result cl call f =

  (* Stop the timer, if any : *)

  stop_retransmission_timer cl call;

  (* Change the state of the call to 'Done': *)

  call.state <- Done;

  (* pass 'f' to the call back function: *)

  try
    cl.within_callback <- true;
    call.get_result f;
    cl.within_callback <- false
  with
    | Keep_call as x ->
	cl.within_callback <- false;
	raise x
    | any ->
	begin  (* pass the exception to the exception handler: *)
	  cl.within_callback <- false;
	  try
	    cl.exception_handler any
	  with
	      _ -> ()
	end


let pass_exception cl call x =
  (* Caution! This function does not remove [call] from the set of pending
   * calls.
   *)
  try
    pass_result cl call (fun () -> raise x)
  with
      Keep_call -> ()          (* ignore *)


let pass_exception_to_all cl x =
  (* Caution! This function does not erase the set of pending calls.  *)
  let fn_list = ref [] in
  let add_fn xid call =
    if not (List.mem_assoc xid !fn_list) then
      fn_list := (xid,call) :: !fn_list
  in

  SessionMap.iter (fun xid call -> add_fn xid call)  cl.pending_calls;
  Queue.iter      (fun call -> add_fn call.xid call) cl.waiting_calls;

  List.iter (fun (xid,call) -> pass_exception cl call x) !fn_list

  (*****)

let rtd = Rpc_transport.descriptor;;  (* abbreviation *)

let is_queue_empty q =
  try ignore(Queue.peek q); false with Queue.Empty -> true
;;

let check_resources cl =
  let cl_d = rtd cl.trans in
  (* OUTPUT *)
  if not(is_queue_empty cl.waiting_calls) && not cl.has_output_resource then begin
    (* add a resource if there was previously no pending call *)
    if !debug then prerr_endline "Adding output resource";
    add_resource cl.esys cl.group (Wait_out cl_d, (-.1.0));
    cl.has_output_resource <- true
  end;
  if  cl.has_output_resource &&                      (* C1 *)
     (is_queue_empty cl.waiting_calls) &&            (* C2 *)
     Rpc_transport.is_sending_complete cl.trans      (* C3 *)
  then begin
    (* Note: we need condition C3 here, because it is possible that the queue
     * is empty, but there is still material to send in cl.trans.
     *)
    if !debug then prerr_endline "Removing output resource";
    remove_resource cl.esys cl.group (Wait_out cl_d);
    cl.has_output_resource <- false
  end;
  (* INPUT *)
  if cl.pending_calls <> SessionMap.empty && not cl.has_input_resource
  then begin
    if !debug then prerr_endline "Adding input resource";
    add_resource cl.esys cl.group (Wait_in cl_d, (-.1.0));
    cl.has_input_resource <- true;
  end;
  if cl.pending_calls = SessionMap.empty && cl.has_input_resource then begin
    if !debug then prerr_endline "Removing input resource";
     remove_resource cl.esys cl.group (Wait_in cl_d);
    cl.has_input_resource <- false
  end
;;


let find_or_make_auth_session cl =
  match cl.unused_auth_sessions with
      [] ->
	cl.current_auth_method # new_session()
    | s :: other ->
	cl.unused_auth_sessions <- other;
	s
;;


(* Note: For asynchronous authentication, it would be sufficient that
 * add_call (and add_call_again) are rewritten such that they first
 * schedule the authentication request, and when the request is replied,
 * the call is scheduled.
 *)

let add_call cl procname param receiver =
  if not cl.ready then
    raise Client_is_down;

  let s = find_or_make_auth_session cl in

  let (cred_flav, cred_data, verf_flav, verf_data) = s # next_credentials cl in

  let value =
    Rpc_packer.pack_call
      cl.prog
      (uint4_of_int cl.next_xid)
      procname
      cred_flav cred_data verf_flav verf_data
      param
  in

  let new_call =
    { proc = procname;
      xdr_value = param;
      value = value;
      get_result = receiver;
      state = Waiting;
      retrans_count = cl.max_retransmissions;
      xid = cl.next_xid;
      call_timeout = cl.timeout;
      timeout_id = None;
      call_auth_session = s;
      call_auth_method = cl.current_auth_method
    }
  in

  Queue.add new_call cl.waiting_calls;
  cl.next_xid <- cl.next_xid + 1;

  check_resources cl
;;


let add_call_again cl call =
  (* Add a call again to the queue of waiting calls. The call is authenticated
   * again.
   *)

  if not cl.ready then
    raise Client_is_down;

  let s = call.call_auth_session in

  let (cred_flav, cred_data, verf_flav, verf_data) = s # next_credentials cl in

  let value =
    Rpc_packer.pack_call
      cl.prog
      (uint4_of_int call.xid)                   (* use old XID again (CHECK) *)
      call.proc
      cred_flav cred_data verf_flav verf_data
      call.xdr_value
  in

  call.value <- value;           (* the credentials may have changed *)
  call.state <- Waiting;
  call.retrans_count <- cl.max_retransmissions;
  call.timeout_id <- None;

  Queue.add call cl.waiting_calls;

  check_resources cl
;;

  (*****)

let remove_pending_call cl call =
  cl.pending_calls <- SessionMap.remove call.xid cl.pending_calls;
  stop_retransmission_timer cl call
;;

  (*****)

let retransmit cl call =
  if call.state = Pending then begin
    if call.retrans_count > 0 then begin
      if !debug then prerr_endline "Retransmitting...";
      (* Make the 'call' waiting again *)
      Queue.add call cl.waiting_calls;
      (* Decrease the retransmission counter *)
      call.retrans_count <- call.retrans_count - 1;
      (* Check state of reources: *)
      check_resources cl
      (* Note: The [call] remains in state [Pending]. This prevents the [call]
       * from being added to [cl.pending_calls] again.
       *)
    end
    else begin
      (* still no answer after maximum number of retransmissions *)
      if !debug then prerr_endline "Call timed out!";
      remove_pending_call cl call;
      pass_exception cl call Message_timeout;
      (* Note: The call_auth_session is dropped. *)
      (* Check state of reources: *)
      check_resources cl
    end
  end


  (*****)


let process_incoming_message cl message =

    (* Got a 'message' for which the corresponding 'call' must be searched: *)

    let xid =
      try
	int_of_uint4 (Rpc_packer.peek_xid message)
      with
	_ -> raise Message_not_processable
	    (* TODO: shut down the connection. This is a serious error *)
    in

    let call =
      try
	SessionMap.find xid cl.pending_calls
      with
	Not_found ->
	  (* Strange: Got a message with a session ID that is not pending.
	   * We assume that this is an answer of a very old message that
	   * has been completely timed out.
	   *)
	  raise Message_not_processable
    in
    assert(call.state = Pending);

    (* Exceptions in the following block are forwarded to the callback
     * function
     *)

    let result_opt = try
      begin match Rpc_packer.peek_auth_error message with
	( Some Auth_rejected_cred
	| Some Auth_rejected_verf
	| Some Auth_bad_cred
	| Some Auth_bad_verf) as erropt ->
	    (* Automatic retry with the same auth_session *)
	    let error = match erropt with Some x -> x | _ -> assert false in
	    call.call_auth_session # server_rejects error;
              (* may raise an exception *)
	    remove_pending_call cl call;
	    add_call_again cl call;
	    None                    (* don't pass a value back to the caller *)
	| Some Auth_too_weak ->
	    (* Automatic retry with next auth_method *)
	    if call.call_auth_method = cl.current_auth_method then begin
	      (* Switch to next best authentication method *)
	      match cl.auth_methods with
		  a :: other ->
		    cl.auth_methods <- other;
		    cl.current_auth_method <- a;
		    cl.unused_auth_sessions <- []    (* drop all old sessions *)
		| [] ->
		    (* No further authentication method. Keep the
		     * current method, but raise Auth_too_weak
		     *)
		    raise (Rpc_server Auth_too_weak)
	    end;
	    (* else: in the meantime the method has already been
	     * switched
	     *)
	    add_call cl call.proc call.xdr_value call.get_result;
	    None                     (* don't pass a value back to the caller *)
	| _ ->
	    let (xid,verf_flavour,verf_data,response) =
	      Rpc_packer.unpack_reply cl.prog call.proc message
		(* may raise an exception *)
            in
	    call.call_auth_session # server_accepts verf_flavour verf_data;
	    Some (fun () -> response)
      end
    with
	error ->
	  (* The call_auth_session is simply dropped. *)
	  (* Forward the exception [error] to the caller: *)
	  Some (fun () -> raise error)
    in

    match result_opt with
	None ->
	  (* There is no result yet *)
	  ()

      | Some result ->

          (* pass result to the user *)

	  try
	    pass_result cl call result;      (* may raise Keep_call *)
	    (* Side effect: Changes the state of [call] to [Done] *)
	    remove_pending_call cl call;
	    cl.unused_auth_sessions <-
	                       call.call_auth_session ::cl.unused_auth_sessions;
	  with
	      Keep_call ->
		call.state <- Pending


  (*****)

let prerr_event ev =
  match ev with
    Input_arrived d -> prerr_endline "Input_arrived"
  | Output_readiness d -> prerr_endline "Output_readiness"
  | Out_of_band d -> prerr_endline "Out_of_band"
  | Timeout (g,Wait_in d) -> prerr_endline "Timeout Wait_in"
  | Timeout (g,Wait_out d) -> prerr_endline "Timeout Wait_out"
  | Timeout (g,Wait_oob d) -> prerr_endline "Timeout Wait_oob"
  | Timeout (g,Wait id) -> prerr_endline "Timeout Wait"
  | Signal  -> prerr_endline "Signal"
  | Extra _ -> prerr_endline "Extra"
;;

let event_handler cl esys esys' ev =
    (* DEBUG *)
    if !debug then prerr_event ev;

    if cl.within_callback then
      failwith "Rpc_client: Cannot handle events from callback function";

    let cl_d = rtd cl.trans in
    match ev with

      (*** event: input data have been arrived ***)

      Input_arrived(_,d) ->
	if d <> cl_d then raise (Equeue.Reject);

	(* Check on End-of-file condition: *)

	if Rpc_transport.at_eof cl.trans then begin
	  if !debug then prerr_endline "EH: eof";

	  if cl.has_input_resource then
	    remove_resource cl.esys cl.group (Wait_in cl_d);
	  cl.has_input_resource <- false;

	  if cl.has_output_resource then
	    remove_resource cl.esys cl.group (Wait_out cl_d);
	  cl.has_output_resource <- false;

	  pass_exception_to_all cl Message_lost;

	  cl.pending_calls <- SessionMap.empty;
	  Queue.clear cl.waiting_calls;

	  cl.close();
	  cl.ready <- false;

	  ()
	end
	else begin

	  (* Receive message parts until there is a complete message *)

	  let again = ref true in

	  while !again do

	    if !debug then prerr_endline "EH: receive_part";

	    (* CHECK: Are there system exception that are better caught
	     * here? If they are not caught, the whole client will be
	     * shut down.
	     *)
	    ignore (Rpc_transport.receive_part cl.trans);

  	    (* Is the message complete? Yes: process it *)

	    if Rpc_transport.is_message_complete cl.trans then begin
	      if !debug then prerr_endline "EH: got message";
	      let message = Rpc_transport.get cl.trans in
	      Rpc_transport.clean_input cl.trans;

	      try
		process_incoming_message cl message
	      with
		Message_not_processable ->
		  (* DEBUG *)
		  if !debug then prerr_endline "EH: message not processable";
		  ()
	    end;

	    again := not (Rpc_transport.is_buffer_empty cl.trans);

	  done;

	  (* (don't) clean up:
	   *   Rpc_transport.clean_input cl.trans;
	   * -- is incorrect here because if the message is large, it is
	   * possible that event has only processed the beginning of the
	   * message.
	   *)

	  (* All calls processed ? *)

	  check_resources cl

	end

      (*** event: ready to output data ***)

    | Output_readiness(_,d) ->
	if d <> cl_d then raise (Equeue.Reject);

	if Rpc_transport.is_sending_complete cl.trans then begin
	  (* send next call *)

	  (* DEBUG *)
	  if !debug then prerr_endline "EH: ready to send new message...";

	  let proceed = ref true in

	  while !proceed do
	    try
	      (* try to get the next waiting call from the queue; may raise
	       * Queue.Empty:
	       *)
	      let call = Queue.take cl.waiting_calls in

	      (* If the call is already 'Done', skip it. *)
  	      (* Change the state of the call. It is now 'pending': *)

	      ( match call.state with
		    Done ->
		      raise Not_found;
		  | Waiting ->
		      cl.pending_calls <-
		        SessionMap.add call.xid call cl.pending_calls;
		      call.state <- Pending;
		  | Pending ->
		      ()
		      (* The call is already member of [pending_calls]
		       * (retransmitted)
		       *)
	      );

	      (* Fill the transport container: *)

	      Rpc_transport.put cl.trans call.value;
	      if !debug then prerr_endline "Rpc_transport.put";

	      (* Add a new resource for the answer: *)

	      check_resources cl;

	      (* If there should be a timeout handler, add it: *)

	      if call.call_timeout >= 0.0 then begin
		let id = new_wait_id cl.esys in
		let op = Wait id in
		let handler ues ev e =
		  if e = Timeout(cl.group,op) && call.timeout_id = Some id
		  then begin
		    if !debug then prerr_endline "Timeout handler";
		    remove_resource ues cl.group op;
		    retransmit cl call;
		    call.timeout_id <- None;
		    raise Equeue.Terminate
		  end
		  else if e = Extra(Stop_retransmission_timer id) then begin
		    if !debug then prerr_endline "Term timeout handler";
		    remove_resource ues cl.group op;
		    raise Equeue.Terminate
		  end
		  else
		    raise Equeue.Reject
		in
		add_resource cl.esys cl.group (op, call.call_timeout);
		add_handler cl.esys cl.group handler;
		call.timeout_id <- Some id
	      end;

	      proceed := false

	    with Queue.Empty ->
	      (* The queue was empty: Do some cleanups. *)

	      (* DEBUG *)
	      if !debug then prerr_endline "EH: no calls";

	      Rpc_transport.clean_output cl.trans;

(* CHECK: What was the idea of the "robustness"?
	      if cl.has_output_resource then
		remove_resource cl.esys cl.group (Wait_out (rtd cl.trans))
	      else begin
		(* robustness *)
		if exists_resource cl.esys (Wait_out (rtd cl.trans)) then (
		  if !debug then
		    prerr_endline "Rpc_client: out of sync with internal state";
		  failwith "out of sync";
		)
	      end;
	      cl.has_output_resource <- false;
*)
	      check_resources cl;

	      proceed := false

	    | Not_found ->
		(* proceed with the next queue member, if any *)

		(* DEBUG *)
		if !debug then
		  prerr_endline "EH: found call that has been done";

		()

	  done
	end;

	if not (Rpc_transport.is_sending_complete cl.trans) then begin
	  (* send next part of calling message *)

	  (* DEBUG *)
	  if !debug then prerr_endline "EH: send_part";

	  ignore (Rpc_transport.send_part cl.trans);
	  (* CHECK: Are there system exception that are better caught
	   * here? If they are not caught, the whole client will be
	   * shut down.
	   *)
	  ()
	end


    | _ -> raise (Equeue.Reject)

  (*****)

let shutdown_connector cl c d =
  match c with
      Descriptor _ -> ()
    | Dynamic_descriptor'(_,f) ->
	begin try
	  (* DEBUG *)
	  if !debug then prerr_endline "Calling shutdown function...";
	  f cl d
	with
	    _ -> ()
	end
    | _ ->
	begin try
	  (* DEBUG *)
	  if !debug then prerr_endline "Closing connection...";
	  Unix.close d
	with
	    _ -> ()
	end


let create ?program_number ?version_number ?(initial_xid=0) 
           ?(shutdown = shutdown_connector)
           esys c prot prog0 =
    (* open connection *)

    let prog = Rpc_program.update ?program_number ?version_number prog0 in

    let open_inet_socket host port prot =
      let sv_addr =
    	try
	  Unix.inet_addr_of_string host
    	with
	  Failure s ->
            try
              let h = Unix.gethostbyname host in
              h.Unix.h_addr_list.(0)
            with
              Not_found ->
            	failwith ("Rpc_client.create: unknown host " ^ host)
      in

      let sock =
      	Unix.socket
	  Unix.PF_INET
	  (if prot = Tcp then Unix.SOCK_STREAM else Unix.SOCK_DGRAM)
	  0
      in

      try
	Unix.connect sock (Unix.ADDR_INET (sv_addr, port));
	sock
      with
	any ->
	  Unix.close sock;
	  raise any
    in

    let descr =
      match c with
	Inet (host,port) ->
	  open_inet_socket host port prot
      |	InetNumeric (a,b,c,d,port) ->
	  open_inet_socket (Printf.sprintf "%d.%d.%d.%d" a b c d) port prot
      |	Unix path ->
	  let s =
      	    Unix.socket
	      Unix.PF_UNIX
	      (if prot = Tcp then Unix.SOCK_STREAM else Unix.SOCK_DGRAM)
	      0
	  in
	  begin try
	    Unix.connect s (Unix.ADDR_UNIX path);
	    s
	  with
	    any ->
	      Unix.close s;
	      raise any
	  end
      |	Descriptor d -> d
      |	Dynamic_descriptor f -> f()
      |	Dynamic_descriptor'(f,_)  -> f()
    in

    try
      let trans = Rpc_transport.create descr prot BiPipe in

      (* TODO: add_close_action. What should happen if there are unprocessed
       * calls ?
       *)

      let cl =
	{ ready = true;
	  trans = trans;
      	  prog = prog;
      	  prot = prot;
      	  esys = esys;
	  group = new_group esys;
	  close = (fun () -> ());
	  shutdown_connector = shutdown;
      	  waiting_calls = Queue.create();
	  pending_calls = SessionMap.empty;
      	  next_xid = initial_xid;
	  has_input_resource = false;
	  has_output_resource = false;
	  timeout = if prot = Udp then 15.0 else (-.1.0);
	  max_retransmissions = 3;
	  exception_handler = (fun _ -> ());
	  within_callback = false;
	  auth_methods = [ ];
	  current_auth_method = auth_none;
	  unused_auth_sessions = []
	}
      in

      let close_connection () =
	cl.pending_calls <- SessionMap.empty;
	Queue.clear cl.waiting_calls;
	cl.ready <- false;
	cl.shutdown_connector cl c descr;
      in

      cl.close <- close_connection;

      let protected_event_handler esys esys' ev =
	try
	  (* DEBUG *)
	  if !debug then prerr_endline "trying event handling...";
	  event_handler cl esys esys' ev;
	  if !debug then prerr_endline "...successful";
	with
	  Equeue.Reject as x ->
	    (* DEBUG *)
	    if !debug then prerr_endline "...rejecting";
	    raise x
	| Equeue.Terminate as x ->
	    (* DEBUG *)
	    if !debug then prerr_endline "...terminating";
	    raise x
	| Abort(g,x') as x ->
	    (* DEBUG *)
	    if !debug then prerr_endline "...aborting";
	    raise x
	| any ->
	    (* DEBUG *)
	    if !debug then
	      prerr_endline ("EXCEPTION: " ^ Printexc.to_string any);

	    raise (Abort(cl.group, any))
      in

      add_handler esys cl.group protected_event_handler;

      (* add an 'abort' handler: *)

      let abort g x =
	(* run through all waiting and pending callbacks and deliver 'x' *)
	(* DEBUG *)
	if !debug then prerr_endline "aborting...";
	pass_exception_to_all cl x;
	cl.pending_calls <- SessionMap.empty;
	Queue.clear cl.waiting_calls;
	close_connection();
	cl.ready <- false;
      in

      add_abort_action esys cl.group abort;

      cl
    with
      any ->
	begin match c with
	  Descriptor _ -> raise any
	| _            -> Unix.close descr; raise any
	end

  (*****)

let configure cl max_retransmission_trials timeout =
  cl.max_retransmissions <- max_retransmission_trials;
  cl.timeout <- timeout


let set_exception_handler cl xh =
  cl.exception_handler <- xh

let shut_down cl =
  if cl.ready then begin
    cl.close();
    pass_exception_to_all cl Message_lost;
    (* DEBUG *)
    if !debug then prerr_endline "clearing client";
    clear cl.esys cl.group
  end


let event_system cl =
  cl.esys

let program cl =
  cl.prog

let get_socket_name cl =
  Unix.getsockname (rtd (cl.trans))

let get_peer_name cl =
  Unix.getpeername (rtd (cl.trans))

let get_protocol cl =
  cl.prot

let verbose b =
  debug := b

  (*****)

(* Now synchronous calls: *)

type result =
    No
  | Reply of xdr_value
  | Error of exn


let sync_call cl proc arg =
  let r = ref No in
  let get_result transmitter =
    try
      r := Reply (transmitter())
    with
      x ->
	r := Error x
  in
  (* push the request onto the queue: *)
  add_call cl proc arg get_result;
  (* run through the queue and process all elements: *)
  Unixqueue.run cl.esys;
  (* now a call back of 'get_result' should have happened. *)
  match !r with
    No -> failwith "Rpc_client.sync_call: internal error"
  | Reply x -> x
  | Error e -> raise e


