/*
 * download.c
 *
 * Copyright (C) 2003 Bastian Blank <waldi@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $LastChangedBy: bastian $
 * $LastChangedDate: 2006-10-20 15:26:44 +0000 (Fr, 20 Okt 2006) $
 * $LastChangedRevision: 1275 $
 */

#include <config.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "check.h"
#include "execute.h"
#include "frontend.h"
#include "download.h"
#include "packages.h"
#include "prepare.h"

int decompress_file (const char *file, const char *extension)
{
  char buf[1024];

  snprintf (buf, sizeof (buf), "gunzip -c %s.%s > %s", file, extension, file);

  return execute (buf);
}

static inline int download_file (const char *source, const char *target, const char *message)
{
  log_message (LOG_MESSAGE_INFO_DOWNLOAD_RETRIEVE, message);
  return frontend_download (source, target);
}

static int download_release (di_release **rel)
{
  char source[256];
  char target[4096];

  build_indices_root ("Release.gpg", source, sizeof (source), target, sizeof (target));

  if (download_file (source, target, "Release.gpg"))
    log_message (LOG_MESSAGE_ERROR_DOWNLOAD, "Release.gpg");

  build_indices_root ("Release", source, sizeof (source), target, sizeof (target));

  if (download_file (source, target, "Release"))
    log_message (LOG_MESSAGE_ERROR_DOWNLOAD, "Release");

  log_message (LOG_MESSAGE_INFO_DOWNLOAD_PARSE, "Release");

  if (!(*rel = di_release_read_file (target)))
    log_message (LOG_MESSAGE_ERROR_PARSE, "Release");

  build_indices ("Release", source, sizeof (source), target, sizeof (target));

  if (download_file (source, target, "Release"))
    log_message (LOG_MESSAGE_ERROR_DOWNLOAD, "Release");

  return 0;
}

static int download_packages_parse (const char *target, di_packages **packages, di_packages_allocator *allocator)
{
  log_message (LOG_MESSAGE_INFO_DOWNLOAD_PARSE, "Packages");

  if ((*packages = di_packages_minimal_read_file (target, allocator)))
    return 0;

  return 1;
}

static int download_packages (di_release *rel, di_packages **packages, di_packages_allocator *allocator)
{
  char source[256];
  char target[4096];
  bool ok = false;
  struct stat statbuf;

  build_indices ("Packages.gz", source, sizeof (source), target, sizeof (target));
    
  /* yeah, the file already exists */
  if (!stat (target, &statbuf))
  {
    /* if it is invalid, unlink them */
    if (check_packages (target, ".gz", rel))
      unlink (target);
    else
      ok = true;
  }

  if (!ok)
  {
    /* try to download the gzip compressed version ... */
    if (download_file (source, target, "Packages.gz"))
      goto error;
    if (check_packages (target, ".gz", rel))
      goto error;
  }

  build_indices ("Packages", 0, 0, target, sizeof (target));

  /* ... decompress it ... */
  if (decompress_file (target, "gz"))
    log_message (LOG_MESSAGE_ERROR_DECOMPRESS, "Packages.gz");

  /* ... and parse them */
  download_packages_parse (target, packages, allocator);
  if (*packages)
    return 0;

error:
  log_message (LOG_MESSAGE_ERROR_DOWNLOAD, "Packages");
  return 1;
}

static int download_indices (di_packages **packages, di_packages_allocator *allocator)
{
  di_release *rel;
  int ret;

  ret = download_release (&rel);
  if (ret)
    return 1;

  ret = suite_init_second (rel->codename);
  if (ret)
    return 1;

  frontend_progress_set (10);

  ret = download_packages (rel, packages, allocator);
  if (ret)
    return 1;

  frontend_progress_set (50);
  return 0;
}

static int download_debs (di_slist *install)
{
  int count = 0, ret, size = 0, size_done = 0, progress;
  struct stat statbuf;
  di_slist_node *node;
  di_package *p;
  char target[4096];

  for (node = install->head; node; node = node->next)
  {
    p = node->data;
    count++;
    size += p->size;
  }

  for (node = install->head; node; node = node->next)
  {
    p = node->data;
    size_done += p->size;
    progress = ((float) size_done/size) * 350 + 50;

    build_target_deb (target, sizeof (target), package_get_local_filename (p));

    if (!stat (target, &statbuf))
    {
      ret = check_deb (target, p, p->package);
      if (!ret)
      {
        frontend_progress_set (progress);
        continue;
      }
    }

    if (download_file (p->filename, target, p->package) || check_deb (target, p, p->package))
      log_message (LOG_MESSAGE_ERROR_DOWNLOAD, p->filename);

    frontend_progress_set (progress);
  }

  return 0;
}

int download (di_packages **packages, di_packages_allocator **allocator, di_slist **install)
{
  int ret;

  *allocator = di_packages_allocator_alloc ();
  if (!*allocator)
    return 1;

  ret = download_indices (packages, *allocator);
  if (ret)
    return 1;

  (*install) = suite_packages_list (*packages, *allocator);
  if (!*install)
    log_message (LOG_MESSAGE_ERROR_INSTALL_LIST);

  ret = download_debs (*install);
  if (ret)
    return 1;

  return 0;
}

int download_init (void)
{
  return prepare_download ();
}

