/*
 * Cheops Network User Interface
 *
 * Copyright (C) 1999, Adtran, Inc.
 * 
 * Distributed under the terms of the GNU GPL
 *
 */

#ifndef __PLUGIN_H
#define __PLUGIN_H

/* General */
#define MAX_SYMS 10

#define TYPE_SNMP 1
#define TYPE_SERVICE 2
#define TYPE_MONITOR 3
#define TYPE_PROBE 4

/* Services */
#define SERVICE_CONNECT 0

/* Monitors */
#define MONITOR_DEFAULT_LEVEL 0
#define MONITOR_MONITOR 1
#define MONITOR_CLEANUP 2
#define MONITOR_SETUP 3
#define MONITOR_SETUP_CLEANUP 4
#define MONITOR_SETUP_STR 5
#define MONITOR_STR_SETUP 6
#define MONITOR_TIMEOUT_CB 7

/* Probes */
#define PROBE_GETPORT 0
#define PROBE_PROBE 1

struct plugin {
	struct plugin *next;	/* For easy linking */
	char name[40];		/* Name of plugin */
	char label[40]; 	/* Label to display (if applicable) */
	int type;		/* Type of plugin (see above, TYPE_* */
	void *syms[MAX_SYMS];	/* Pointer to symbols, see above */
};

/* All plugins must export the following symbols:

	char *plugin_name();		Return name of plugin 
	int plugin_type();		Return type of plugin (see above)
	char *plugin_label();		Return convenient label for menu

   Service plugins must export:
   	int service_connect(struct service *svc, 
	                    char *hostname, char *ip, char *username, int port)	;
			    
			    		Connect to ip (host) using
					username, and port, and formatting in
					service structure
   
   SNMP plugins must export:
       int can_handle(struct net_object *no);
       
       					Returns TRUE if this module can properly
					manage the device no.  Use the
					sysObjectID field, etc to determine
					that.

       void launch(struct net_object *no);
       
       					Do whatever you do on this network
					object.
   		
   Monitoring plugins must export:
       int default_level();
       					The default level a failure should
					be considered as
       
       void monitor(struct net_object *, void **data, void *setupdata);
       
       					net_object * is a pointer to the object
					to be monitored.
					
					data is a pointer to a pointer which
					is intially null, so the monitor can
					store arbitrary code on a per-object
					basis (file descriptors, for example).
					
					setupdata is any setup data which
					had been assicated with the object
       
       void cleanup(void **data);
       
       					this is the same data as above, but
					after an object is no longer being
					monitored, this is called to cleanup
					any allocation you might have done
       
       the monitor operation should *not* block, and upon success or failure
       should call:
       
       monitor_report(struct net_object *, struct net_page *np, int status, char *service, char *msg);
       
       					should be called, where net_object is the
					original object, np was its page, status is
					an integer reporting TRUE or FALSE depending
					on whether it is nominal, service is the
					name of this monitor, and msg is some kind
					of text representation of the alert status 
	
	Optionally, a monitor plugin can provide a routine:
	
	void *setup(void *oldsetup, struct net_object *no);
	
					which provides a pointer to some private
					structure containing setup information
					for a given host and
					
	void setup_cleanup(void *setup);
	
					which deallocates that private structure
					if setup_cleanup is not specified,
					g_free will be called on setupdata
			
	char *setup2str(void *setup);
					convert a setup structure into a simple
					string for inclusion in a config file
					
	void *str2setup(char *buf, struct net_object *no);
					convert a string (as created above)
					to a setup structure

	Additionally, monitors may provide a routine:
	void timeout_callback(struct network_object *no, void **data, int expired);
					
					which is called when the timeout has
					expired.  If (and only if) this routine
					does not exist, a "timeout" will be
					automatically logged.  The expired flag
					is non-zero if this is the last acceptable
					failure.

	Probe routines provide:
	
	GList *get_port();		Port(s) to probe
	
	char *probe(unsigned int addr, unsigned short port);
					return the version of software or
					any other information running at
					a given address on a given port

*/

extern int load_plugins();

/* Retrieve critical symbol from plugin */
extern void *get_plugin(char *name, int type, int symbol);

extern char *str_plugin(void *data, int type, int symbol);

extern GList *list_plugin(int type);

extern char *label_plugin(char *name);

#endif
