/*
 * Distributor.cpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright (c) 2005-2006  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  @file Distributor.cpp
   
  Utility for communication between several classes. See "Distributor.hpp".
  
  @NOTE printf() seems to be unsuitable for output of function pointers:
   `printf("%p",fp)' gives the warning: non-void* converts to void*, and 
   `printf("%p",(void*)fp)' gives the warning: ISO C++ forbids converting
   of function pointers to object pointers. On the other side, `cout << fp'
   gave (g++ 3.3.4) a simple "1" even for different function pointers? A bug?
   So I ended at the moment with `cout << (void*)fp' and the above ISO C++ 
   warning.
*/

#include <iostream>
#include "Distributor.hpp"

//#define BR_DEBUG_DISTRIBUTOR

using std::cout;

//====================
//  DistributorBase...
//====================
/**
*   login()  -  login a participant. 
* 
*   @param fp: pointer of the callback function, which shall be called.
*   @param userdata: data, which shall be given as second argument in
*      the callback function.
* 
*   Notes: 
*    - Double logins of identic (fp,userdata)-pairs are not prevented; would 
*       lead to (ordinar undangerous) multiple callback calls.
*    - Several participants can have identic fp's if static class element
*       functions are used (all instances have here the same fp!)
*    - Several participants can have identic userdata if not the this-pointer
*       is used, but something others.
*
*/
void DistributorBase::login (Callback* fp, void* userdata)
{
    array_.push_back (Entry(fp,userdata));

#ifdef BR_DEBUG_DISTRIBUTOR    
    cout << "DistributorBase::" <<__func__<< "( fp=" << (void*)fp 
         << ", userdata=" << userdata << " ): new size()=" << array_.size() << '\n';      
    //report(0);
#endif
}


/**
*   Logout a participant identified by its callback pointer `fp'.
* 
*   @param fp: function pointer as it was logged in (should be unique in array)
* 
*   Note: Call with non-registered or already removed Callbacks harmless
*/  
void DistributorBase::logout (Callback* fp)
{
    for (unsigned i=0; i < array_.size(); i++) {
      if (array_[i].fp == fp) {
        array_.erase (array_.begin()+i);

#ifdef BR_DEBUG_DISTRIBUTOR        
        cout << "DistributorBase::"<<__func__<<"( fp="<< (void*)fp 
             << " ): new size()=" << array_.size()<<'\n';
#endif 
        return;
      }
    }
    cout << "DistributorBase::" <<__func__<< "(callback=" << (void*)fp 
         << ": not found): size()=" << array_.size()<<'\n';
}


/**
*   Logout a participant identified by its userdata.
* 
*   @param userdata: userdata as it was logged in (should be unique in array)
* 
*   Note: Call with non-registered or already removed Users harmless
*/  
void DistributorBase::logout (void* userdata)
{
    for (unsigned i=0; i < array_.size(); i++) {  
      if (array_[i].userdata == userdata) {
        array_.erase (array_.begin()+i);

#ifdef BR_DEBUG_DISTRIBUTOR        
        cout << "DistributorBase::"<<__func__<<"( userdata=" << userdata
             << " ): new size()=" << array_.size() << '\n';
#endif 
        return;
      }
    }
    cout << "DistributorBase::"<<__func__<<"( userdata=" << userdata
         << ": not found ): new size()=" << array_.size() << '\n';
}


/**
*   distribute() -  Calls the callback functions `fp(pData,userdata)' for all
*    logged in participants with `pData' and the deposited `userdata' as second
*    argument.
* 
*   @param pData: void pointer to the data ("info"), which is to distribute
*/

void DistributorBase::distribute (const void* pData)
{
    for (size_t i=0; i < array_.size(); i++)
      array_[i].fp (pData, array_[i].userdata);
}


/**
*   report() -  List current array of callbacks and userdata on console.
* 
*   @param label: Prefix of the report title, e.g. "Event"
*/  
void DistributorBase::report (const char* label) const
{
    cout << '\"' << label << "\"-Distributor-Report:\n";   
    
    for (size_t i=0; i < array_.size(); i++)
      cout << '\t' << i << ": callback = " << (void*)array_[i].fp 
           << ",  userdata = " << array_[i].userdata << '\n';
}


// END OF FILE
