/*
 * testtools.cpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright (c) 2005-2006  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  testtools.cpp
*/
#include "testtools.hpp"
#include "Br2HdrManager.hpp"
#include "br_Image.hpp"           // Image
#include "br_Image_utils.hpp"     // fillBuffer_...()
#include "br_macros.hpp"          // NOT_IMPL_IMAGE_TYPE()


namespace br {

/**+*************************************************************************\n
  Add set of images with values 0,1,2... 
******************************************************************************/
void add_TestImages (Br2HdrManager & manager, int xdim, int ydim, int N, ImageType type)
{
    switch (type)
    {
      case IMAGE_RGB_U8:  add_TestImages_RGB_U8  (manager, xdim,ydim, N); break;
      case IMAGE_RGB_U16: add_TestImages_RGB_U16 (manager, xdim,ydim, N); break;
      default:            NOT_IMPL_IMAGE_TYPE(type); break;
    }
}

void add_TestImages_RGB_U8 (Br2HdrManager & manager, int xdim, int ydim, int N)
{
    char s[80]; 
    
    for (int i=0; i < N; i++)
    {
      sprintf (s, "img%d - RGB U8", i);
      
      BrImage img (xdim, ydim, IMAGE_RGB_U8, s, pow(2., i));
      fillBuffer_RGB_U8 (img, i);
      
      manager.add_Image (img);
    }
    manager.init_Calctor();
}

void add_TestImages_RGB_U16 (Br2HdrManager & manager, int xdim, int ydim, int N)
{
    char s[80]; 
    
    for (int i=0; i < N; i++)
    {
      sprintf (s, "img%d - RGB U16", i);
      
      BrImage img (xdim, ydim, IMAGE_RGB_U16, s, pow(2., i));
      fillBuffer_RGB_U16 (img, i);
      
      manager.add_Image (img);
    }
    manager.init_Calctor();
}


/**+*************************************************************************\n
  Add images which produce errors while response computation.
******************************************************************************/
void add_BadImages (Br2HdrManager & manager, int xdim, int ydim, int N, ImageType type)
{
    switch (type)
    {
      case IMAGE_RGB_U8:  add_BadImages_RGB_U8  (manager, xdim,ydim, N); break;
      case IMAGE_RGB_U16: add_BadImages_RGB_U16 (manager, xdim,ydim, N); break;
      default:            NOT_IMPL_IMAGE_TYPE(type); break;
    }
}

void add_BadImages_RGB_U8 (Br2HdrManager & manager, int xdim, int ydim, int N)
{
    char s[80]; 
    
    for (int n=0; n < N; n++)
    {
      sprintf (s, "img%d - RGB U8", n);
      
      BrImage img (xdim, ydim, IMAGE_RGB_U8, s, pow(2., n));
      ImgScheme2DView< Rgb<uint8> >  A(img);
      
      for (int i=0; i < A.dim1(); i++)
        for (int j=0; j < A.dim2(); j++) {
          Rgb<uint8> rgb(0,0,0);
          rgb [n % 3] = 50;  // jedes 3. Bild R-Kanal etc.
          A[i][j] = rgb;
        }
      manager.add_Image (img);
    }
    manager.init_Calctor();
}

void add_BadImages_RGB_U16 (Br2HdrManager & manager, int xdim, int ydim, int N)
{
    char s[80]; 
    
    for (int n=0; n < N; n++)
    {
      sprintf (s, "img%d - RGB U16", n);
      
      BrImage img (xdim, ydim, IMAGE_RGB_U16, s, pow(2., n));
      ImgScheme2DView< Rgb<uint16> >  A(img);
      
      for (int i=0; i < A.dim1(); i++)
        for (int j=0; j < A.dim2(); j++) {
          Rgb<uint16> rgb(0,0,0);
          rgb [n % 3] = 50*256;  // jedes 3. Bild R-Kanal etc.
          A[i][j] = rgb;
        }
      manager.add_Image (img);
    }
    manager.init_Calctor();
}

/**+*************************************************************************\n
  @class RadianceSceneSlopeXY  -  A simulated radiance scene
******************************************************************************/
class RadianceSceneSlopeXY 
{
    int    xdim_, ydim_; 
    double Vmin_, Vmax_;
    double f_;
public:
    RadianceSceneSlopeXY (int xdim, int ydim, double Vmin=0.0, double Vmax=1.0) 
      : xdim_(xdim), ydim_(ydim),
        Vmin_(Vmin), Vmax_(Vmax)
      {
        if (xdim<2 && ydim<2)
          f_ = (Vmax - Vmin);  // sinnlos, da nur (0,0) moegl. mit rad.=Vmin
        else if (xdim<2)
          f_ = (Vmax - Vmin) / (ydim - 1);
        else if (ydim<2)
          f_ = (Vmax - Vmin) / (xdim - 1);
        else
          f_ = (Vmax - Vmin) / ((xdim - 1) * (ydim - 1));
      }
    
    double radiance (int ix, int iy) const  {return Vmin_ + f_ * ix * iy;}
};


/**+*************************************************************************\n
  @class SimulResponseLinear  -  A simulated linear response function.
******************************************************************************/
class SimulResponseLinear 
{
    double   Xmin_, Xmax_;      // exposure values to z=0 and z=zmax.
    unsigned zmax_;             // 255 for 8-bit, 1024 for 10-bit etc.
    double   f_;
public:
    SimulResponseLinear (double Xmin, double Xmax, unsigned zmax) 
      : 
        Xmin_(Xmin), Xmax_(Xmax), zmax_(zmax) 
      {
        f_ = zmax / (Xmax - Xmin);
      }
    
    unsigned response (double X)  
      { if (X <= Xmin_) return 0;
        if (X >= Xmax_) return zmax_;
        return unsigned (f_ * (X - Xmin_) + 0.5); 
      }
};

/**+*************************************************************************\n
  Add images with a X-Y-slope.
******************************************************************************/
void add_SlopeXYImages (Br2HdrManager & manager, int xdim, int ydim, int N, ImageType type)
{
    switch (type)
    {
      case IMAGE_RGB_U8:  add_SlopeXYImages_RGB_U8  (manager, xdim,ydim, N); break;
      case IMAGE_RGB_U16: add_SlopeXYImages_RGB_U16 (manager, xdim,ydim, N); break;
      default:            NOT_IMPL_IMAGE_TYPE(type); break;
    }
}

void add_SlopeXYImages_RGB_U8 (Br2HdrManager & manager, 
                               int xdim, int ydim, int N)
{
    RadianceSceneSlopeXY  scene (xdim, ydim, 0.0, 1.0);
    SimulResponseLinear  response_R (0.1, 1.0, 255);
    SimulResponseLinear  response_G (0.1, 2.0, 255);
    SimulResponseLinear  response_B (0.1, 3.0, 255);
    
    for (int n=0; n < N; n++)
    {
      char s[80];  
      sprintf (s, "SlopeXY.%d - RGB U8", n);
      double time = pow (2., n);

      BrImage  img (xdim, ydim, IMAGE_RGB_U8, s, time);
      ImgScheme2DView< Rgb<uint8> >  A(img);
      
      for (int i=0; i < A.dim1(); i++)
        for (int j=0; j < A.dim2(); j++) {
          Rgb<uint8> rgb;
          rgb[0] = response_R.response (scene.radiance(j,i) * time);  // j==xIndex, i==yIndex!
          rgb[1] = response_G.response (scene.radiance(j,i) * time);  // j==xIndex, i==yIndex!
          rgb[2] = response_B.response (scene.radiance(j,i) * time);  // j==xIndex, i==yIndex!
          A[i][j] = rgb;
        }
      manager.add_Image (img);
    }
    manager.init_Calctor();
}

void add_SlopeXYImages_RGB_U16 (Br2HdrManager & manager, 
                                int xdim, int ydim, int N)
{
    RadianceSceneSlopeXY  scene (xdim, ydim, 0.0, 1.0);
    SimulResponseLinear  response_R (0.1, 1.0, 65535);
    SimulResponseLinear  response_G (0.1, 2.0, 65535);
    SimulResponseLinear  response_B (0.1, 3.0, 65535);
    
    for (int n=0; n < N; n++)
    {
      char s[80];  
      sprintf (s, "SlopeXY.%d - RGB U16", n);
      double time = pow (2., n);

      BrImage  img (xdim, ydim, IMAGE_RGB_U16, s, time);
      ImgScheme2DView< Rgb<uint16> >  A(img);
      
      for (int i=0; i < A.dim1(); i++)
        for (int j=0; j < A.dim2(); j++) {
          Rgb<uint16> rgb;
          rgb[0] = response_R.response (scene.radiance(j,i) * time);  // j==xIndex, i==yIndex!
          rgb[1] = response_G.response (scene.radiance(j,i) * time);  // j==xIndex, i==yIndex!
          rgb[2] = response_B.response (scene.radiance(j,i) * time);  // j==xIndex, i==yIndex!
          A[i][j] = rgb;
        }
      manager.add_Image (img);
    }
    manager.init_Calctor();
}

}  // namespace

// END OF FILE
