#include "cp_types.h"
#include "cp_proto.h"

/* routines for drawing faces, edges, complex */

/* Drawing flag scheme: flags indicate drawing instructions
   for objects -- circles/lines/faces. Bits set as follows:
      1  ->  draw object?
      2  ->  fill? (4 and 16 imply 2, also)
      4  ->  off = foreground, on = background 
             (applies to interior, not border, overriden by bit 16)
      8  ->  border color? (off = foreground, on = recorded color)
      16 ->  interior color? (default set by bit 4, on --> recorded color)
      32 ->  display label?

Eg.  flag=3: filled object, in foreground
     flag=9: open object, border in (recorded) color
             (for 'edge', this gives colored edge)
     flag=19: filled in color, border in foreground
     flag=27: filled, border and interior in color
     flag=15: filled with background, border in color
     flag=32: label only (face or circle)

Normally, flag for each type of object; often, when passed on to
subroutine, you may need to pass color code too:
Eg. (cflag, ecol, ccol) for circle flag, border color code, int color 
code.

Note: see 'bc_offset' which may effect border color indices.
*/

int layout_facelist(struct p_data *p,int fflag,struct Vertlist **facelist,
		    int fix,int l_face,int show)
/* lay out faces from facelist; display as you go along if show. 
Standard flag for colors. fix=1 means to locate faces successively 
as drawn (changing stored centers). facelist is cleared. 
l_face="live face" may be given; */
{
	int ecol=FG_COLOR,fcol=FG_COLOR,n,v1,v2,v3,vv=0;
	double o1,o2,o3;
	struct Vertlist *trace;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;

	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
	if ((*facelist)==NULL) return 0;
	if (fflag & 4) fcol=BG_COLOR;
	trace=*facelist;
		/* find first */
	if (l_face>0 && l_face<=p->facecount) vv=l_face;
	do {
			/* if fix, draw successive ones as contig. */
	  if (fix)
	    {
	      if ( (n=nghb_tri(p,vv,trace->v))<0 )
		n=p->faces[trace->v].index_flag;
	      /* if not contig, use index_flag */
	      v1=p->faces[trace->v].vert[n];
	      v2=p->faces[trace->v].vert[(n+1)%3];
	      v3=p->faces[trace->v].vert[(n+2)%3];
	      
	      if (p->overlap_status) /* oj for edge opposite vj */
		{
		  o1=pK_ptr[v2].overlaps[nghb(p,v2,v3)];
		  o2=pK_ptr[v3].overlaps[nghb(p,v3,v1)];
		  o3=pK_ptr[v1].overlaps[nghb(p,v1,v2)];
		}
	      else o1=o2=o3=1.0;
	      any_compcenter(p->hes,
		pR_ptr[v1].center,pR_ptr[v2].center,&pR_ptr[v3].center,
		pR_ptr[v1].rad,pR_ptr[v2].rad,&pR_ptr[v3].rad,o1,o2,o3);
			/* compute and store new center */
	    }
	  
	  while (trace!=NULL && trace->v==vv) trace=trace->next;
					/* skip repeats */
	  if (trace==NULL) break;
	  vv=trace->v;
	  if ((fflag & 1) && (vv>0) && (vv <= p->facecount))
	    {
	      if (fflag & 16) fcol=p->faces[vv].color;
	      if (fflag & 8)
		{
		  ecol=p->faces[vv].color;
		  if (ecol>=bc_min && ecol<=bc_max) ecol+=bc_offset;
		}
	      draw_any_face(p,vv,fflag,ecol,fcol,show);
	      if (fflag & 32) draw_face_number(p,vv,vv,show);
	    }
	  trace=trace->next;
	} while (trace!=NULL);
	vert_free(facelist);
	return vv;
} /* layout_facelist */

int draw_complex(struct p_data *p,int fflag,int show)
/* draw whole complex. Standard flag */
{
  int k,ecol=FG_COLOR,fcol=FG_COLOR;

  if (!p->status) return 0;
  for (k=1;k<=p->facecount;k++) 
    {
      if (fflag & 16) fcol=p->faces[k].color;
      if (fflag & 8)
	{
	  ecol=p->faces[k].color;
	  if (ecol>=bc_min && ecol<=bc_max) ecol+=bc_offset;
	}
      draw_any_face(p,k,fflag,ecol,fcol,show);
    }
  return 1;
} /* draw_complex */

int draw_edge_points(struct s_data *q,int hes,complex v_cent,
		     complex w_cent,int col,int show)
{
 if (hes<0) 
   hgeo(q,v_cent,w_cent,col,show);
 else if (hes>0) 
   sgeo(q,v_cent,w_cent,col,show);
 else  
   kline(q,v_cent,w_cent,col,show);
 if (show) refresh_canvas(q);
 return 1; 
} /* draw_edge_points */

int draw_edge(struct p_data *p,int v1,int v2,int col,int show)
{
	if (!p->status 
	    || !p->packK_ptr[v1].plot_flag
	    || !p->packK_ptr[v2].plot_flag) return 0;
	if (p->hes<0) 
		hgeo(p->screen,p->packR_ptr[v1].center,
			p->packR_ptr[v2].center,col,show);
	else if (p->hes>0) 
		sgeo(p->screen,p->packR_ptr[v1].center,
			p->packR_ptr[v2].center,col,show);
	else  
		kline(p->screen,p->packR_ptr[v1].center,
			p->packR_ptr[v2].center,col,show);
	if (show) refresh_canvas(p->screen);
	return 1;
} /* draw_edge */

int draw_bdry_seg(struct p_data *p,int n,int eflag,int ecol)
/* draw boundary segment, given starting face and index of begin vert.
Use eflag bits for options: 4=label; 8=in color; 16=circles also. */
{
  int indx=0,w_indx,colr;
  complex v_cent,w_cent;
  struct RedList *trace;

  if (!(trace=p->edge_pair[n].edge)) return 0;
  if (trace->next->face==trace->prev->face
      && trace->v_flag!=(indx=p->edge_pair[n].edge_indx)) 
    /* blue face, second edge, get data from prev face. */
    w_cent=trace->prev->center;
  else w_cent=trace->center;
  colr=FG_COLOR;
  if (eflag & 8) colr=ecol;
  if (eflag & 16) /* draw/label circles forming the edge */
    {
      w_indx=p->faces[trace->face].vert[trace->v_flag];
      draw_any_circle_data(p->screen,p->hes,p->packR_ptr[w_indx].rad,
			   w_cent,41,colr,BG_COLOR,w_indx,1);
    }
  do
    {
      v_cent=w_cent;
      if (trace->next->face==trace->prev->face
	  && trace->v_flag==indx) /* blue face, want second edge */
	{
	  indx=(indx+1) %3;
	  w_cent=trace->prev->center;
	}
      else 
	{
	  trace=trace->next_edge;
	  indx=(nghb_tri(p,trace->prev->face,trace->face)+2) % 3;
	  w_cent=trace->center;
	}
      draw_edge_points(p->screen,p->hes,v_cent,w_cent,colr,1);
      if (eflag & 16) 
	{
	  w_indx=p->faces[trace->face].vert[trace->v_flag];
	  draw_any_circle_data(p->screen,p->hes,p->packR_ptr[w_indx].rad,
			       w_cent,41,colr,BG_COLOR,w_indx,1);
	}
    } while (!(trace->corner_flag[indx] & 3));
  if (eflag & 4) draw_bdry_seg_num(p,n);
  return 1;
} /* draw_bdry_seg */

int draw_bdry_seg_num(struct p_data *p,int n)
/* put segment number at first circle */
{
  complex ctr;
  struct RedList *trace;

  if (!(trace=p->edge_pair[n].edge)) return 0;
  if (trace->next->face==trace->prev->face
      && trace->v_flag!=p->edge_pair[n].edge_indx) 
    /* blue face, second edge, get data from prev face. */
    ctr=trace->prev->center;
  else ctr=trace->center;
  shownumber(p->screen,n,ctr,1);
  return 1;
} /* draw_bdry_seg_num */
	
int draw_any_face(struct p_data *p,int face,int fflag,
	      int ecol,int fcol,int show)
/* use approp geom */
{
	int front;
	complex p1,p2,p3,ctr;
	struct R_data *pR_ptr;

	if (!p->status || face<1 || face > p->facecount
	    || !p->faces[face].plot_flag)
		return 0;
	pR_ptr=p->packR_ptr;
	p1=pR_ptr[p->faces[face].vert[0]].center;
	p2=pR_ptr[p->faces[face].vert[1]].center;
	p3=pR_ptr[p->faces[face].vert[2]].center;
	if (p->hes<0) 
	  {
	    if (fflag & 1)
	      h_triangle(p->screen,p1,p2,p3,fflag,ecol,fcol,show);
	    if (fflag & 32)
	      {
		ctr.re=(p1.re+p2.re+p3.re)*.33333;
		ctr.im=(p1.im+p2.im+p3.im)*.33333;
		shownumber(p->screen,face,ctr,show);
	      }
	  }
	else if (p->hes==0) 
	  {
	    if (fflag & 1)
	      e_triangle(p->screen,p1,p2,p3,fflag,ecol,fcol,show);
	    if (fflag & 32)
	      {
		ctr.re=(p1.re+p2.re+p3.re)*.33333;
		ctr.im=(p1.im+p2.im+p3.im)*.33333;
		shownumber(p->screen,face,ctr,show);
	      }
	  }
	else if (p->hes>0) 
	  {
	    if (fflag & 1)
	      s_triangle(p->screen,ss_view(p->screen,p1,1,&front),
		       ss_view(p->screen,p2,1,&front),
		       ss_view(p->screen,p3,1,&front),
		       fflag,ecol,fcol,show); 
	    if (fflag & 32)
	      {
		ctr=sph_tri_center(p1,p2,p3);
		ctr=ss_view(p->screen,ctr,1,&front);
		if (!front) return 0;
		ctr=s_pt_to_visual_plane(ctr);
		shownumber(p->screen,face,ctr,show);
	      }
	  }
	return 1;
} /* draw_any_face */



	

