#include "cp_types.h"
#include "cp_proto.h"

/* Compute radii of packing to meet angle_sum targets, specified as
"aims". A negative "aim" means that that circle should not have its
radius adjusted. aim of zero is possible only for boundary 
circles in hyp setting. 

All routines are iterative, inspired by routine suggested
by Thurston. This file includes recent algorithms, old reliable
versions, and experimental stuff being tried in spherical case.

Radii are of type r,h, or s (for special): r is euclidean radius,
h is hyperbolic, and s=exp(-h). Other relationships are:
h=log((1+r)/(1-r)), s=(1-r)/(1+r), r=(1-s)/(1+s).
Will be storing radii as s_radii. If h is infinite, then will store
eucl radius with negative value as the s_radius so it can be
used in graphing the appropriate circle. 

There currently is no packing algorithm for spherical geometry. Only
approach is to force packing into unit disc, do maximal packing
there and project back to sphere. Often, this involves a "puncture"
of the complex; missing circle is replaced as northern hemisphere
after max packing is projected back to sph. 

Overlap angles specified > Pi/2 can lead to incompatibilities. To
avoid domain errors, routines computing angles in such cases return
angle Pi. */

extern int repack_activity_msg();

int e_riffle(struct p_data *p,int passes)
/* CRC - modified 5/28/97 -- uses model with super steps, 
safety checks. Adjust eucl radii to meet curvature targets in 
'aim'. Neg. aim means that vertex is free - not subject to adjustment. */
{
  int i,j,aimnum=0,*index,count=0,key=0,key0,N,flag;
  double r, r2, fbest, faim, del, bet;
  double ttoler, cut=100, fact=-1, cut0, fact0;
  double lmax, rat,rr,*R0,ftol=.05;
  struct R_data *pR_ptr;
  struct K_data *pK_ptr;

  index=(int *)malloc(5*(p->nodecount+1)*sizeof(int));
  R0 = (double *)malloc((p->nodecount+1)*sizeof(double));
  pR_ptr=p->packR_ptr;
  pK_ptr=p->packK_ptr;
  for (i=1;i<=p->nodecount;i++)
    if (pR_ptr[i].aim>0)
      {
	index[aimnum]=i;
	aimnum++;
      }
  if (aimnum==0) 
    {free(index); free(R0);return count; }
  ttoler = 3*aimnum*toler; /* adjust tolerance */

  while (cut >ttoler && count<passes)
    {
      /* save values */
      cut0 = cut;key0 = key;fact0 = fact; 
      for (i=1;i<=p->nodecount;i++) R0[i] = pR_ptr[i].rad;
      /* do update using uniform label model */
      cut = 0;
      for (j=0;j<aimnum;j++) 
	{
	  i = index[j]; 		/* point to active node */
	  faim = pR_ptr[i].aim;	/* get target sum */
	  r = pR_ptr[i].rad;	/* get present label */
	  e_anglesum_overlap(p,i,r,&fbest,&flag);  /* compute sum */
	  /* use the model to predict the next value */
	  N = 2*pK_ptr[i].num;
	  del = sin(faim/N);
	  bet = sin(fbest/N);
	  r2 = r*bet*(1-del)/(del*(1-bet));
	  /* store as new radius label */
	  if (r2<0) goto ABORT;
	  pR_ptr[i].rad = r2;
	  pR_ptr[i].curv = fbest;	/* store new angle sum */
	  fbest -= faim;
	  cut += fbest*fbest;	/* accum abs error */
	}
      cut = sqrt(cut);
      /* do super step? */
      key = 1;
      if (key0==1)
	{
	  fact = cut/cut0;
	  if (fabs(fact-fact0)<ftol) fact = fact/(1-fact);
	  lmax = 1000;
	  for (j=0;j<aimnum;j++)	 /* find max step */
	    {
	      i = index[j];
	      r = pR_ptr[i].rad;
	      rat = r - R0[i];
	      if (rat<0)
		lmax = (lmax < (rr=(-r/rat))) ? lmax : rr;
	    }
	  fact = (fact < 0.5*lmax) ? fact : 0.5*lmax; 
	  /* compute new step */
	  for (j=0;j<aimnum;j++)     /* interpolate to new radius */
	    {
	      i = index[j];
	      pR_ptr[i].rad += fact*(pR_ptr[i].rad-R0[i]);
	      if (pR_ptr[i].rad<0)
		{pR_ptr[i].rad=.001;goto ABORT;}
	    }
	  key = 0;
	}

      /* show activity */
      repack_activity_msg("^");

      count++;
    } /* end of while */

  free(index);
  free(R0);
  return count;
 ABORT:
  free(index);
  free(R0);
  sprintf(msgbuf,"Aborting repack: radius went negative.");
  emsg();
  return 0; 
} /* e_riffle */

int old_e_riffle(struct p_data *p,int passes)
/* based on original riffle routine (slow, but generally dependable) */
{
  int i,j,aimnum=0,*index,count=0,dummy,flag;
  double recip,accum=0,verr,err,cut;
  struct R_data *pR_ptr;

  index=(int *)malloc(5*(p->nodecount+1)*sizeof(int));
  pR_ptr=p->packR_ptr;
  for (i=1;i<=p->nodecount;i++)
    if (pR_ptr[i].aim>0)
      {
	index[aimnum]=i;
	aimnum++;
	err=pR_ptr[i].curv-pR_ptr[i].aim;
	accum += (err<0) ? (-err) : err;
      }
  if (aimnum==0) {free(index);return count;}
  recip=.333333/aimnum;
  cut=accum*recip;
  while (cut >toler && count<passes)
    {
      for (j=0;j<aimnum;j++)
	{
	  i=index[j];
	  e_anglesum_overlap(p,i,pR_ptr[i].rad,
			     &pR_ptr[i].curv,&flag);
	  verr=pR_ptr[i].curv-pR_ptr[i].aim;
	  if (fabs(verr)>cut)
	    {
	      pR_ptr[i].rad=e_radcalc(p,i,
				      pR_ptr[i].rad,pR_ptr[i].aim,&dummy);
	      count++;
	    }
	}
      accum=0;
      for (j=0;j<aimnum;j++)
	{
	  i=index[j];
	  err=pR_ptr[i].curv-pR_ptr[i].aim;
	  accum += (err<0) ? (-err) : err;
	}
      cut=accum*recip;

      /* show activity */
      repack_activity_msg("e");
	
    } /* end of while */
  free(index);
  return count;
} /* old_e_riffle */

int new_e_riffle(struct p_data *p,int passes)
/* CRC - modified 8/1/01 from e_riffle 
   uses uniform neighbor model (UNM) and two levels of super steps
   also saves previous approximation to replace super step
   approximates if they are not better
   
   Has same checks as in e_riffle, same goal (meet aims,
   aims<0 not adjusted. */

/* Basic outline:
   apply UNM
   while (not within tolerance)
      apply UNM
      apply superstep of type 1 or type 2
      check results:
         OK? accept and continue
         Not? use pre-superstep value
      update superstep performance
   end while

   type 1 super step is just like taking m extra steps where
       m increases as we have successful iterations
       can predict next iteration failure here, switch
       to type 2 if overall performance is becoming uniform
   type 2 super step is the 'super' super step  x/(1-x)
       which attempts to jump to the end.  */
{
  int i,j,aimnum=0,*index,count=0,key,N,flag;
  int sct, fct;
  double m, mm, ftol, c0, c1, per, act, pred, factor, lambda, mmax;
  double r, r2, fbest, faim, del, bet;
  double ttoler, fact0;
  double lmax, rat,rr,*R1,*R2;
  struct R_data *pR_ptr;
  struct K_data *pK_ptr;

/* construct index and allocate space for storage of radius labels */

  index=(int *)malloc(5*(p->nodecount+1)*sizeof(int));
  R1 = (double *)malloc((p->nodecount+1)*sizeof(double));
  R2 = (double *)malloc((p->nodecount+1)*sizeof(double));
  pR_ptr=p->packR_ptr;
  pK_ptr=p->packK_ptr;
  for (i=1;i<=p->nodecount;i++)
    if (pR_ptr[i].aim>0)
      {
        index[aimnum]=i;
        aimnum++;
      }
  if (aimnum==0)
    {free(index); free(R1); free(R2); return count; }

/* set up some parameters */

  ttoler = 3*aimnum*toler; /* adjust tolerance */
  key = 1;                 /* which type of superstep to start with */
  m = 1;                   /* begining Type 1 multiplier */
  sct = 1;                 /* count of number of times for Type 1 SS */
  fct = 2;                 /* number of iters before a Type 2 SS */
  ftol = 0.0005;           /* tolerance for Type 2 SS */

/* do one iteration to get started */
  c0 = 0;
  for (j=0;j<aimnum;j++)
    {
      i = index[j];                 /* point to active node */
      faim = pR_ptr[i].aim;         /* get target sum */
      r = pR_ptr[i].rad;            /* get present label */
      e_anglesum_overlap(p,i,r,&fbest,&flag);  /* compute sum */
      /* use the model to predict the next value */
      N = 2*pK_ptr[i].num;
      del = sin(faim/N);
      bet = sin(fbest/N);
      r2 = r*bet*(1-del)/(del*(1-bet));
      /* store as new radius label */
      if (r2<0) goto ABORT;
      pR_ptr[i].rad = r2;
      pR_ptr[i].curv = fbest;       /* store new angle sum */
      fbest -= faim;
      c0 += fbest*fbest;   /* accum abs error */
    }
  c0 = sqrt(c0);
     
  while (c0 >ttoler && count<passes)
    {
      /* save values */
      for (i=1;i<=p->nodecount;i++) R1[i] = pR_ptr[i].rad;

/* do update using uniform label model */
      c1 = 0;
      per = 0;
      for (j=0;j<aimnum;j++)
        {
          i = index[j];                 /* point to active node */
          faim = pR_ptr[i].aim; /* get target sum */
          r = pR_ptr[i].rad;    /* get present label */
          e_anglesum_overlap(p,i,r,&fbest,&flag);  /* compute sum */
          /* use the model to predict the next value */
          N = 2*pK_ptr[i].num;
          del = sin(faim/N);
          bet = sin(fbest/N);
          r2 = r*bet*(1-del)/(del*(1-bet));
          /* store as new radius label */
          if (r2<0) goto ABORT;
          if (r2>r) per++;              /* count when r increases */
          pR_ptr[i].rad = r2;
          pR_ptr[i].curv = fbest;       /* store new angle sum */
          fbest -= faim;
          c1 += fbest*fbest;   /* accum abs error */
        }
      c1 = sqrt(c1);
      per = per/aimnum;
     
      factor = c1/c0;       /* error reduction ratio */

/* superstep calculation */

      /* save values */
      for (i=1;i<=p->nodecount;i++) R2[i] = pR_ptr[i].rad;

      /* find maximum step can safely take */
      lmax = 10000;
      for (j=0;j<aimnum;j++)         /* find max step */
        {
          i = index[j];
          r = pR_ptr[i].rad;
          rat = r - R1[i];
          if (rat<0)
             lmax = (lmax < (rr=(-r/rat))) ? lmax : rr;
         }
      lmax = lmax/2;
     
      /* do super step */
      if (key==1)              /*  type 1  SS */
        {
          lambda = m*factor;
          mmax = 0.75/(1-factor);               /* upper limit on m */
          m = (mmax < (mm=1+0.8/(sct+1)*m)) ? mmax : mm;
        }
      else                     /*  type 2 SS */
        {
          if (sct>fct && fabs(factor-fact0)<ftol)  /* try SS-2 */
            {
              lambda = factor/(1-factor);
              sct = -1;
            }
          else
            lambda = factor;                  /* do something */
        }
      lambda = (lambda>lmax) ? lmax : lambda;

      for (j=0;j<aimnum;j++)     /* interpolate to new radius */
        {
          i = index[j];
          pR_ptr[i].rad += lambda*(pR_ptr[i].rad-R1[i]);
          if (pR_ptr[i].rad<0)
            {pR_ptr[i].rad=.001;goto ABORT;}
        }
      sct++;
      fact0 = factor;

/* end of superstep */


/* do update/check using uniform label model */
      c0 = 0;
      per = 0;
      for (j=0;j<aimnum;j++)
        {
          i = index[j];                 /* point to active node */
          faim = pR_ptr[i].aim; /* get target sum */
          r = pR_ptr[i].rad;    /* get present label */
          e_anglesum_overlap(p,i,r,&fbest,&flag);  /* compute sum */
          /* use the model to predict the next value */
          N = 2*pK_ptr[i].num;
          del = sin(faim/N);
          bet = sin(fbest/N);
          r2 = r*bet*(1-del)/(del*(1-bet));
          /* store as new radius label */
          if (r2<0) goto ABORT;
          if (r2>r) per++;    /* count when r increases */
          pR_ptr[i].rad = r2;
          pR_ptr[i].curv = fbest;       /* store new angle sum */
          fbest -= faim;
          c0 += fbest*fbest;   /* accum abs error */
        }
      c0 = sqrt(c0);
      per = per/aimnum;
     
/* check results */

     pred = exp(lambda*log(factor));          /* predicted improvement */
     act = c0/c1;                   /* actual improvement */
     if (act<1)                     /* did some good */
       {
          if (act>pred)             /* not as good as expected: reset */
            {
              m = 1;
              sct = 0;
              if (key==1) key = 2;
            }                       /* implied else: accept result */
        }
     else                           /* reset to before superstep */
        {
          m = 1;
          sct =0;
          for (i=1;i<=p->nodecount;i++) pR_ptr[i].rad  = R2[i];
          c0 = c1;
          if (key==2) key = 1;
        }

      /* show activity */
      repack_activity_msg("^");
      count++;
    } /* end of while */

  free(index);
  free(R1);
  free(R2);
  return count;
 ABORT:
  free(index);
  free(R1);
  free(R2);
  sprintf(msgbuf,"Aborting repack: radius went negative.");
  emsg();
  return 0;
} /* new_e_riffle */

int e_riffle_vert(struct p_data *p,int i)
{
  int n=0,dummy,flag;
  double diff;
  struct R_data *pR_ptr;

  pR_ptr=p->packR_ptr;
  e_anglesum_overlap(p,i,pR_ptr[i].rad,&pR_ptr[i].curv,&flag);
  diff=pR_ptr[i].curv-pR_ptr[i].aim;
  while (n<20 && (diff>okerr || diff<(-okerr)) )
    /* limit number of attempts */
    {
      pR_ptr[i].rad=e_radcalc(p,i,
			      pR_ptr[i].rad,pR_ptr[i].aim,&dummy);
      e_anglesum_overlap(p,i,pR_ptr[i].rad,&pR_ptr[i].curv,&flag);
      diff=pR_ptr[i].curv-pR_ptr[i].aim;
      n++;
    }
  return n;
} /* e_riffle_vert */







