package Language::INTERCAL::Interface::Curses;

# Text (Curses) interface for sick and intercalc

# This file is part of CLC-INTERCAL

# Copyright (c) 2006 Claudio Calvelli, all rights reserved.

# CLC-INTERCAL is copyrighted software. However, permission to use, modify,
# and distribute it is granted provided that the conditions set out in the
# licence agreement are met. See files README and COPYING in the distribution.

use strict;
use vars qw($PERVERSION);
$PERVERSION = "CLC-INTERCAL INTERCAL/Interface/Curses.pm 1.-94.-4";

use Carp;
use Curses;

use Language::INTERCAL::Exporter '1.-94.-4';
use Language::INTERCAL::Interface::common '1.-94.-4';
use vars qw(@ISA);
@ISA = qw(Language::INTERCAL::Interface::common);

my %keymap = (
    'Left' => KEY_LEFT,
    'BackSpace' => KEY_BACKSPACE,
    'Enter' => KEY_ENTER,
    'Return' => "\cM",
    'Linefeed' => "\cJ",
    (map { ("F$_" => KEY_F($_)) } (1..12)),
    # XXX
);

my %reserved = (
    &KEY_LEFT    => \&_move_left,
    &KEY_RIGHT   => \&_move_right,
    &KEY_UP      => \&_move_up,
    &KEY_DOWN    => \&_move_down,
    &KEY_ENTER   => \&_activate,
    "\cM"        => \&_activate,
    "\cJ"        => \&_activate,
);

sub new {
    @_ == 1 or croak "Usage: Language::INTERCAL::Interface::Curses->new";
    my ($class) = @_;
    initscr();
    clearok(1);
    noecho();
    cbreak();
    leaveok(0);
    eval "END { eval { keypad(0) }; endwin(); print '\n' }";
    keypad(1);
    meta(1);
    my $curse = bless {
	keypress => {},
	keylist => [],
	keyrows => [],
	keycols => [],
	resize => 0,
	windows => [],
	pending => ERR,
    }, $class;
    $curse->_initialise;
    $SIG{WINCH} = sub { $curse->{resize} = 1 };
    $curse;
}

sub is_interactive { 1 }

sub window {
    @_ == 4 or croak "Usage: Curses->window(NAME, DESTROY, DEFINITION)";
    my ($curse, $name, $destroy, $def) = @_;
    $curse->{keypress} = {};
    $curse->{keylist} = [];
    my $window = $curse->_parse_def(@$def);
    _place($window, 0, COLS, 0, LINES);
    $curse->{keyrows} = [];
    $curse->{keycols} = [];
    $curse->{lastkey} = [0, 0];
    if (@{$curse->{keylist}}) {
	$curse->{keylist} =
	    [ sort { $a->{y} <=> $b->{y} || $a->{x} <=> $b->{x} }
		   @{$curse->{keylist}} ];
	for (my $kp = 0; $kp < @{$curse->{keylist}}; $kp++) {
	    my $k = $curse->{keylist}[$kp];
	    push @{$curse->{keyrows}[$k->{y}]}, $kp;
	    push @{$curse->{keycols}[$k->{x}]}, $kp;
	}
	$curse->{lastkey}[1] = $curse->{keylist}[0];
    }
    &{$window->{show}}($curse, $window);
    push @{$curse->{windows}}, [
	$window,
	$curse->{keypress},
	$curse->{keylist},
	$curse->{keyrows},
	$curse->{keycols},
	$curse->{lastkey},
    ];
    $window;
}

sub show {
    @_ == 2 or croak "Usage: Curses->show(WINDOW)";
    my ($curse, $window) = @_;
    &{$window->{show}}($curse, $window);
}

sub enable {
    @_ == 2 or croak "Usage: Curses->enable(WINDOW)";
    my ($curse, $window) = @_;
    $window->{enabled} = 1;
    &{$window->{show}}($curse, $window);
}

sub disable {
    @_ == 2 or croak "Usage: Curses->disable(WINDOW)";
    my ($curse, $window) = @_;
    $window->{enabled} = 0;
    &{$window->{show}}($curse, $window);
}

sub update {
    @_ == 1 or croak "Usage: Curses->update";
    my ($curse) = @_;
    refresh();
}

sub start {
    @_ == 1 or croak "Usage: Curses->start";
    refresh();
}

sub run {
    @_ == 1 or croak "Usage: Curses->run";
    my ($curse) = @_;
    refresh();
    while (1) {
	if ($curse->{resize}) {
	    $curse->{resize} = 0;
	    endwin();
	    clearok(1);
	    $curse->_redraw();
	}
	cbreak();
	nodelay(0);
	my $key = $curse->{pending} eq ERR ? getch() : $curse->{pending};
	$curse->{pending} = ERR;
	return if $key eq ERR;
	if (exists $reserved{$key}) {
	    &{$reserved{$key}}($curse);
	    next;
	}
	if (exists $curse->{keypress}{$key}) {
	    $key = $curse->{keypress}{$key};
	    next unless $key->{enabled};
	    if ($curse->{lastkey}[1] != $key) {
		my $ok = $curse->{lastkey}[1];
		$curse->{lastkey}[1] = $key;
		for (my $kp = 0; $kp < @{$curse->{keylist}}; $kp++) {
		    next if $curse->{keylist}[$kp] != $key;
		    $curse->{lastkey}[0] = $kp;
		}
		$curse->show($ok);
	    }
	    $curse->show($key);
	    nodelay(1);
	    $curse->{pending} = getch() if $curse->{pending} eq ERR;
	    refresh() if $curse->{pending} eq ERR;
	    &{$key->{action}};
	    nodelay(1);
	    $curse->{pending} = getch() if $curse->{pending} eq ERR;
	    refresh() if $curse->{pending} eq ERR;
	    cbreak();
	    nodelay(0);
	    next;
	}
    }
}

sub pending_events {
    @_ == 1 or croak "Usage: Curses->pending_events";
    my ($curse) = @_;
    if ($curse->{pending} eq ERR) {
	nodelay(1);
	$curse->{pending} = getch();
	cbreak();
	nodelay(0);
    }
    return $curse->{pending} ne ERR;
}

sub _activate {
    my ($curse) = @_;
    return unless $curse->{lastkey}[1];
    return unless $curse->{lastkey}[1]->{enabled};
    &{$curse->{lastkey}[1]->{action}};
}

sub _move_left {
    my ($curse) = @_;
    return unless $curse->{lastkey}[1];
    my $i = $curse->{lastkey}[0];
    my $k = $curse->{lastkey}[1];
    my $r = $curse->{keyrows}[$k->{y}];
    my $ok = $curse->{lastkey}[1];
    if ($r->[0] == $i) {
	$i = $#$r;
    } else {
	my $j = 1;
	$j++ while $j < @$r && $r->[$j] != $i;
	$j--;
	$i = $j;
    }
    $curse->{lastkey}[0] = $r->[$i];
    $curse->{lastkey}[1] = $curse->{keylist}[$r->[$i]];
    $curse->show($ok);
    $curse->show($curse->{lastkey}[1]);
}

sub _move_right {
    my ($curse) = @_;
    return unless $curse->{lastkey}[1];
    my $i = $curse->{lastkey}[0];
    my $k = $curse->{lastkey}[1];
    my $r = $curse->{keyrows}[$k->{y}];
    my $ok = $curse->{lastkey}[1];
    if ($r->[-1] == $i) {
	$i = 0;
    } else {
	my $j = $#$r;
	$j-- while $j >= 0 && $r->[$j] != $i;
	$j++;
	$i = $j;
    }
    $curse->{lastkey}[0] = $r->[$i];
    $curse->{lastkey}[1] = $curse->{keylist}[$r->[$i]];
    $curse->show($ok);
    $curse->show($curse->{lastkey}[1]);
}

sub _move_up {
    my ($curse) = @_;
    return unless $curse->{lastkey}[1];
    my $i = $curse->{lastkey}[0];
    my $k = $curse->{lastkey}[1];
    my $r = $curse->{keycols}[$k->{x}];
    my $ok = $curse->{lastkey}[1];
    if ($r->[0] == $i) {
	$i = $#$r;
    } else {
	my $j = 1;
	$j++ while $j < @$r && $r->[$j] != $i;
	$j--;
	$i = $j;
    }
    $curse->{lastkey}[0] = $r->[$i];
    $curse->{lastkey}[1] = $curse->{keylist}[$r->[$i]];
    $curse->show($ok);
    $curse->show($curse->{lastkey}[1]);
}

sub _move_down {
    my ($curse) = @_;
    return unless $curse->{lastkey}[1];
    my $i = $curse->{lastkey}[0];
    my $k = $curse->{lastkey}[1];
    my $r = $curse->{keycols}[$k->{x}];
    my $ok = $curse->{lastkey}[1];
    if ($r->[-1] == $i) {
	$i = 0;
    } else {
	my $j = $#$r;
	$j-- while $j >= 0 && $r->[$j] != $i;
	$j++;
	$i = $j;
    }
    $curse->{lastkey}[0] = $r->[$i];
    $curse->{lastkey}[1] = $curse->{keylist}[$r->[$i]];
    $curse->show($ok);
    $curse->show($curse->{lastkey}[1]);
}

sub _redraw {
    my ($curse) = @_;
    erase();
    $@ = '';
    for my $w (@{$curse->{windows}}) {
	eval { _place($w->[0], 0, $COLS, 0, $LINES) };
	last if $@;
	$curse->show($w->[0]);
    }
    if ($@) {
	clearok(1);
	erase();
	my $line = 0;
	for my $s (split(/\n/, $@)) {
	    addstr($line++, 0, $s) if $line < $LINES;
	}
    }
    refresh();
}

sub _offset {
    my ($window, $x, $y) = @_;
    $window->{x} += $x;
    $window->{y} += $y;
    return unless exists $window->{children};
    for my $child (@{$window->{children}}) {
	_offset($child, $x, $y);
    }
}

sub _place {
    my ($window, $x, $width, $y, $height) = @_;
    my $diff = $width - $window->{width};
    $diff = 0 if $diff < 0;
    $x += int($diff / 2);
    $window->{x} ||= 0;
    $diff = $height - $window->{height};
    $diff = 0 if $diff < 0;
    $y += int($diff / 2);
    $window->{y} ||= 0;
    _offset($window, $x - $window->{x}, $y - $window->{y});
}

sub close {
    @_ == 2 or croak "Usage: Curses->close(WINDOW)";
    my ($curse, $window) = @_;
    my @nw = grep { $_->[0] != $window } @{$curse->{windows}};
    $curse->{windows} = \@nw;
    my $w;
    ($w, @$curse{qw(keypress keylist keyrows keycols lastkey)}) =
	@nw ? @{$nw[-1]} : (0, {}, {}, [], [], [0, 0]);
    $curse->_redraw;
}

sub _extend_width {
    my ($e, $cw) = @_;
    return if $e->{width} >= $cw;
    my $diff = $cw - $e->{width};
    $e->{width} = $cw;
    return unless exists $e->{children};
    my $d0 = int($diff / scalar @{$e->{colwidth}});
    my $d1 = $diff % scalar @{$e->{colwidth}};
    my $d = 0;
    my @d = ();
    for (my $c = 0; $c < @{$e->{colwidth}}; $c++) {
	$d[$c] = $d;
	$d += $d0 + (($c < $d1) ? 1 : 0);
	$e->{colwidth}[$c] += $d0 + (($c < $d1) ? 1 : 0);
    }
    for my $child (@{$e->{children}}) {
	my ($c0, $c1, $r0, $r1) = @{$child->{table}};
	$d = -$e->{border};
	for (my $c = $c0; $c < $c1; $c++) {
	    $d += $e->{colwidth}[$c] + $e->{border};
	}
	_extend_width($child, $d);
	_offset($child, $d[$c0], 0);
    }
}

sub _extend_height {
    my ($e, $rh) = @_;
    return if $e->{height} >= $rh;
    my $diff = $rh - $e->{height};
    $e->{height} = $rh;
    return unless exists $e->{children};
    my $d0 = int($diff / scalar @{$e->{rowheight}});
    my $d1 = $diff % scalar @{$e->{rowheight}};
    my $d = 0;
    my @d = ();
    for (my $r = 0; $r < @{$e->{rowheight}}; $r++) {
	$d[$r] = $d;
	$d += $d0 + (($r < $d1) ? 1 : 0);
	$e->{rowheight}[$r] += $d0 + (($r < $d1) ? 1 : 0);
    }
    for my $child (@{$e->{children}}) {
	my ($c0, $c1, $r0, $r1) = @{$child->{table}};
	$d = -$e->{border};
	for (my $r = $r0; $r < $r1; $r++) {
	    $d += $e->{rowheight}[$r] + $e->{border};
	}
	_extend_height($child, $d);
	_offset($child, 0, $d[$r0]);
    }
}

sub _make_table {
    my ($curse, $rows, $cols, $border, $elements) = @_;
    my @width = (0) x $cols;
    my @height = (0) x $rows;
    $border = $border ? 1 : 0;
    # try to determine row/column sizes using one cell elements
    for my $te (@$elements) {
	my ($e, $c0, $c1, $r0, $r1) = @$te;
	$width[$c0] = $e->{width}
	    if $c0 + 1 == $c1 && $width[$c0] < $e->{width};
	$height[$r0] = $e->{height}
	    if $r0 + 1 == $r1 && $height[$r0] < $e->{height};
    }
    # now adjust it for multirow/multicolumn
    for my $te (@$elements) {
	my ($e, $c0, $c1, $r0, $r1) = @$te;
	if ($c1 - $c0 > 1) {
	    my $cw = ($c1 - $c0 - 1) * $border;
	    for (my $c = $c0; $c < $c1; $c++) {
		$cw += $width[$c];
	    }
	    if ($cw < $e->{width}) {
		my $diff = $e->{width} - $cw;
		my $d0 = int($diff / ($c1 - $c0));
		my $d1 = $diff % ($c1 - $c0);
		for (my $c = $c0; $c < $c1; $c++) {
		    $width[$c] += $d0;
		    $width[$c] ++ if $c < $d1;
		}
	    }
	}
	if ($r1 - $r0 > 1) {
	    my $rh = ($r1 - $r0 - 1) * $border;
	    for (my $r = $r0; $r < $r1; $r++) {
		$rh += $height[$r];
	    }
	    if ($rh < $e->{height}) {
		my $diff = $e->{height} - $rh;
		my $d0 = int($diff / ($r1 - $r0));
		my $d1 = $diff % ($r1 - $r0);
		for (my $r = $r0; $r < $r1; $r++) {
		    $height[$r] += $d0;
		    $height[$r] ++ if $r < $d1;
		}
	    }
	}
    }
    # determine total window size and cell starting points
    my $width = $border;
    my @x = ();
    for (my $c = 0; $c < $cols; $c++) {
	$x[$c] = $width;
	$width += $width[$c] + $border;
    }
    my $height = $border;
    my @y = ();
    for (my $r = 0; $r < $rows; $r++) {
	$y[$r] = $height;
	$height += $height[$r] + $border;
    }
    # place all elements and extend them to fill cell if required
    my @children = ();
    for my $te (@$elements) {
	my ($e, $c0, $c1, $r0, $r1) = @$te;
	_offset($e, $x[$c0], $y[$r0]);
	my $cw = ($c1 - $c0 - 1) * $border;
	for (my $c = $c0; $c < $c1; $c++) {
	    $cw += $width[$c];
	}
	_extend_width($e, $cw);
	my $rh = ($r1 - $r0 - 1) * $border;
	for (my $r = $r0; $r < $r1; $r++) {
	    $rh += $height[$r];
	}
	_extend_height($e, $rh);
	$e->{table} = [$c0, $c1, $r0, $r1];
	push @children, $e;
    }
    # ready to go...
    return {
	type => 'table',
	width => $width,
	height => $height,
	colwidth => \@width,
	rowheight => \@height,
	show => \&_show_table,
	children => \@children,
	border => $border,
    };
}

sub _show_table {
    my ($curse, $table) = @_;
    $table->{type} eq 'table' or die "Internal error";
    # draw border, if required
    # XXX multirow fields may show '+' where '|' should be
    if ($table->{border}) {
	my $y = $table->{y};
	my $row = 0;
	for my $rh (@{$table->{rowheight}}, 0) {
	    move($y, $table->{x});
	    my $col = 0;
	    for my $cw (@{$table->{colwidth}}, 0) {
		my $plus = '-';
		for my $e (@{$table->{children}}) {
		    next if $e->{table}[0] != $col && $e->{table}[1] != $col;
		    next if $e->{table}[2] != $row && $e->{table}[3] != $row;
		    $plus = '+';
		    last;
		}
		addstr($plus . ('-' x $cw));
		$col++;
	    }
	    $y++;
	    for (my $x = 0; $x < $rh; $x++) {
		move($y, $table->{x});
		for my $cw (@{$table->{colwidth}}, 0) {
		    addstr('|' . (' ' x $cw));
		}
		$y++;
	    }
	    $row++;
	}
    }
    # draw elements
    for my $e (@{$table->{children}}) {
	&{$e->{show}}($curse, $e);
    }
}

sub _make_text {
    my ($curse, $value, $align, $size) = @_;
    $size ||= length $value;
    return {
	type => 'text',
	width => $size,
	height => 1,
	value => $value,
	enabled => 1,
	align => $align,
	show => \&_show_text_key,
    };
}

sub _show_text_key {
    my ($curse, $text) = @_;
    $text->{type} eq 'text' || $text->{type} eq 'key'
	or die "Internal error";
    move($text->{y}, $text->{x});
    my $diff0 = $text->{width} - length($text->{value});
    my $diff1 = int($diff0 / 2);
    my $diff2 = $diff0 - $diff1;
    eval { attrset(A_NORMAL) };
    eval { attron(A_BOLD) } if $text->{enabled};
    eval { attron(A_REVERSE) } if $text == $curse->{lastkey}[1];
    addstr(' ' x $diff0) if $diff0 > 0 && $text->{align} =~ /^r/i;
    addstr(' ' x $diff1) if $diff1 > 0 && $text->{align} =~ /^c/i;
    addstr($text->{value});
    addstr(' ' x $diff0) if $diff0 > 0 && $text->{align} =~ /^l/i;
    addstr(' ' x $diff2) if $diff2 > 0 && $text->{align} =~ /^c/i;
    eval { attrset(A_NORMAL) };
}

sub _set_text {
    my ($curse, $text, $value) = @_;
    $text->{type} eq 'text' or die "Internal error";
    $value = substr($value, 0, $text->{width});
    $text->{value} = $value;
    _show_text_key($curse, $text);
}

sub _get_text {
    my ($curse, $text) = @_;
    $text->{type} eq 'text' or die "Internal error";
    $text->{value};
}

sub _make_key {
    my ($curse, $label, $action, $keys) = @_;
    my $key = {
	type => 'key',
	width => length $label,
	height => 1,
	action => $action,
	align => 'c',
	enabled => 1,
	value => $label,
	show => \&_show_text_key,
    };
    push @{$curse->{keylist}}, $key;
    for my $k (@$keys) {
	$k = $keymap{$k} if exists $keymap{$k};
	next if exists $reserved{$k};
die "k=$k\n" if length $k > 1 && $k =~ /\D/; # XXX
	$curse->{keypress}{$k} = $key;
    };
    return $key;
}

1;
