import py
from py.xml import Namespace, Tag
#from py.__.xmlobj.visit import SimpleUnicodeVisitor
#from py.__.xmlobj.html import HtmlVisitor


def itertext(text):
    """ Generator: like string.split, but ''.join(itertext(t)) == t

    >>> list(itertext('word\n  second word')) 
    ['word', '\n', '  ', 'second', ' ', 'word']
    """
    state_word = 'not isspace()' 
    state_whitespace = '\t\n\x0b\x0c\r'
    state_space = ' '
    def compute_state(char):
        if char in state_whitespace:
            return state_whitespace
        if char == ' ':
            return state_space
        return state_word

    word = ''
    state = None 
    for char in text:
        if state is None:
            # init
            state = compute_state(char)
            word = char
            continue
        
        next_state = compute_state(char)
        if state != state_whitespace and state == next_state:
            word += char
            continue
        yield word
        word = char
            
        state = next_state
    yield word

            
class Out:
    """ wrap text like textwarp.wrap, but preserves \n

    all lines are left aligned
    arguments to write must be strings or iterable and
    return strings of length 1 (chars)
    """

    def __init__(self, width=70):
        self.width = width
        self.lines = ['']

    def copy(self):
        ' like copy, except self.lines = ['']'
        return self.__class__(width = self.width)

    def write(self, arg, literal=False):
        line = self.lines.pop()
        for w in itertext(arg):
            if w == '\n':
                self.lines.append(line)
                line = ''
                continue
            if literal or self.width is None or \
                   line and not line[-1].isspace():
                line += w
                continue
            if not line and w == ' '*len(w):
                continue
            if len(line) + len(w) > self.width:
                if line != '':
                    self.lines.append(line)
                line = w.lstrip()
                continue
            line += w

        self.lines.append(line)
        return self   
    
    def writeln(self, arg='', literal=False):
        self.write(arg, literal)
        self.write('\n')

    def write_literal(self, arg):
        self.write(arg, literal=True)

    def writeln_literal(self, arg):
        self.writeln(arg, literal=True)

    def append(self, out, indent = '', join=''):
        keepends = True
        self.write_literal(join.join(
            [indent + line
             for line in out.render().splitlines(keepends)]))

    def extend(self, out_list, indent = '', infix = ' ',
               join = '', literal = False):
        l = list(out_list)
        if not l: return
        for out in l[:-1]:
            self.append(out, indent, join=join)
            self.write(infix, literal=literal)
        self.append(l[-1], indent, join=join)

    def max_length(self):
        return max([len(l) for l in self.lines])

    def render(self):
        return '\n'.join(self.lines)


class RestTag(Tag):
    start_string = ''
    end_string = ''
    sep = ''
    write_literal = False
    
    def __init__(self, *args, **kwargs):
        super(RestTag, self).__init__(*args, **kwargs)
        self.parse_options(self.attr)

    def parse_options(self, attr):
        pass

    def text(self, width = 70):
        out = Out(width = width)
        self.__rest__(out)
        return out.render()
    
    def __rest__(self, out):
        out.write(self.sep)
        out.write(self.start_string)
        self.write_children(out, self.render_children(out))
        out.write(self.end_string)
        out.write(self.sep)

    def write_children(self, out, child_outs):
        out.extend(child_outs)
        
    def render_children(self, out):
        outs = []
        for child in self:
            child_out = out.copy()
            if isinstance(child, RestTag):
                child.__rest__(child_out)
            else:
                child_out.write(child, literal = self.write_literal)
            outs.append(child_out)
        return outs

class DirectiveMetaclass(type): 
    def __getattr__(self, name): 
        if name[:1] == '_': 
            raise AttributeError(name)
        # convert '_' to '-'
        name = name.replace('_','-')
        classattr = {'name': name}
        cls = type(name, (self.__tagclass__,), classattr) 
        setattr(self, name, cls) 
        return cls 

class UniversalDirective(RestTag):
    sep = '\n\n'
    start_string = '.. '

    def write_children(self, out, child_outs):
        out.write(self.name)
        out.write(':: ')
        out.writeln(child_outs[0].render())
        keys = [attr for attr in dir(self.attr) if not attr.startswith('__')]
        keys.sort()
        for key in keys:
            value = str(getattr(self.attr, key))
            out.write_literal(' ' * len(self.start_string) +':%s: %s\n' \
                              % (key,value))
            
        if len(child_outs) > 1:
            # directive block
            out.writeln(' ' * len(self.start_string))
            out.extend(child_outs[1:], indent = ' ' * len(self.start_string))
    


class rest(UniversalDirective):
    __tagclass__ = RestTag
    __stickname__ = True

    class hyperref(RestTag):
        start_string = '`'
        end_string = '`_'
        write_literal = True

    class emph(RestTag):
        start_string = '*'
        end_string = start_string

    class strongemph(RestTag):
        start_string = '**'
        end_string = start_string

    class inline_literal(RestTag):
        start_string = "``"
        end_string = "``"

    class interpreted_text(RestTag):
        start_string ='`'
        end_string = '`'
        role = ''

        def parse_options(self, attr):
            self.role = getattr(attr, 'role', self.role)
            if self.role:
                self.start_string = ':%s:%s' % (self.role, self.start_string)

    class paragraph(RestTag):
        sep = '\n\n'

    class explicit_markup(RestTag):
        sep = '\n\n'
        start_string = '.. '

        def write_children(self, out, child_outs):
            out.extend(child_outs, join = ' ' * len(self.start_string))

    class hypertarget(RestTag):
        sep = '\n\n'
        start_string = '.. _'
        end_string = ':'
        write_literal = True

    class title(RestTag):
        sep = '\n'
        start_string = '#'
        end_string = '#'
        quotes = """! " # $ % & ' ( ) * + , - . / : ; < = > ? @ [ \ ] ^ _ ` { | } ~""".split()

        def parse_options(self, attr):
            self.start_string = getattr(attr, 'overline', '#')
            self.end_string = getattr(attr, 'underline', '#')
        
        def __rest__(self, out):
            child_outs = self.render_children(out)
            max_length = max([o.max_length() for o in child_outs])
            out.write(self.sep)
            out.writeln_literal(self.start_string * max_length)
            out.extend(child_outs)
            out.writeln()
            out.writeln_literal(self.end_string * max_length)
            out.write(self.sep)

    class list_item(RestTag):
        sep = '\n\n'
        start_string = '* '

        def parse_options(self, attr):
            self.start_string = getattr(attr, 'bullet', '*') + ' '
            if  getattr(attr, 'auto_enumerate', False):
                self.start_string = '#. '
            
        def write_children(self, out, child_outs):
            out.extend(child_outs, join = ' ' * len(self.start_string))
            
    class literal_block(RestTag):
        sep = '\n\n'
        start_string = '::\n\n'
        indent = '  '
        quote = '  '
        quotes = """! " # $ % & ' ( ) * + , - . / : ; < = > ? @ [ \ ] ^ _ ` { | } ~""".split() + [' ']
        write_literal = True
        
        def parse_options(self, attr):
            self.quote = getattr(attr, 'quote', '  ')

        def write_children(self, out, child_outs):
            out.extend(child_outs, indent = self.quote, infix ='', literal = True)

    unidirective = UniversalDirective

    class directive:
        __metaclass__ = DirectiveMetaclass
        __tagclass__ = UniversalDirective
