# Cohoba - a GNOME client for Telepathy
#
# Copyright (C) 2006 Collabora Limited
#
# This package is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This package is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this package; if not, write to the Free Software
# Foundation, 51 Franklin Street, Fifth Floor, Boston, MA, 02110-1301 USA.

from os.path import join
import gtk, gconf, gobject
from gettext import gettext as _

import cohoba
from cohoba.common.Utils import load_icon, global_error_handler
from cohoba.contacts.MergedContactList import get_contact_list, MergedContactList
from cohoba.contacts.Contact import Contact
from cohoba.contacts.ContactGroup import ContactGroup, OverviewContactGroup, get_group_filter
from cohoba.contacts.ui.ContactListWindowManager import get_contact_list_window_manager
from cohoba.contacts.ui.ContactListView import ContactListView		
from cohoba.common.ui.PresenceComboBox import PresenceComboBox
from cohoba.common.DBusProxyProvider import get_proxy_provider
from cohoba.common.TelepathyConstants import (MISSION_CONTROL_UI_INTERFACE,
	MISSION_CONTROL_OBJECT_PATH, MISSION_CONTROL_BUS_NAME)

GCONF = gconf.client_get_default()

class CohobaAppletMainWindow:
	def __init__(self):
		glade = gtk.glade.XML(join(cohoba.SHARED_DATA_DIR, "main.glade"), "main")
		self.main = glade.get_widget("main")
		self.main.hide()
		
		self.contacts = ContactListView(OverviewContactGroup())
		self.contacts.show()
		glade.get_widget("contacts_scroll").add(self.contacts)
		
		self.message = glade.get_widget("message")
	
		self.presence_combobox = PresenceComboBox(self.message)
		self.presence_combobox.show()
		glade.get_widget("presence_table").pack_start(self.presence_combobox, False)
		
		self.main.connect('delete-event', self.on_window_delete)
	
	def on_window_delete(self, win, event):
		self.main.hide()
		return True

class CohobaAppletWidget:
	def __init__(self, applet):
		self.applet = applet
		
		self.applet.connect("button-press-event", self.on_applet_button_press)
		self.applet.setup_menu_from_file (
			cohoba.SHARED_DATA_DIR, "Cohoba.xml",
			None, [
			("About", self.on_about),
			("Accounts", self.on_accounts),
			])
			
	def make_menu(self, event):
		raise NotImplementedError
		
	def on_about (self, component, verb):
		print 'FIXME: to be implemnted', component, verb
	
	def on_accounts (self, component, verb):
		mission_control_ui_iface = get_proxy_provider().get_iface(
				MISSION_CONTROL_OBJECT_PATH,
				MISSION_CONTROL_UI_INTERFACE)
		mission_control_ui_iface.ShowAccounts(reply_handler=lambda: None, error_handler=global_error_handler)
								
	def on_applet_button_press(self, widget, event):
		if event.button == 1:
			self.make_menu(event)
#			self.main_ui.main.present_with_time(event.time)
			return True
			
		return False

class CohobaIconAppletWidget(CohobaAppletWidget, gtk.Image):
	def __init__(self, applet, icon):
		CohobaAppletWidget.__init__(self, applet)
		gtk.Image.__init__(self)
		self.applet_size = applet.get_size()
		self.set_img(icon)
		applet.set_background_widget(self)
		
		# FIXME: For some reasons this isn't working, the applet shrinks to (0,0)
		#applet.connect("size_allocate", self.on_size_allocate)
		applet.connect("change-orient", self.on_change_orient)
		applet.connect("change-size", self.on_change_size)
		
		self.preferences_path = applet.get_preferences_key() or cohoba.GCONF_COHOBA_DIR
		self.tooltips = gtk.Tooltips()
			
	def set_img(self, icon=None, pixbuf=None):
		if icon != None:
			self.icon = icon
		
		if pixbuf == None:
			pixbuf = load_icon(self.icon, self.applet_size-2, self.applet_size-2)
		
		if pixbuf.get_height() != self.applet_size-2:
			pixbuf = pixbuf.scale_simple(self.applet_size-2, self.applet_size-2, gtk.gdk.INTERP_BILINEAR)
			
		self.set_from_pixbuf(pixbuf)
		
	def on_size_allocate(self, widget, allocation):
		if allocation.height == self.applet_size:
			return
		
		self.applet_size = allocation.height
		self.set_img()
	
	def on_change_orient(self, applet, orient):
		pass
		
	def on_change_size(self, applet, size):
		pass

class CohobaContactIconAppletWidget(CohobaIconAppletWidget):
	def __init__(self, applet):
		CohobaIconAppletWidget.__init__(self, applet, "stock_person")
		self.contact = None
		self.contact_updated_id = 0
		self.check_stored_contact_id = 0
		
		self.tooltips.set_tip (self.applet, _("Choose a contact in the menu"))
		
		contact_name = GCONF.get_string(join(self.preferences_path, "contact"))
		contact_conn_path = GCONF.get_string(join(self.preferences_path, "connection"))
		if contact_name != None and contact_conn_path != None:
			# I can't think of a better way right now to poll for updates.
			# I think the contact list should now when the contacts can be queried for their names
			self.check_stored_contact_id = gobject.timeout_add(1000, self.check_stored_contact, contact_name, contact_conn_path)
	
	def check_stored_contact(self, name, conn_path):
		contact = [contact for contact in get_contact_list() if not contact.is_group and contact.name == name and contact.conn_path == conn_path]
		if len(contact) != 0:
			self.set_contact(contact[0])
			return False
		
		return True
		
	def set_contact(self, contact):
		# Stop looking for a contact
		if self.check_stored_contact_id != 0:
			gobject.source_remove(self.check_stored_contact_id)
			
		if self.contact_updated_id != 0:
			self.contact.disconnect(self.contact_updated_id)
			
		self.contact = contact
		self.contact_updated_id = self.contact.connect('updated', self.on_contact_updated)
		self.on_contact_updated(self.contact)
		
		print 'Storing contact name=%s in %s/contact' % (self.contact.name, self.preferences_path)
		GCONF.set_string(join(self.preferences_path, "contact"), self.contact.name)
		GCONF.set_string(join(self.preferences_path, "connection"), self.contact.conn_path)
		
		self.tooltips.set_tip (self.applet, self.contact.alias or self.contact.name)
	
	def on_contact_updated(self, contact):
		self.set_img(pixbuf=self.contact.get_icon())
		
	def make_menu(self, event):
		menu = gtk.Menu()
		menu.attach_to_widget(self, None)
		
		if self.contact != None:
			item = gtk.MenuItem(_("Actions for  '%s'") % (self.contact.alias or self.contact.name))
			item.set_sensitive(False)
			menu.append(item)
		
			def contact_clicked(item, contact):
				contact.action()
				menu.popdown()
		
			item = gtk.MenuItem(_("Chat"))
			item.connect("activate", contact_clicked, self.contact)
			menu.append(item)
			
			menu.append(gtk.SeparatorMenuItem())
			
		item = gtk.MenuItem(_("Choose Contact"))
		menu.append(item)
		
		submenu = gtk.Menu()
		item.set_submenu(submenu)	

		def contact_clicked(item, contact):
			self.set_contact(contact)
			menu.popdown()
			
		for contact in [contact for contact in get_contact_list() if not contact.is_group]:
			label = gtk.Label()
			label.set_markup(contact.get_description())
			label.set_alignment(0.0, 0.0)
					
			image = gtk.Image()
			image.set_from_pixbuf(contact.get_icon())

			box = gtk.HBox(False, 6)
			box.pack_start(image, expand=False)
			box.pack_start(label)

			item = gtk.MenuItem()
			item.add(box)		
			item.connect('activate', contact_clicked, contact)
			submenu.append(item)
		
		

		menu.show_all()
		menu.popup(None, None, None, event.button, event.time)
	
class CohobaGroupIconAppletWidget(CohobaIconAppletWidget):
	def __init__(self, applet):
		CohobaIconAppletWidget.__init__(self, applet, "stock_people")
		self.group = OverviewContactGroup()
		
		self.tooltips.set_tip (self.applet, _("Uncategorized contacts"))
		
		group_name = GCONF.get_string(join(self.preferences_path, "group"))
		if group_name != None:
			group = [contact for contact in get_contact_list() if contact.is_group and contact.name == group_name]
			if len(group) != 0:
				self.set_group(group[0])
			
	def set_group(self, group):
		self.group = group
		self.set_img("stock_people")
		
		print 'Storing group name=%s in %s/group' % (self.group.name, self.preferences_path)
		GCONF.set_string(join(self.preferences_path, "group"), self.group.name)
		
		self.tooltips.set_tip (self.applet, _("Contacts in '%s'") % self.group.name)
		
	def make_menu(self, event):
		menu = gtk.Menu()
		menu.attach_to_widget(self, None)

		if self.group != None:
			item = gtk.MenuItem(_("Contacts in '%s'") % self.group.name)
			item.set_sensitive(False)
			menu.append(item)
			menu.append(gtk.SeparatorMenuItem())

		def contact_clicked(item, contact):
			contact.action()
			menu.popdown()
			
		# ------ Start horrible Hack to prevent apparition of a Assertion Failure when dealing with
		# python iteration of the model iter->stamp stuff.
		# As a workaround use the foreach() of the model
		flst = get_group_filter(self.group, get_contact_list())
		def append_items(model, path, iter, data):
			contact = model[path][0]
			if not contact.is_group and self.group.is_in_group(contact):
				label = gtk.Label()
				label.set_markup(contact.get_description())
				label.set_alignment(0.0, 0.0)
						
				image = gtk.Image()
				image.set_from_pixbuf(contact.get_icon())

				box = gtk.HBox(False, 6)
				box.pack_start(image, expand=False)
				box.pack_start(label)

				item = gtk.MenuItem()
				item.add(box)		
				item.connect('activate', contact_clicked, contact)
				menu.append(item)
		flst.foreach(append_items, None)
		
		menu.append(gtk.SeparatorMenuItem())
		
		item = gtk.MenuItem(_("Choose Group"))
		menu.append(item)
		
		submenu = gtk.Menu()
		item.set_submenu(submenu)

		def group_clicked(item, group):
			self.set_group(group)
			menu.popdown()
			
		for group in [group for group in get_contact_list() if group.is_group]:
			label = gtk.Label()
			label.set_markup(group.get_description())
			label.set_alignment(0.0, 0.0)
					
			image = gtk.Image()
			image.set_from_pixbuf(group.get_icon())

			box = gtk.HBox(False, 6)
			box.pack_start(image, expand=False)
			box.pack_start(label)

			item = gtk.MenuItem()
			item.add(box)		
			item.connect('activate', group_clicked, group)
			submenu.append(item)
			
		menu.show_all()
		menu.popup(None, None, None, event.button, event.time)			
		
class CohobaMainIconAppletWidget(CohobaIconAppletWidget):
	def __init__(self, applet):
		CohobaIconAppletWidget.__init__(self, applet, "cohoba.png")
		self.main_ui = CohobaAppletMainWindow()
		
	def make_menu(self, event):
		menu = gtk.Menu()
		menu.attach_to_widget(self, None)
		
		def contact_clicked(item, contact):
			contact.action()
			menu.popdown()
		
		item = gtk.MenuItem(_("Open Main Window"))
		item.connect("activate", lambda item: self.main_ui.main.present_with_time(gtk.get_current_event_time()))
		menu.append(item)
		
		menu.append(gtk.SeparatorMenuItem())
		
		for contact in [row[0] for row in get_group_filter(OverviewContactGroup(), get_contact_list())]:
			label = gtk.Label()
			label.set_markup(contact.get_description())
			label.set_alignment(0.0, 0.0)
					
			image = gtk.Image()
			image.set_from_pixbuf(contact.get_icon())

			box = gtk.HBox(False, 6)
			box.pack_start(image, expand=False)
			box.pack_start(label)

			item = gtk.MenuItem()
			item.add(box)		
			item.connect('activate', contact_clicked, contact)
			menu.append(item)	
		
		menu.show_all()
		menu.popup(None, None, None, event.button, event.time)
