# Cohoba - a GNOME client for Telepathy
#
# Copyright (C) 2006 Collabora Limited
#
# This package is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This package is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this package; if not, write to the Free Software
# Foundation, 51 Franklin Street, Fifth Floor, Boston, MA, 02110-1301 USA.

import gobject, dbus

from cohoba.common.Utils import global_error_handler, SignalConnection
from cohoba.common.DBusProxyProvider import get_proxy_provider

from telepathy.client.channel import Channel
from telepathy.constants import CONNECTION_HANDLE_TYPE_CONTACT, CHANNEL_TEXT_MESSAGE_TYPE_NORMAL, CHANNEL_TEXT_MESSAGE_FLAG_TRUNCATED
from telepathy.interfaces import (CONN_INTERFACE, CHANNEL_INTERFACE,
	CHANNEL_TYPE_TEXT)

class TelepathyConversation(gobject.GObject):
	"""Proxy for a text channel: converts relevant DBUS signals into
	glib signals.
	"""

	__gsignals__ = {
		"message-received" : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, [gobject.TYPE_UINT, gobject.TYPE_UINT, gobject.TYPE_UINT, gobject.TYPE_UINT, gobject.TYPE_UINT, gobject.TYPE_STRING]),
		"closed" : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, []),
	}
	
	def __init__(self, conn_path, obj_path, contact=None, acknowledge_messages=False):
		"""Constructor.

		conn_path is the DBUS object path of a Connection.
		obj_path is the DBUS object path of a TextChannel created by
		that Connection.
		contact is a Contact object, if any - it may be None,
		in the case of multi-user chat.
		acknowledge_messages is True iff we will be showing messages
		to the user, i.e. it is appropriate for us to acknowledge
		them.
		"""
		gobject.GObject.__init__(self)

		self.contact = contact
		self.acknowledge_messages = acknowledge_messages
		
		self.conn_iface = get_proxy_provider().get_iface(conn_path, CONN_INTERFACE)
		service_name = get_proxy_provider().get_service(conn_path)
		self.textchannel_iface = Channel(service_name, obj_path,
			ready_handler=self.__on_channel_introspected)
	
	def __on_channel_introspected(self, chan):
		self.__Closed_conn = SignalConnection(self.textchannel_iface[CHANNEL_INTERFACE], 'Closed', self.handle_closed)
		text_interface = self.textchannel_iface[CHANNEL_TYPE_TEXT]
		self.__Received_conn = SignalConnection(text_interface, 'Received', self.handle_incoming)
		self.__Sent_conn = SignalConnection(text_interface, 'Sent', self.handle_outgoing)
		text_interface.ListPendingMessages(False, reply_handler=self.on_pending_messages_listed, error_handler=global_error_handler)

	def on_pending_messages_listed(self, pending_messages):
		for msg in pending_messages:
			self.handle_incoming(*msg)

	# Subclasses use that ----------------------
	def handle_incoming(self, id, timestamp, sender, message_type, flags, message):
		print ("message from %s at %s: id %d, type %d %r (flags %d)" %
			(sender, timestamp, id, message_type, message, flags))
		if self.acknowledge_messages:
			self.textchannel_iface[CHANNEL_TYPE_TEXT].AcknowledgePendingMessages(dbus.Array([id],signature='u'), reply_handler=lambda: None, error_handler=global_error_handler)
		if flags & CHANNEL_TEXT_MESSAGE_FLAG_TRUNCATED:
			print "According to flags, text message was truncated"
		self.emit('message-received', id, timestamp, sender, message_type, flags, message)
	
	def handle_outgoing(self, timestamp, message_type, message):
		pass
	
	def handle_closed(self):
		self.emit('closed')

	# External interface ---------------------		                       
	def send_message(self, message, message_type=CHANNEL_TEXT_MESSAGE_TYPE_NORMAL):
		self.textchannel_iface[CHANNEL_TYPE_TEXT].Send(dbus.UInt32(message_type), message, reply_handler=lambda: None, error_handler=global_error_handler)

	def close(self):
		self.textchannel_iface[CHANNEL_INTERFACE].Close(reply_callback=lambda: None, error_callback=global_error_handler)
