# Cohoba - a GNOME client for Telepathy
#
# Copyright (C) 2006 Collabora Limited
#
# This package is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This package is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this package; if not, write to the Free Software
# Foundation, 51 Franklin Street, Fifth Floor, Boston, MA, 02110-1301 USA.

import gtk

from cohoba.common.Utils import global_error_handler
from cohoba.contacts.Contact import Contact
from cohoba.contacts.ContactGroup import ContactGroup, FavoriteContactGroup, group_for_serialized_data
from cohoba.common.TelepathyGroup import TelepathyGroup
from cohoba.common.Store import GroupStore
from cohoba.common.PyListStore import PyListStore


class _TelepathySubscribeList(TelepathyGroup):
	"""The 'subscribe' list, used by MergedContactList.

	Values in `members` are GTK TreeRowReference instances.

	get_members, get_local_members, get_remote_members return
	Contact instances.
	"""

	# FIXME: this temporarily uses internal interfaces of
	# MergedContactList

	def __init__(self, conn_path, obj_path, handle, merged_contact_list):
		TelepathyGroup.__init__(self, conn_path, obj_path, handle)
		self.__contact_list = merged_contact_list

	def _member_for_data(self, row):
		return self.__contact_list._member_for_data(row)

	def _add_member(self, handle, flag):
		return self.__contact_list._add_member(self.conn_path, handle, flag)

	def _remove_member(self, row):
		self.__contact_list._remove_member(row)

	def _channel_closed(self):
		self.__contact_list._channel_closed(self.conn_path)


class MergedContactList(PyListStore):	
	"""A contact list consisting of the union of all server-side
	subscribe lists.
	"""

	def __init__(self):
		PyListStore.__init__(self)
		# Sorting
		self.set_sort_func(1, self.on_sort_item)
		self.set_sort_column_id(1, gtk.SORT_ASCENDING)

		self.telepathy_groups = {}
		
		# Load saved groups
		self.load_contact_groups()
	
	def on_sort_item(self, model, iter1, iter2):
		item1 = model[iter1].get_sort_key()
		item2 = model[iter2].get_sort_key()
		#print '********SORT::::::::::', item1, item2, item1 < item2, item1 > item2
		if item1 < item2:
			return -1
		elif item1 > item2:
			return 1
		else:
			return 0
	
	# --- Contact Group related methods ---
	def load_contact_groups(self):
		contact_groups = GroupStore().load()
				
		favorites_found = False
		if contact_groups != None:
			for data in contact_groups:
				group = group_for_serialized_data(data)
				if group == None:
					continue
					
				self.add_contact_group(group)

				if group.is_favorite():
					favorites_found = True
			
		if not favorites_found:
			self.add_contact_group(FavoriteContactGroup())
		
	def save_contact_groups(self):
		GroupStore().save([group.serialize() for group in self.get_contact_groups()])
	
	def get_contact_groups(self):
		"""Return all the stored contact groups."""
		return [group for group in self if group.is_group]
		
	def add_contact_group(self, group):
		self.append(group)
		self.save_contact_groups()
	# ---
	
	# --- Public interface for contacts ---
	def add_telepathy_group(self, conn_path, obj_path, handle):
		"""Arrange to be notified about changes to the Telepathy
		``subscribe`` contact list for a given Connection.

		:Parameters:
			conn_path : string
				The DBus object path of the Connection
			obj_path : string
				The DBus object path of the subscribe list,
				which implements Channel.Type.ContactList
				(and Channel)
			handle : integer
				The handle of the subscribe list Channel
		"""

		if conn_path in self.telepathy_groups:
			return
		self.telepathy_groups[conn_path] = _TelepathySubscribeList(conn_path, obj_path, handle, self)
	
	def add_member(self, conn_path, name, reason="No Reason."):
		"""Request that the named contact is added to the subscribe list.

		:Parameters:
			conn_path : string
				The Connection whose subscribe list is to be
				altered
			name : string
				The name (e.g. JID) of the contact to be
				added
			reason : string
				Reason to be passed to Telepathy
				(e.g. shown to the contact in the
				subscription request)
		"""
		print 'Trying to add:', conn_path, name, reason
		return self.telepathy_groups[conn_path].add_member(name, reason="No Reason.")
	
	def remove_member(self, conn_path, handle, reason="No Reason."):
		"""Request that the given contact handle is removed from
		the subscribe list.

		:Parameters:
			conn_path : string
				The Connection whose subscribe list is to be
				altered
			handle : integer
				The handle of a contact currently on the
				subscribe list
			reason : string
				Reason to be passed to Telepathy
		"""
		print 'Trying to remove:', conn_path, handle, reason
		self.telepathy_groups[conn_path].remove_member(handle, reason="No Reason.")
	
	def get_contacts(self):
		"""Return a list of the the Contact objects held by self."""
		return [contact for contact in self if not contact.is_group]
		
	def get_contact(self, conn_path, handle):
		"""If we have a Contact for the given (conn_path,handle),
		return it. Otherwise return None.
		"""
		contact = [contact for contact in self.get_contacts() if contact.conn_path == conn_path and contact.handle == handle]
		if len(contact) >= 1:
			return contact[0]
	# --- 
	
	# --- Abstract methods from TelepathyGroup ---
	def _member_for_data(self, row):
		# Even though not all self's members are TelepathyGroupMembers,
		# if row is one of the values that we gave to TelepathyGroup,
		# then this will return a Contact, which *is* a
		# TelepathyGroupMember, so this is safe to do.
		return self[row.get_path()]
			
	def _add_member(self, conn_path, handle, flag):
		print 'Adding member:', conn_path, handle
		contact=Contact(conn_path, handle)
		contact.set_flag(flag)
		iter = self.append(contact)
		contact.connect('updated', self.on_contact_updated, conn_path)
		
		return gtk.TreeRowReference(self, self.get_path(iter))
		
	def _remove_member(self, row):
		print 'Removing member:', self[row.get_path()]
		iter = self.get_iter(row.get_path())
		if iter != None:
			self.remove(iter)
	
	def _channel_closed(self, conn_path):
		del self.telepathy_groups[conn_path]
		
	def on_contact_updated(self, contact, conn_path):
		# Watch out if a Contact updates itself while the connection isn't here anymore
		if conn_path in self.telepathy_groups and contact.handle in self.telepathy_groups[conn_path].members:
			row = self.telepathy_groups[conn_path].members[contact.handle]
			if row.get_path() == None:
				return
			path, iter = row.get_path(), self.get_iter(row.get_path())		
			self.row_changed(path, iter)
	# --- 
	
	
contact_list = MergedContactList()
def get_contact_list():
	"""Return the single global MergedContactList instance."""
	return contact_list
