# Cohoba - a GNOME client for Telepathy
#
# Copyright (C) 2006 Collabora Limited
#
# This package is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This package is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this package; if not, write to the Free Software
# Foundation, 51 Franklin Street, Fifth Floor, Boston, MA, 02110-1301 USA.

import gobject, gtk, dbus

from telepathy.constants import (CONNECTION_STATUS_DISCONNECTED,
	CONNECTION_STATUS_CONNECTED, CONNECTION_STATUS_CONNECTING,
	CONNECTION_STATUS_REASON_NONE_SPECIFIED)
from telepathy.interfaces import (CONN_MGR_INTERFACE, CONN_INTERFACE,
	CONN_INTERFACE_PRESENCE)

from cohoba.common.Utils import (global_error_handler, ignore_error_handler,
	conn_reason_to_string, conn_status_to_string, SignalConnection,
	get_logger)
from cohoba.common.DBusProxyProvider import get_proxy_provider

LOG = get_logger("Houston")

class Account(gobject.GObject):
	__gsignals__ = {
		"account-changed": (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, []),
		"connected": (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, [gobject.TYPE_BOOLEAN]),
		"enabled": (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, [gobject.TYPE_BOOLEAN])
	}
	def __init__(self, name=None):
		gobject.GObject.__init__(self)
		
		self.name = name
		
		# Saved Values
		self.params = {}		
		self.profile = None
		self.save_password = False
		self.enabled = False
		self.cached_conn_obj = ""
		
		# Dynamic value
		self.connection = None
		self.status = (CONNECTION_STATUS_DISCONNECTED, CONNECTION_STATUS_REASON_NONE_SPECIFIED)
		self.reconnect_required = False
		self.failed_connection_attempts = 0
		
		# Read from gconf
		self.load()
		
	def is_connectable(self):
		return (self.is_valid() and self.status[0] == CONNECTION_STATUS_DISCONNECTED)
	
	def is_connected(self):
		return (self.status[0] == CONNECTION_STATUS_CONNECTED)
	
	def is_connecting(self):
		return (self.status[0] == CONNECTION_STATUS_CONNECTING)
		
	def is_valid(self):
		return (self.name != None and self.profile != None and self.profile.has_required_parameters(self.params))
		
	def get_username(self):
		if "account" in self.params:
			return self.params["account"]
		
	def get_password(self):
		if "password" in self.params:
			return self.params["password"]
	
	def get_name(self):
		return self.name
	
	def is_connection(self, conn_obj):
		return (self.cached_conn_obj == conn_obj)
	
	def set_username(self, username):
		self.set_param("account", username)
	
	def set_password(self, password):
		self.set_param("password", password)
		
	def set_param(self, name, param):
		if param == None or param == "":
			if name in self.params:
				del self.params[name]
		else:
			self.params[name] = param
		self.account_changed(setting_change=True)
	
	def set_params(self, params):
		self.params = params
		self.account_changed(setting_change=True)
		
	def set_profile(self, profile):
		if profile != self.profile:
			self.profile = profile
			self.account_changed(setting_change=True)
	
	def set_save_password(self, save_password):
		if save_password != self.save_password:
			self.save_password = save_password
			self.account_changed(setting_change=True)
	
	def set_enabled(self, enabled):
		if enabled != self.enabled:
			self.enabled = enabled
			self.account_changed()
			self.emit("enabled", enabled)
							
	def set_presence(self, name, message):
		presence_iface = get_proxy_provider().get_iface(self.connection[1], CONN_INTERFACE_PRESENCE)
		presence_iface.SetStatus(
			{name: dbus.Dictionary({"message": message}, signature = "sv")},
			reply_handler=lambda: None,
			# We can ignore errors here. Presence will just not be set.
			error_handler=ignore_error_handler)
	
	def account_changed(self, setting_change=False, save=True):
		if save:
			self.save()
		if setting_change and (self.is_connected() or self.is_connecting()):
			self.reconnect_required = True
			
		self.emit('account-changed')
		
	def connect_account(self):
		if self.status[0] == CONNECTION_STATUS_CONNECTING or self.status[0] == CONNECTION_STATUS_CONNECTED:
			return
				
		if self.connection == None:
			LOG.info("Requesting connection: %s", self)
			get_proxy_provider().create_proxy(self.profile.bus_name, self.profile.obj_path)
			connmgr_iface = get_proxy_provider().get_iface(self.profile.obj_path, CONN_MGR_INTERFACE)

			params = dbus.Dictionary(self.profile.prepare_params(self.params), signature="sv")

			self.connection = connmgr_iface.RequestConnection(self.profile.protocol, params)
			
		# To avoid making multiple RequestConnection Calls simultaneously, we do it sync.
		self.connect_account_to_connection(*self.connection)
		
	def connect_account_to_connection(self, conn_service, conn_obj, do_connect=True):
		get_proxy_provider().create_proxy(conn_service, conn_obj)
		conn_iface = get_proxy_provider().get_iface(conn_obj, CONN_INTERFACE)

		# Carefully connect only once to the signal per RequestConnection calls. Disconnected when Status signal is emitted with
		# disconnection parameter
		self.connection = (conn_service, conn_obj, SignalConnection(conn_iface, 'StatusChanged', self.on_status_changed))
		self.cached_conn_obj = str(conn_obj)
		self.account_changed()
		
		LOG.info('Connection Created: %s', self)
		
		exception = None
		try:
			if do_connect:
				conn_iface.Connect()
		except dbus.DBusException, e:
			# FIXME: How to get the proper exception type here ? We only want NetworkErrors..
			self.failed_connection_attempts += 1
			self.disconnect_account()
			exception = e
		
		# If we couldn't connect, trow the exception now we have the status
		if exception != None:
			raise exception
					
	def on_status_changed(self, status, reason=None):
		# Use the new reason, or use the previous (still valid) one
		if reason == None:
			reason = self.status[1]
		
		if status == CONNECTION_STATUS_DISCONNECTED:
			# Don't listen to status changes anymore
			self.connection[2].disconnect()
			# Since We're getting disconnected, we don't need a disconnection again
			self.reconnect_required = False
			# Delete existing cached connection so we go through RequestConnection again
			self.connection = None
			# We're disconnected !
			self.emit("connected", False)
		elif status == CONNECTION_STATUS_CONNECTED:
			# Reset failed connection counter
			self.failed_connection_attempts = 0
			
			self.emit("connected", True)
		
		self.status = (status, reason)
		self.account_changed(False, False)
		LOG.debug('Status changed: %s: %s(%s) %s(%s)', self, status, conn_status_to_string(status), reason, conn_reason_to_string(reason))
	
	def disconnect_account(self):
		if self.connection == None:
			return
			
		LOG.info('Disconnecting: %s', self)
		conn_iface = get_proxy_provider().get_iface(self.connection[1], CONN_INTERFACE)				
		try:
			conn_iface.Disconnect()
		except:
			LOG.exception("Error while disconnecting account: %s", self)
	
	def save(self):
		pass
	
	def load(self):
		pass
		
	def __repr__(self):
		return "Account(%s): %s / %s" % (self.name, dict([(key, value) for key, value in self.params.items() if key != "password"]), self.profile)
