import gobject, cohoba, gconf
from cohoba.common.Utils import singleton, get_logger
from os.path import join, basename
from cohoba.houston.Account import Account
from cohoba.common.Profile import Profile
from cohoba.common.ProfileList import ProfileList

GCONF = gconf.client_get_default()
LOG = get_logger("Houston")

def gconf_value_for_py_value(value):
	gconf_val = None
	if type(value) in (str, unicode):
		gconf_val = gconf.Value(gconf.VALUE_STRING)
		gconf_val.set_string(value)
	elif type(value) == int:
		gconf_val = gconf.Value(gconf.VALUE_INT)
		gconf_val.set_int(value)
	elif type(value) == bool:
		gconf_val = gconf.Value(gconf.VALUE_BOOL)
		gconf_val.set_bool(value)
	else:
		LOG.warning('No gconf value for: %s %s', type(value), value)
		
	return gconf_val

def py_value_for_gconf_value(value):
	if value.type == gconf.VALUE_BOOL:
		return value.get_bool()
	elif value.type == gconf.VALUE_INT:
		return value.get_int()
	elif value.type == gconf.VALUE_STRING:
		return value.get_string()
	else:
		LOG.warning('No python type for gconf value: %s', value.type)
			
def get_account_names():
	return [basename(path) for path in GCONF.all_dirs(cohoba.GCONF_HOUSTON_ACCOUNTS_DIR)]
		
def get_next_account_name(name):
	names = get_account_names()

	n = 0
	next_name = "%s%d" % (name, n)
	while next_name in names:
		n += 1
		next_name = "%s%d" % (name, n)
			
	return next_name
	
def load_all_accounts():
	return [GConfAccount(account_name) for account_name in get_account_names()]

def create_new_account():
	return GConfAccount()
	
class GConfAccount(Account):
	def __init__(self, name=None):
		Account.__init__(self, name)
				
	def save(self):
		if not self.is_valid():
			return
		
		LOG.debug('Saving account in GCONF: %s', self)
		# First purge old values
		self.delete()
		
		path = join(cohoba.GCONF_HOUSTON_ACCOUNTS_DIR, self.get_name())
		data = self.serialize()
		
		for key, value in data.items():
			GCONF.set(join(path, key), gconf_value_for_py_value(value))
				
	def load(self):
		if self.get_name() == None:
			return
		
		LOG.debug('Loading account in GCONF: %s', self.get_name())
		path = join(cohoba.GCONF_HOUSTON_ACCOUNTS_DIR, self.get_name())
		self.unserialize(dict([(basename(entry.key), py_value_for_gconf_value(entry.value)) for entry in GCONF.all_entries(path)]))
		LOG.debug('--Loaded')
		
	def delete(self):
		if self.get_name() == None:
			return
			
		GCONF.recursive_unset(join(cohoba.GCONF_HOUSTON_ACCOUNTS_DIR, self.get_name()), 1)
		GCONF.suggest_sync()
	
	def serialize(self):
		# FIXME save_password and gnome-keyring stuff too
		params = self.params.copy()
		params.update({
			"profile": self.profile.name,
			"protocol": self.profile.protocol,
			"manager": self.profile.manager,
			"save_password": self.save_password,
			"enabled": self.enabled,
			"cached_conn_obj": self.cached_conn_obj,
		})
		return params

	def unserialize(self, data):
		LOG.debug('Unserializing: %r', dict([(key, value) for key, value in data.items() if key != "password"]))
		
		# Detect bogus entries
		if "manager" not in data:
			return
		if "profile" not in data or "protocol" not in data:
			return
		
		profile = data["protocol"]
		if "profile" in data:
			profile = data["profile"]
			
		manager = data["manager"]
		
		self.profile = ProfileList().find(manager, profile)
		self.save_password = data["save_password"]
		self.enabled = data["enabled"]
		self.cached_conn_obj = data["cached_conn_obj"]
		
		del data["save_password"]
		del data["enabled"]
		del data["profile"]
		del data["protocol"]
		del data["manager"]
		del data["cached_conn_obj"]
		
		self.params = data
		
		self.account_changed(setting_change=True, save=False)
	
	def generate_name(self):
		if self.profile == None or self.name != None:
			return
			
		self.name = get_next_account_name(self.profile.get_id())
		self.account_changed()
