# Cohoba - a GNOME client for Telepathy
#
# Copyright (C) 2006 Collabora Limited
#
# This package is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This package is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this package; if not, write to the Free Software
# Foundation, 51 Franklin Street, Fifth Floor, Boston, MA, 02110-1301 USA.

import gtk, gobject, time
import pango
from gettext import gettext as _

from telepathy.constants import CHANNEL_TEXT_MESSAGE_TYPE_NORMAL, CHANNEL_TEXT_MESSAGE_TYPE_ACTION, CHANNEL_TEXT_MESSAGE_TYPE_NOTICE, CONNECTION_HANDLE_TYPE_CONTACT
#CHANNEL_TEXT_MESSAGE_TYPE_NORMAL, CHANNEL_TEXT_MESSAGE_TYPE_ACTION, CHANNEL_TEXT_MESSAGE_TYPE_NOTICE = range(3)

class ConversationTextView(gtk.TextView):
	def __init__(self, conversation):
		gtk.TextView.__init__(self)
		
		self.conversation = conversation
		self.conversation.connect('message-received', self.on_message_received)
		self.conversation.connect('closed', self.on_conversation_closed)
		
		properties = {
			"overwrite": False,
			"accepts_tab": True,
			"justification": gtk.JUSTIFY_LEFT,
			"wrap_mode": gtk.WRAP_WORD,
			"pixels_above_lines": 0,
			"pixels_below_lines": 0,
			"pixels_inside_wrap": 0,
			"left_margin": 6,
			"right_margin": 6,
			"indent": 0,
		}
		for prop, val in properties.items():
			self.set_property(prop, val)
			
		buf = self.get_buffer()
		
		# Appearance Preferences
		#self.modify_font(pango.FontDescription('Monospace 9'))
		
		# Buffer Tags
		buf.create_tag("other",
			editable=False,
			left_margin=36,
			pixels_above_lines=12)
			
		buf.create_tag("me",
			editable=False,
			foreground="#555555",
			left_margin=12,
			pixels_above_lines=12)
		
		buf.create_tag("name-me",
			editable=False,
			foreground="#555555",
			left_margin=6,
			weight=pango.WEIGHT_BOLD,
			pixels_above_lines=12)
		
		buf.create_tag("name-other",
			editable=False,
			left_margin=6,
			weight=pango.WEIGHT_BOLD,
			pixels_above_lines=12)
			
		buf.create_tag("separator",
			editable=False)
		
		buf.create_tag("timestamp",
			editable=False,
			scale=pango.SCALE_X_SMALL,
			foreground="#bbbbbb",
			justification=gtk.JUSTIFY_CENTER,
			pixels_above_lines=12,
			)
		
		buf.create_tag("action",
			editable=False,
			foreground="#bbbb88",
			justification=gtk.JUSTIFY_RIGHT,
			pixels_above_lines=12,
			)
		
		buf.create_tag("notice",
			editable=False,
			foreground="#bb88bb",
			justification=gtk.JUSTIFY_CENTER,
			pixels_above_lines=12,
			)
		
		self.tags = {
			CHANNEL_TEXT_MESSAGE_TYPE_ACTION: "action",
			CHANNEL_TEXT_MESSAGE_TYPE_NOTICE: "notice",
		}
		
		# Buffer Marks
		buf.insert_with_tags_by_name(self.get_me_start_iter(), "\n", "separator")
		buf.create_mark("me-start", self.get_me_start_iter(), True)
		buf.create_mark("conversation", self.get_conversation_start_iter(), self.get_conversation_gravity())
		buf.place_cursor(self.get_me_start_iter())
		
		# Remember who was the last person talking
		self.talking = "me"
		
		# View/Buffer Signals
		self.connect('key-press-event', self.on_key_press_event)
		self._insert_text_id = buf.connect("insert-text", lambda buf, iter, text, length: None)

		self.print_timestamp = True
		self.timestamp_tick_id = gobject.timeout_add(2*60*1000, self.on_timestamp_tick)
		self.on_timestamp_tick()
	
	def on_timestamp_tick(self):
		if self.print_timestamp:
			self.add_message(time.strftime("%H:%M:%S"), None, "timestamp")
			self.print_timestamp = False
		
	def on_conversation_closed(self, conversation):
		print 'Conversation closed'
		self.set_sensitive(False)
		gobject.source_remove(self.timestamp_tick_id)
		
	def on_key_press_event(self, view, event):
		buf = view.get_buffer()
		if event.keyval == gtk.keysyms.Return:		
			if (event.state & gtk.gdk.CONTROL_MASK) != 0:
				# Pressing Ctrl-Enter means insert a newline
				buf.insert(buf.get_iter_at_mark(buf.get_insert()), "\n")
			else:
				# Pressing Enter means send message
				text = buf.get_slice(buf.get_iter_at_mark(buf.get_mark("me-start")), self.get_me_stop_iter(), False)
				text, type = self.prepare_outgoing_message(text)
				
				buf.delete(buf.get_iter_at_mark(buf.get_mark("me-start")), self.get_me_stop_iter())
				self.add_message(text, type, "me", self.get_conversation_gravity(), name=_("Me"))
				self.get_buffer().insert_with_tags_by_name(self.get_me_start_iter(), "\n", "separator")
				
				buf.move_mark(buf.get_mark("conversation"), self.get_conversation_mark_iter())
				buf.move_mark(buf.get_mark("me-start"), self.get_me_start_iter())
				
				buf.place_cursor(self.get_me_start_iter())

				self.scroll_mark_onscreen(buf.get_mark("me-start"))
				
				self.send_message(text, type)
					
			return True
		
		# If we type in an uneditable zone, move to editable zone
		iter = buf.get_iter_at_mark(buf.get_insert())
		if not iter.can_insert(True) or iter.compare(self.get_conversation_start_iter()) == 0:
			buf.place_cursor(self.get_me_stop_iter())

		return False

	def get_contact_name(self, sender):
		contact = self.conversation.contact
		if contact is not None and sender == contact.handle:
			name = contact.alias
			if name == None:
				name = contact.name
		else:
			# this is only likely to happen during a MUC
			# FIXME: make a better API for this - we probably
			# already inspected the handle
			name = self.conversation.conn_iface.InspectHandles(CONNECTION_HANDLE_TYPE_CONTACT, [sender])[0]
		return name
		
	def on_message_received(self, conversation, id, timestamp, sender, message_type, flags, message):
		name = self.get_contact_name(sender)
			
		if message_type == CHANNEL_TEXT_MESSAGE_TYPE_ACTION:				
			message = "%s %s" % (name, message)
			
		self.add_message(message, message_type, "other", name=name)
		self.print_timestamp = True
	
	def prepare_outgoing_message(self, message):
		message = message.strip()
		type = CHANNEL_TEXT_MESSAGE_TYPE_NORMAL
		
		if message.startswith("/me "):
			type = CHANNEL_TEXT_MESSAGE_TYPE_ACTION
			message = _("Me %s") % message[len("/me "):]
		
		return message, type
		
	def add_message(self, message, type, talking="me", right_pad=True, name=None):
		buf = self.get_buffer()
		iter = buf.get_iter_at_mark(buf.get_mark("conversation"))

		if type == None or type not in self.tags:
			tag = talking
		else:
			tag = self.tags[type]
		
		if right_pad:
			message= "%s\n" % message
		
		# Print name when switching speaker
		if type == CHANNEL_TEXT_MESSAGE_TYPE_NORMAL and self.talking != talking:
				buf.insert_with_tags_by_name(iter, "%s:\n" % name, "name-%s" % talking)
				self.talking = talking
				
		buf.insert_with_tags_by_name(iter, message, tag)
				
		self.scroll_mark_onscreen(buf.get_mark("me-start"))
				
		return True
	
	def send_message(self, message, type):
		print "Sending: '%s' / %s" % (message, type)
		self.conversation.send_message(message, type)
		self.print_timestamp = True

class ConversationUpTextView(ConversationTextView):
	def __init__(self, conversation):
		ConversationTextView.__init__(self, conversation)
	
	def get_me_start_iter(self):
		return self.get_buffer().get_end_iter()
	
	def get_me_stop_iter(self):
		return self.get_buffer().get_end_iter()
	
	def get_conversation_start_iter(self):
		return self.get_buffer().get_start_iter()
	
	def get_conversation_gravity(self):
		return False

	def get_conversation_mark_iter(self):
		iter = self.get_me_start_iter()
		iter.backward_chars(1)
		return iter

class ConversationDownTextView(ConversationTextView):
	def __init__(self, conversation):
		ConversationTextView.__init__(self, conversation)
	
	def get_me_start_iter(self):
		return self.get_buffer().get_start_iter()
	
	def get_me_stop_iter(self):
		return self.get_buffer().get_iter_at_mark(self.get_buffer().get_mark("conversation"))
	
	def get_conversation_start_iter(self):
		return self.get_buffer().get_end_iter()
	
	def get_conversation_gravity(self):
		return True
	
	def get_conversation_mark_iter(self):
		iter = self.get_me_start_iter()
		iter.forward_chars(1)
		return iter

if __name__ == "__main__":
	import os, random
	def tick(conv):
		print 'Tick'
		conv.on_message_received(None, os.popen("fortune").read().strip(), random.choice([CHANNEL_TEXT_MESSAGE_TYPE_NORMAL,CHANNEL_TEXT_MESSAGE_TYPE_NORMAL,CHANNEL_TEXT_MESSAGE_TYPE_NORMAL,CHANNEL_TEXT_MESSAGE_TYPE_NORMAL, CHANNEL_TEXT_MESSAGE_TYPE_ACTION, CHANNEL_TEXT_MESSAGE_TYPE_NOTICE]))
		return True

	v=ConversationUpTextView(None)
	w=gtk.Window()
	w.set_property("default-height", 500)
	w.set_property("default-width", 500)
	s=gtk.ScrolledWindow()
	w.add(s)
	s.add(v)
	w.show_all()
	gobject.timeout_add(2500, tick, v)
	gtk.main()
