/*
 * Copyright (c) 2006 Darryll Truchan 
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this 
 * software and associated documentation files (the "Software"), to deal in the Software 
 * without restriction, including without limitation the rights to use, copy, modify, 
 * merge, publish, distribute, sublicense, and/or sell copies of the Software, and to 
 * permit persons to whom the Software is furnished to do so, subject to the following 
 * conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies 
 * or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A 
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT 
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF 
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR 
 * THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */


#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <X11/Xatom.h>
#include <X11/extensions/Xrender.h>

#include <compiz.h>

#define BORDER_DISPLAY_OPTION_TOGGLE            0
#define BORDER_DISPLAY_OPTION_NUM               1

#define MwmHintsDecorations (1L << 1)

#define MWM_DECOR_NONE      0
#define MWM_DECOR_ALL      (1L << 0)
#define MWM_DECOR_BORDER   (1L << 1)
#define MWM_DECOR_HANDLE   (1L << 2)
#define MWM_DECOR_TITLE    (1L << 3)
#define MWM_DECOR_MENU     (1L << 4)
#define MWM_DECOR_MINIMIZE (1L << 5)
#define MWM_DECOR_MAXIMIZE (1L << 6)

#define PROP_MOTIF_WM_HINT_ELEMENTS 3

typedef struct _MwmHints {
    unsigned long flags;
    unsigned long functions;
    unsigned long decorations;
} MwmHints;

static int displayPrivateIndex;

typedef struct _BorderDisplay {
    int		    screenPrivateIndex;
    CompOption opt[BORDER_DISPLAY_OPTION_NUM];
} BorderDisplay;


typedef struct _BorderScreen {
    int windowPrivateIndex;
} BorderScreen;

typedef struct _BorderWindow {
    MwmHints mwmHints;
    Bool     hasBorder;
} BorderWindow;

#define GET_BORDER_DISPLAY(d) ((BorderDisplay *) (d)->privates[displayPrivateIndex].ptr)
#define BORDER_DISPLAY(d) BorderDisplay *bd = GET_BORDER_DISPLAY (d)
#define GET_BORDER_SCREEN(s, bd) ((BorderScreen *) (s)->privates[(bd)->screenPrivateIndex].ptr)
#define BORDER_SCREEN(s) BorderScreen *bs = GET_BORDER_SCREEN (s, GET_BORDER_DISPLAY (s->display))
#define GET_BORDER_WINDOW(w, bs) ((BorderWindow *) (w)->privates[(bs)->windowPrivateIndex].ptr)
#define BORDER_WINDOW(w) BorderWindow *bw = GET_BORDER_WINDOW  (w, GET_BORDER_SCREEN  (w->screen, GET_BORDER_DISPLAY (w->screen->display)))

#define NUM_OPTIONS(s) (sizeof ((s)->opt) / sizeof (CompOption))

static Bool
borderToggle (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompWindow *w;
    Window xid;

    xid = getIntOptionNamed (option, nOption, "window", 0);

    if (!xid)
        xid = d->activeWindow;

    w = findWindowAtDisplay (d, xid);

    if (w)
    {
        BORDER_WINDOW (w);

        if (!bw->hasBorder)
            bw->mwmHints.decorations = MWM_DECOR_ALL;
        else
            bw->mwmHints.decorations = MWM_DECOR_NONE;

        XChangeProperty (
                w->screen->display->display, 
                w->id, 
                w->screen->display->mwmHintsAtom, 
                w->screen->display->mwmHintsAtom,
                8, 
                PropModeReplace, 
                (unsigned char *) &bw->mwmHints, 
                sizeof (bw->mwmHints));

        bw->hasBorder = !bw->hasBorder;

    }
    return TRUE;
}

static CompOption * 
borderGetDisplayOptions (CompDisplay *display, int *count) 
{
    BORDER_DISPLAY (display);
    *count = NUM_OPTIONS (bd);
    return bd->opt;
}

static Bool 
borderSetDisplayOption (CompDisplay *display, char *name, CompOptionValue *value) 
{
    CompOption *o;
    int	       index;

    BORDER_DISPLAY (display);

    o = compFindOption (bd->opt, NUM_OPTIONS (bd), name, &index);
    if (!o) {
        return FALSE;
    }
    
    switch (index) 
    {
        case BORDER_DISPLAY_OPTION_TOGGLE:
            if (setDisplayAction (display, o, value))
                return TRUE;
	    break;

        default:
	    break;
    }

    return FALSE;
}

static void 
borderDisplayInitOptions (BorderDisplay *bd, Display *d) 
{
    CompOption *o;

    o = &bd->opt[BORDER_DISPLAY_OPTION_TOGGLE];
    o->name			        = "toggle";
    o->shortDesc		        = N_("Toggle Border");
    o->longDesc			        = N_("Toggle Border");
    o->type                             = CompOptionTypeAction;
    o->value.action.initiate            = borderToggle;
    o->value.action.terminate           = 0;
    o->value.action.bell                = FALSE;
    o->value.action.edgeMask            = 0;
    o->value.action.state               = CompActionStateInitKey;
//    o->value.action.state              |= CompActionStateInitButton;
    o->value.action.type                = CompBindingTypeNone;

/*    
    o->value.action.key.modifiers       = BORDER_TOGGLE_MODIFIERS_DEFAULT;
    o->value.action.key.keycode         = XKeysymToKeycode (d, XStringToKeysym (BORDER_TOGGLE_KEY_DEFAULT));
*/
}

static Bool 
borderInitDisplay (CompPlugin *p, CompDisplay *d) 
{
    BorderDisplay *bd;

    bd = malloc (sizeof (BorderDisplay));
    if (!bd) 
        return FALSE;

    bd->screenPrivateIndex = allocateScreenPrivateIndex (d);
    if (bd->screenPrivateIndex < 0) 
    {
        free (bd);
        return FALSE;
    }

    borderDisplayInitOptions (bd, d->display);

    d->privates[displayPrivateIndex].ptr = bd;

    return TRUE;
}

static void 
borderFiniDisplay (CompPlugin *p, CompDisplay *d) 
{
    BORDER_DISPLAY (d);
    freeScreenPrivateIndex (d, bd->screenPrivateIndex);
    free (bd);
}

static Bool 
borderInitScreen (CompPlugin *p, CompScreen *s) 
{
    BorderScreen *bs;

    BORDER_DISPLAY (s->display);

    bs = malloc (sizeof (BorderScreen));
    if (!bs) 
        return FALSE;

    bs->windowPrivateIndex = allocateWindowPrivateIndex (s);
    if (bs->windowPrivateIndex < 0)
    {
        free (bs);
        return FALSE;
    }

    s->privates[bd->screenPrivateIndex].ptr = bs;

    return TRUE;
}

static void 
borderFiniScreen (CompPlugin *p, CompScreen *s) 
{
    BORDER_SCREEN (s);

    free (bs);
}

static Bool
borderInitWindow (CompPlugin *p, CompWindow *w)
{
    BorderWindow *bw;

    BORDER_SCREEN (w->screen);

    bw = malloc (sizeof (BorderWindow));
    if (!bw)
        return FALSE;

    memset (&bw->mwmHints, 0, sizeof (bw->mwmHints));
    bw->mwmHints.flags = MwmHintsDecorations;
    bw->mwmHints.decorations = 1;
    bw->hasBorder = TRUE;
    
    w->privates[bs->windowPrivateIndex].ptr = bw;
    return TRUE;
}

static void
borderFiniWindow (CompPlugin *p, CompWindow *w)
{
    BORDER_WINDOW (w);
    free (bw);
}

static Bool
borderInit (CompPlugin *p) 
{
    displayPrivateIndex = allocateDisplayPrivateIndex ();
    if (displayPrivateIndex < 0) 
        return FALSE;

    return TRUE;
}

static void 
borderFini (CompPlugin *p) 
{
    if (displayPrivateIndex >= 0) 
        freeDisplayPrivateIndex (displayPrivateIndex);
}
/*
CompPluginDep borderDeps[] = {
    { CompPluginRuleAfter, "decoration" },
};
*/

static int
borderGetVersion (CompPlugin *plugin, int version)
{
    return ABIVERSION;
}

static CompPluginVTable 
borderVTable = {
    "border",
    N_("Toggle Window Border"),
    N_("Toggle window border on and off"),
    borderGetVersion,
    borderInit,
    borderFini,
    borderInitDisplay,
    borderFiniDisplay,
    borderInitScreen,
    borderFiniScreen,
    borderInitWindow,
    borderFiniWindow,
    borderGetDisplayOptions,
    borderSetDisplayOption,
    0, /* GetScreenOptions */
    0, /* SetScreenOption */
    0, /* borderDeps, */
    0, /* sizeof(borderDeps) / sizeof(borderDeps[0]) */
    0,
    0
};

CompPluginVTable * 
getCompPluginInfo(void) 
{
    return &borderVTable;
}



