/**
 * 
 * dock.c:
 * author: david mueller, 2006
 * e-mail: d.mueller@e-aktuell.at OR davidlukas.mueller@reflex.at
 * version: 0.1.6b
 * released: 05/18/2006 21:11 UTC
 * feel free to reuse anything you may find interesting.
 * this is a fork of miniwin as miniwin is intended to be a testbed for new ideas only.
 *
**/

/*

Last Changes

- bugfixes (highlighted windows, cpu overload (though it still uses too much cpu))

Changes to 0.1.6

- added: option highlight_time
- - highlight a window after highlight_time (in ms) on mouse over its mindow (default 1/2 sec.)
- - a value of 0 means: don't highlight
- added: option close_window_key - defaults to Button3
- - click on a mindow with that key and its window gets closed
- - if you don't want this, set it to the same as uniconify_key which has a higher priority
- changed: uniconify_key-click on mindows while their parent is maximized will minimize it (if it is the active window)
- bugfixes (no more (or at least a lot less) crashes, thanks to PlayerX)

known bugs
- switcher & dock don't like each other

Changes to 0.1.5

- added: mindow grouping
- - mindow with the same resName (same program) now group together
- - and do some fancy animation when they get selected (moving around and aligning automatically)
- added: option group_timeout
- - after some time of no input, an opened mindow group closes again (default 1.5 sec.)
- improved: some more animations (dock hiding/showing/scaling)
- bugfixes (mainly hover issues)

known bugs:
- still problems with switcher
- dock can't handle some apps properly (eg. gnome-screenshot, firefox) - they _may_ (not necessarily) crash compiz

Changes to 0.1.4

- added: option dock_size
- added: option hidden_dock_size
- changed: some defaults and option names
- improved: mindows now look much nicer if you use POTS textures (mipmap)
- improved: mindow resizing on hover (mindows now grow from the center)
- some bugfixes

known bugs:
- there are some problems with switcher

Changes to 0.1.3

- code rewrite
- removed: dynamic dock background
- added: some more animations (dynamic dock and mindow resize on hover)

Changes to 0.1.2

- added: option show_maximized_windows
- - this option (if you deactivate on_all_workspaces) makes dock act similar to the gnome window list applet
- added: notification will stop after a while (and restart on each mouseover)
- changed: on_all_workspaces changed its name to from_all_workspaces
- - it is now only intended to be used in show_maximized_windows mode (it didn't work in simple dock mode, anyway)
- changed: defaults configured to use show_maximized_windows
- bugfixes:
- - several dock issues
- - on_all_workspaces things
- - unexpected window/mindow destroy

known bugs:
- if you change on_all_workspaces from false to true, you will have to reload the plugin

Changes to 0.1.1

- dock changes length when a mindow gets hovered
- mindows now do a special animation to notify users when something happened in their parent windows
- much smoother animations
- bugfixes (memory management, animations, ...)
- _some_ font changing for the dock is available hard-coded

Changes from miniwin 0.1.0 alpha to dock 0.1.0

- code cleaning, bugfixes (thanks to QuinnStorm)
- sane defaults
- animation fixes
- panel shows program name on mouse over
- panel is now colored
- panel can now have another position on the screen (bottom, top, left, right)
- - attention: changing from bottom/top to left/right and vice versa may require a reload of the plugin

*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <assert.h> //don't blame me, blame the weatherman!
#include <unistd.h>

#include <X11/Xatom.h>
#include <X11/extensions/Xrender.h>

#include <compiz.h>


static char *winType[] = {
    "Splash",
    "Normal",
    "Dialog",
    "ModalDialog",
};
#define N_WIN_TYPE (sizeof (winType) / sizeof (winType[0]))

static int displayPrivateIndex;


#define GET_MINIWIN_DISPLAY(d) \
         ((MiniwinDisplay *) (d)->privates[displayPrivateIndex].ptr)
         
#define MINIWIN_DISPLAY(d) \
         MiniwinDisplay* md = GET_MINIWIN_DISPLAY(d)
         
#define GET_MINIWIN_SCREEN(s, fd) \
   ((MiniwinScreen*) (s)->privates[(fd)->screenPrivateIndex].ptr)
   
#define MINIWIN_SCREEN(s) \
   MiniwinScreen* ms = GET_MINIWIN_SCREEN(s, GET_MINIWIN_DISPLAY(s->display))
   
#define GET_MINIWIN_WINDOW(w, ss)					  \
    ((MiniwinWindow *) (w)->privates[(ss)->windowPrivateIndex].ptr)

#define MINIWIN_WINDOW(w)					       \
    MiniwinWindow *mw = GET_MINIWIN_WINDOW  (w,		       \
		      GET_MINIWIN_SCREEN  (w->screen,	       \
		      GET_MINIWIN_DISPLAY (w->screen->display)))

#define _NDEBUG 
#ifndef _NDEBUG
#define DEBUG(X) fprintf (stderr, X);
#else
#define DEBUG(X) 
#endif

#define NUM_OPTIONS(s) (sizeof ((s)->opt) / sizeof (CompOption))

#define MINIWIN_MINDOW_WIDTH_DEFAULT 64
#define MINIWIN_MINDOW_WIDTH_MIN 20
#define MINIWIN_MINDOW_WIDTH_MAX 512

#define MINIWIN_SCALE_ON_HOVER_DEFAULT 1.25f
#define MINIWIN_SCALE_ON_HOVER_MIN 0.90f
#define MINIWIN_SCALE_ON_HOVER_MAX 10.0f
#define MINIWIN_SCALE_ON_HOVER_STEP 0.01f

#define MINIWIN_OPACITY_ON_HOVER_DEFAULT 95
#define MINIWIN_OPACITY_ON_HOVER_MIN 0
#define MINIWIN_OPACITY_ON_HOVER_MAX 100

#define MINIWIN_MINDOW_OPACITY_DEFAULT 75
#define MINIWIN_MINDOW_OPACITY_MIN 0
#define MINIWIN_MINDOW_OPACITY_MAX 100

#define MINIWIN_DOCK_OPACITY_DEFAULT 60
#define MINIWIN_DOCK_OPACITY_MIN 0
#define MINIWIN_DOCK_OPACITY_MAX 100

#define MINIWIN_ANIMATION_TIME_DEFAULT 200
#define MINIWIN_ANIMATION_TIME_MIN 0
#define MINIWIN_ANIMATION_TIME_MAX 2000

#define MINIWIN_DOCK_COLOUR_DEFAULT 0xAFE0F0
#define MINIWIN_DOCK_COLOUR_MIN 0x000000
#define MINIWIN_DOCK_COLOUR_MAX 0xFFFFFF

#define MINIWIN_DOCK_FONT_DEFAULT "courier"

#define MINIWIN_DOCK_SIZE_DEFAULT MINIWIN_MINDOW_WIDTH_DEFAULT/1.5f
#define MINIWIN_DOCK_SIZE_MIN 1
#define MINIWIN_DOCK_SIZE_MAX 256

#define MINIWIN_HIDDEN_DOCK_SIZE_DEFAULT MINIWIN_MINDOW_WIDTH_DEFAULT/10.0f
#define MINIWIN_HIDDEN_DOCK_SIZE_MIN 1
#define MINIWIN_HIDDEN_DOCK_SIZE_MAX 256

#define MINIWIN_HIDE_GROUP_TIMEOUT_DEFAULT 1500
#define MINIWIN_HIDE_GROUP_TIMEOUT_MIN 100
#define MINIWIN_HIDE_GROUP_TIMEOUT_MAX 50000

#define MINIWIN_STRESS_TIMEOUT_DEFAULT 500
#define MINIWIN_STRESS_TIMEOUT_MAX 5000
#define MINIWIN_STRESS_TIMEOUT_MIN 0

#define BOTTOM 0
#define RIGHT 1
#define TOP 2
#define LEFT 3

#define MINIWIN_DOCK_SIDE_DEFAULT BOTTOM
#define MINIWIN_DOCK_SIDE_MIN 0
#define MINIWIN_DOCK_SIDE_MAX 3

#define MINIWIN_SHOW_MAXIMIZED_WINDOWS_DEFAULT TRUE
#define MINIWIN_MINDOWS_ON_ALL_WORKSPACES_DEFAULT FALSE
#define MINIWIN_DOCK_HIDE_ON_CLICK_DEFAULT TRUE
#define MINIWIN_RUNS_ON_AIGLX_DEFAULT FALSE

#define MINIWIN_TASKBAR_SPACING_DEFAULT 40
#define MINIWIN_TASKBAR_SPACING_MIN -MINIWIN_MINDOW_WIDTH_MIN
#define MINIWIN_TASKBAR_SPACING_MAX MINIWIN_MINDOW_WIDTH_MAX

#define MINIWIN_DEMINDOW_BUTTON_DEFAULT 	Button1
#define MINIWIN_DEMINDOW_MODIFIERS_DEFAULT 0

#define MINIWIN_CLOSE_WINDOW_BUTTON_DEFAULT 	Button3
#define MINIWIN_CLOSE_WINDOW_MODIFIERS_DEFAULT 0

enum
{
   //General mindow settings
   MINIWIN_SCREEN_OPTION_MINDOW_WIDTH,
   MINIWIN_SCREEN_OPTION_ANIMATION_TIME,
   MINIWIN_SCREEN_OPTION_MINDOW_OPACITY,
   MINIWIN_SCREEN_OPTION_SCALE_ON_HOVER,
   MINIWIN_SCREEN_OPTION_OPACITY_ON_HOVER,
   MINIWIN_SCREEN_OPTION_MINDOWS_ON_ALL_WORKSPACES,
   MINIWIN_SCREEN_OPTION_SHOW_MAXIMIZED_WINDOWS,
   /*MINIWIN_SCREEN_OPTION_DEMINDOW_KEY,
   MINIWIN_SCREEN_OPTION_CLOSE_WINDOW_KEY,*/
   MINIWIN_SCREEN_OPTION_HIGHLIGHT_TIMEOUT,
   //Taskbar settings
   MINIWIN_SCREEN_OPTION_TASKBAR_SPACING,
   MINIWIN_SCREEN_OPTION_DOCK_OPACITY,
   MINIWIN_SCREEN_OPTION_DOCK_COLOUR,
   MINIWIN_SCREEN_OPTION_DOCK_HIDE_ON_CLICK,
   MINIWIN_SCREEN_OPTION_DOCK_SIDE,
   MINIWIN_SCREEN_OPTION_DOCK_FONT,
   MINIWIN_SCREEN_OPTION_DOCK_SIZE,
   MINIWIN_SCREEN_OPTION_HIDDEN_DOCK_SIZE,
   MINIWIN_SCREEN_OPTION_HIDE_GROUP_TIMEOUT,
   //Internal settings
   MINIWIN_SCREEN_OPTION_WINDOW_TYPE,
   MINIWIN_SCREEN_OPTION_RUNS_ON_AIGLX,

   MINIWIN_SCREEN_OPTION_NUM
};

typedef enum _MiniwinWindowState
{
   //Normal window states
   MINIWIN_WINDOW_STATE_NORMAL_NONE,
   MINIWIN_WINDOW_STATE_NORMAL_INITIALIZING,
   MINIWIN_WINDOW_STATE_NORMAL_HIDING,
   MINIWIN_WINDOW_STATE_NORMAL_UNHIDING,
   MINIWIN_WINDOW_STATE_NORMAL_HIDDEN,
   MINIWIN_WINDOW_STATE_NORMAL_WAITING,
   //Dock window states
   MINIWIN_WINDOW_STATE_DOCK_CREATED,
   MINIWIN_WINDOW_STATE_DOCK_NONE,
   MINIWIN_WINDOW_STATE_DOCK_HIDDEN,
   //Mindow states
   MINIWIN_WINDOW_STATE_MINDOW_CREATED,
   MINIWIN_WINDOW_STATE_MINDOW_HIDDEN,
   MINIWIN_WINDOW_STATE_MINDOW_TASKBAR
} MiniwinWindowState;

#define MwmHintsDecorations (1L << 1)

typedef struct {
    unsigned long flags;
    unsigned long functions;
    unsigned long decorations;
} MwmHints;

typedef struct _MindowSlot {
   Window mindow;
   CompWindow *window;
   
   void *next;      
} MindowSlot;

typedef struct _MiniwinDisplay
{
   int screenPrivateIndex;
   HandleEventProc handleEvent;
} MiniwinDisplay;

typedef struct _MiniwinScreen
{
   int windowPrivateIndex;
   
   XTextItem *hoveredWindow; 
   char *dockFont;  
   
   //General mindow settings
   unsigned int animationTime;
   unsigned int mindowWidth;
   unsigned int mindowOpacity;
   float scaleOnHover;
   double hoverFactor;
   unsigned int opacityOnHover;
   Bool mindowsOnAllWorkspaces;
   
   //Taskbar settings
   int taskbarSpacing;
   unsigned int dockOpacity;
   unsigned int dockColour;
   unsigned int dockSide;
   unsigned int dockSize;
   unsigned int hiddenDockSize;
   
   Bool dockHideOnClick;
   Bool showMaximized;
      
   //Internal settings
   Bool runsOnAiglx;   
   unsigned int nSlotsPerRow;
   unsigned int pendingMindows;
   
   int activeGroup;
   int resetGroupTimeout;
   int groupTimeout;
   unsigned int stressTimeout;
   
   Window dock;
   MindowSlot *taskbar;
   MindowSlot *mindows;

   int nEvents;   
   Bool hover;
   Bool animation;
   
   CompOption opt[MINIWIN_SCREEN_OPTION_NUM];
   int wMask;

   DrawWindowTextureProc drawWindowTexture;
   DrawWindowGeometryProc drawWindowGeometry;
   PreparePaintScreenProc preparePaintScreen;
   DamageWindowRectProc   damageWindowRect;
   PaintScreenProc paintScreen;
   PaintWindowProc paintWindow;
   
} MiniwinScreen;

typedef struct _MiniwinWindow {
    MiniwinWindowState state;   
    
    CompWindow *parent;
    CompWindow *icon;
        
    int stressCountdown;
    Bool stressed;
    Bool hover;
    Bool urgent;
    Bool onViewport;
    
    double rotate_speed;
    double angle;
    double max_angle;
    int step;
    GLushort savedOpacity;
    
    int originalX;
    int originalY;
    
    int destinationX;
    int destinationY;
    
} MiniwinWindow;


typedef struct _ReturnHelper
{
   Window id;
   CompWindow *window;
} ReturnHelper;

//Slot functions
static MindowSlot* addMindowSlot (MindowSlot *slot);
static MindowSlot *getLastSlot (MindowSlot *first);
static int numOccupiedSlots (MindowSlot *first);
static void removeAllSlots (MindowSlot *current);
static MindowSlot *getFreeSlot (MindowSlot *first);
static MindowSlot *getSlotOf (MindowSlot *first, Window id);
static void miniwinRearrangeSlots (MindowSlot *first);

//Taskbar/dock functions
static void miniwinCreateDock (CompScreen *s);
static void miniwinInitializeDock (CompWindow *dock);
static void miniwinShowDock (CompWindow *dock);
static void miniwinHideDock (CompWindow *dock);
static void miniwinRearrangeTaskbar (CompScreen *screen);
static void miniwinCheckDock (CompWindow *dock);

//Basic functions
static Bool miniwinCreateMindow (CompWindow *from_parent);
static void miniwinInitializeMindow (CompWindow *mini);
static void miniwinChangeState (CompWindow *window, MiniwinWindowState state);
static void miniwinHide (CompWindow *window);
static void miniwinShow (CompWindow *window);
static void miniwinDestroyMindow (CompWindow *mindow);
static void miniwinCheckMindow (CompWindow *mini);

static void miniwinShowWindow (CompWindow *window);
static void miniwinHideWindow (CompWindow *window);
static void miniwinHideWindowComplete (CompWindow *window);
static void miniwinShowWindowComplete (CompWindow *window);

static void miniwinShowMindow (CompWindow *mini);
static void miniwinHideMindow (CompWindow *mini);
static void miniwinShowMindowOnViewport (CompWindow *mini);
static void miniwinHideMindowFromViewport (CompWindow *mini);

static void miniwinMaximize (CompWindow *mini);
static void miniwinMinimize (CompWindow *window);

static void miniwinHoverOn (CompWindow *mini);
static void miniwinHoverOff (CompWindow *mini);





//Implementation: taskbar functions / taskbar display functions

static MindowSlot* 
addMindowSlot (MindowSlot *slot)
{
   if (!slot) return malloc (sizeof(MindowSlot));
   
   MindowSlot *new_slot = malloc (sizeof(MindowSlot));
   new_slot->next = 0;
   new_slot->mindow = 0;
   new_slot->window = 0;
   
   while (slot->next) slot = slot->next;
   
   slot->next = new_slot;
   
   return new_slot;
}

static MindowSlot *
getLastSlot (MindowSlot *first)
{
   if (!first) return 0;
   while (first->next) first = first->next;
   return first;
}

static int
numOccupiedSlots (MindowSlot *first)
{
   if (!first) return 0;
   int count = 0;
   do
   {
      if (first->mindow) count++;
   } while (first = first->next);
   return count;
}

static void
removeAllSlots (MindowSlot *current)
{
   if (!current) return;
   MindowSlot *next;
   do
   {
      next = current->next;
      free (current);            
   } while (current = next);
}

static MindowSlot *
getFreeSlot (MindowSlot *first)
{
   if (!first) return malloc (sizeof(MindowSlot));
   MindowSlot *current = first;
   do
   {
      if (current->mindow == 0) return current;
   } while (current = current->next);
   return addMindowSlot (getLastSlot(first));
}

static MindowSlot *
getSlotOf (MindowSlot *first, Window id)
{
   do
   {
      if (first->mindow == id) return first;
   } while (first = first->next);
   return 0;
}

static void
miniwinRaiseMindows (MindowSlot *first, CompDisplay *d)
{
   if (first->window)
   {
        MINIWIN_SCREEN(first->window->screen);
        if (ms->dock)
        {
          lowerWindow (findWindowAtDisplay(d, ms->dock));
        }
   }
   do
   {
      if (first->mindow)
         raiseWindow (findWindowAtDisplay(d, first->mindow));
   } while (first = first->next);
}

static Bool
miniwinRemoveDeadMindows (MindowSlot *current, Screen *screen)
{
   Bool pending = TRUE;//FALSE;
/*   do {
      if (current->mindow)
      {
         CompWindow *mindow = findWindowAtDisplay (screen->display, current->mindow);
         if (!mindow)
         {
            DEBUG ("Found a dead mindow!\n");
            current->mindow = 0;
            pending = TRUE;
         }
      }
      else pending = TRUE;
   } while (current = current->next);*/
   return pending;
}

static int
miniwinDockTextOffset (CompWindow *dock)
{
   MINIWIN_SCREEN (dock->screen);
   Display *dpy = dock->screen->display->display;
   
   if (!ms->hoveredWindow) return 0;
   
   XGCValues val;
   int num_fonts;
         
   char *font_search = malloc (1024);
   strcpy (font_search, "*");
   strcat (font_search, ms->dockFont);
   strcat (font_search, "-medium-r-normal--14*");
   char **fonts = XListFonts (dock->screen->display->display, font_search, 1024, &num_fonts);
   free (font_search);

   Font font = XLoadFont (dock->screen->display->display, fonts[0]);
   if (!font) DEBUG ("error loading font\n");       
   GC gc = XCreateGC (dock->screen->display->display, dock->id, 0, &val);  
   XSetFont (dock->screen->display->display, gc, font);
   
   //calculate the font's height
   XFontStruct *fontinfo = XQueryFont (dpy, font);
   int d, fa, fd;
   XCharStruct r;
   XTextExtents (fontinfo, ms->hoveredWindow->chars, ms->hoveredWindow->nchars, &d, &fa, &fd, &r);
   if (ms->dockSide == TOP) return 2 + (fa-fd) * ms->scaleOnHover;
   else if (ms->dockSide == TOP) return - ( 2 + (fa-fd) * ms->scaleOnHover );
   return 0;
}

static void
miniwinWriteOnDock (CompWindow *dock, XTextItem *textItem)
{
   CompScreen *screen = dock->screen;
   Display *dpy = dock->screen->display->display;
   MINIWIN_SCREEN (screen);
   
   XGCValues val;
   int num_fonts;
         
   char *font_search = malloc (1024);
   strcpy (font_search, "*");
   strcat (font_search, ms->dockFont);
   strcat (font_search, "-medium-r-normal--14*");
   char **fonts = XListFonts (dock->screen->display->display, font_search, 1024, &num_fonts);
   free (font_search);
         
   if (!fonts) fprintf (stderr, "no fonts??\n");
         
   Font font = XLoadFont (dock->screen->display->display, fonts[0]);
   if (!font) DEBUG ("error loading font\n");       
   GC gc = XCreateGC (dock->screen->display->display, dock->id, 0, &val);  
   XSetFont (dock->screen->display->display, gc, font);
         
   XTextItem text[3];
   text[1] = *textItem;
   text[0].chars = "::";
   text[0].nchars = 2;
   text[0].font = None;
   text[0].delta = 5;
   text[2] = text[0];
         
   if (ms->dockSide == BOTTOM)
      XDrawText (dock->screen->display->display, dock->id, gc, 5, dock->height - 5, text, 3);
   else if (ms->dockSide == TOP || ms->dockSide == LEFT || ms->dockSide == RIGHT)
      XDrawText (dock->screen->display->display, dock->id, gc, 5, 15, text, 3);
}

static void
miniwinMoveWindow (CompWindow *w, int x, int y)
{
   MINIWIN_WINDOW (w);
   mw->destinationX = x;
   mw->destinationY = y;
   addWindowDamage (w);
}

static void
miniwinClearDock (CompWindow *dock)
{
   MINIWIN_SCREEN (dock->screen);
   XGCValues val;   
   GC gc = XCreateGC (dock->screen->display->display, dock->id, 0, &val); 
   XSetForeground (dock->screen->display->display,gc, ms->dockColour);
   XFillRectangle (dock->screen->display->display, dock->id, gc, 0, 0, dock->width, dock->height);
}

static void
miniwinGrowDock (CompWindow *dock, int slots, int rows)
{
   CompScreen *screen = dock->screen;
   MINIWIN_SCREEN (screen);
   
   int dockLength = 0;
   if (rows == 1)
      dockLength += (ms->mindowWidth * 2 + ms->taskbarSpacing) * slots + ms->taskbarSpacing;
   else
      dockLength += ms->mindowWidth * 2 * ms->nSlotsPerRow;   
         
   if (ms->hover) dockLength += (ms->hoverFactor - 1.0) * 2.0 * ms->taskbarSpacing;
      
   MINIWIN_WINDOW (dock);
      
   if (ms->dockSide == TOP || ms->dockSide == BOTTOM)
      XResizeWindow (screen->display->display, dock->id, dockLength, ms->dockSize * 2.0f);
   else if (ms->dockSide == LEFT || ms->dockSide == RIGHT)
      XResizeWindow (screen->display->display, dock->id, ms->dockSize * 2.0f, dockLength);
      XFlush (screen->display->display);
      
   Display *dpy = dock->screen->display->display;
      
   if (slots == 0) //move the dock out of the screen
   {
      //XMoveWindow (dpy, dock->id, dock->serverX, screen->height);   
      miniwinMoveWindow (dock, dock->serverX, screen->height);
   }
   else if (mw->state == MINIWIN_WINDOW_STATE_DOCK_HIDDEN && ms->dockHideOnClick)
   {
      if (ms->dockSide == BOTTOM)
         //XMoveWindow (dpy, dock->id, (screen->width - dockLength) / 2, screen->height - ms->hiddenDockSize);   
         miniwinMoveWindow (dock, (screen->width - dockLength) / 2, screen->height - ms->hiddenDockSize);
      else if (ms->dockSide == TOP)
         //XMoveWindow (dpy, dock->id, (screen->width - dockLength) / 2, 0 - (dock->height - ms->hiddenDockSize));
         miniwinMoveWindow (dock, (screen->width - dockLength) / 2, 0 - (dock->height - ms->hiddenDockSize));
      else if (ms->dockSide == LEFT)
         //XMoveWindow (dpy, dock->id, 0 - (dock->width - ms->hiddenDockSize), (screen->height - dockLength) / 2);
         miniwinMoveWindow (dock, 0 - (dock->width - ms->hiddenDockSize), (screen->height - dockLength) / 2);
      else if (ms->dockSide == RIGHT)
         //XMoveWindow (dpy, dock->id, screen->width - ms->hiddenDockSize, (screen->height - dockLength) / 2);
         miniwinMoveWindow (dock, screen->width - ms->hiddenDockSize, (screen->height - dockLength) / 2);
   }
   else
   {
      int x, y;
      switch (ms->dockSide)
      {
      case TOP:
         x = (screen->width - dockLength) / 2;
         y = 0;
         break;
      case LEFT:
         x = 0;
         y = (screen->height - dockLength) / 2;
         break;
      case BOTTOM:
         x = (screen->width - dockLength) / 2;
         y = screen->height - dock->height;
         break;
      case RIGHT:
         x = screen->width - dock->width;
         y = (screen->height - dockLength) / 2;
      }
      //XMoveWindow (dpy, dock->id, x, y);
      miniwinMoveWindow (dock, x, y);
   }
   
   if (ms->hoveredWindow) miniwinWriteOnDock (dock, ms->hoveredWindow);  
   else miniwinClearDock (dock);   
}

static int
miniwinCountMindowGroups (MindowSlot *current)
{
   if (!current) return 0;

   int mindows = numOccupiedSlots (current);
   char ** progs = malloc (sizeof(char*) * mindows);
   
   int i;
   int count = 0;
   for (i = 0; i < mindows; ++i, current = current->next)
   {
      Bool found = FALSE;
      int j;
      if (!current->window)
         continue;
      for (j = 0; j < count; ++j)
      {
         if (strcmp(progs[j], current->window->resName) == 0)
         {
            found = TRUE;
            break;
         }
      }
      if (!found)
      {
         progs[count++] = current->window->resName;
      }
   }
   
   free (progs);
   
   return count;
}

static int
miniwinNumberOfSlots (CompScreen *screen)
{
   MINIWIN_SCREEN (screen);
   //return numOccupiedSlots (ms->taskbar);
   return miniwinCountMindowGroups (ms->taskbar);
}

static int
miniwinNumberOfRows (CompScreen *screen)
{
   MINIWIN_SCREEN (screen);
   return miniwinNumberOfSlots (screen) / ms->nSlotsPerRow + 1;
}

static int //returns the number of mindows in group number num
miniwinGetMindowGroup (MindowSlot *current, int num, MindowSlot ***slots)
{ 
   int mindows = numOccupiedSlots (current);
   char ** progs = malloc (sizeof(char*) * mindows);
   MindowSlot *begin = current;
   
   int i;
   int count = 0;
   for (i = 0; i < mindows; ++i, current = current->next)
   {
      Bool found = FALSE;
      int j;
      for (j = 0; j < count; ++j)
      {
         if (strcmp(progs[j], current->window->resName) == 0)
         {
            found = TRUE;
            break;
         }
      }
      if (!found)
      {
         progs[count++] = current->window->resName;
      }
   }
   
   current = begin;
   count = 0;
   for (i = 0; i < mindows; ++i, current = current->next)
   {
      if (strcmp(progs[num], current->window->resName) == 0)
         count ++;
   }
   
   *slots = malloc (sizeof(MindowSlot*) * count);
   current = begin;
   count = 0;
   for (i = 0; i < mindows; ++i, current = current->next)
   {
      if (strcmp(progs[num], current->window->resName) == 0)
      {
         (*slots)[count++] = current;
      }
   }
   
   free (progs);
   return count;
}

static int
miniwinGetMindowGroupOf (CompWindow *mindow, MindowSlot *current)
{
   int mindows = numOccupiedSlots (current);
   char ** progs = malloc (sizeof(char*) * mindows);
   MindowSlot *begin = current;
   
   int i;
   int count = 0;
   for (i = 0; i < mindows; ++i, current = current->next)
   {
      Bool found = FALSE;
      int j;
      for (j = 0; j < count; ++j)
      {
         if (strcmp(progs[j], current->window->resName) == 0)
         {
            found = TRUE;
            break;
         }
      }
      if (!found)
      {
         progs[count++] = current->window->resName;
      }
   }
   
   current = begin;
   for (i = 0; i < count; ++i, current = current->next)
   {
      if (strcmp(progs[i], mindow->resName) == 0)
      {
         free (progs);
         return i;
      }
   }
      
   free (progs);
   return -1;
}

static void
miniwinRearrangeTaskbar (CompScreen *screen)
{
   MINIWIN_SCREEN (screen);
   
   MindowSlot *current = ms->taskbar;
   CompWindow *dock = 0;
   if (miniwinRemoveDeadMindows (current, screen))
      miniwinRearrangeSlots (current);
  
   int hiddenOffset = 0;   
   int textOffset = 0;
   
   if (ms->dock)
   {
      dock = findWindowAtScreen(screen, ms->dock);
      MINIWIN_WINDOW (dock);
      if (mw->state == MINIWIN_WINDOW_STATE_DOCK_HIDDEN && ms->dockHideOnClick)
      {
         hiddenOffset = 1024;
      }     
      
      miniwinGrowDock (dock, miniwinNumberOfSlots(screen), miniwinNumberOfRows(screen));   
      miniwinCheckDock (dock);
      textOffset = miniwinDockTextOffset (dock);
      
      lowerWindow (dock);
   }


   int index;
            
   int offset = ms->taskbarSpacing / 2;
   int nMinis = miniwinNumberOfSlots (screen);
                  
   int count = 0;
   int row = 1;
   int nRows = miniwinNumberOfRows (screen);
   
   int x, y;
   int group;
   int groups = miniwinCountMindowGroups (current);
      
   //for (index = 0; current; current = current->next, ++index)
   for (group = 0; group < groups; ++group)
   {

      MindowSlot ** slots;
      int num_in_group = miniwinGetMindowGroup (current, group, &slots);
      int mini;
      for (mini = 0; mini < num_in_group; ++mini)
      {
         CompWindow *mindow = findWindowAtDisplay (screen->display, slots[mini]->mindow);
         
         double alpha = (2 * 3.14159 / num_in_group) * mini;
         int r = (group == ms->activeGroup) ? sqrt(num_in_group * ms->mindowWidth * ms->mindowWidth / 2) : 10;
         
         if (ms->dockSide == BOTTOM || ms->dockSide == TOP)
         {
            if (ms->dockSide == BOTTOM)
            {
               y = screen->height - row * mindow->height - (row%2) * offset + hiddenOffset;
               y = y - r * (1 - cos(alpha));
               if (alpha && (group == ms->activeGroup)) y -= ms->mindowWidth / 2;
            }
            else if (ms->dockSide == TOP)
            {
               y = (row - 1) * mindow->height + (row%2) * offset * 1.5f - hiddenOffset;
               y = y + r * (1 - cos(alpha));
               if (alpha && (group == ms->activeGroup)) y += ms->mindowWidth / 2;
            }
   
            int nMindowsInRow = row == nRows ? nMinis % ms->nSlotsPerRow : ms->nSlotsPerRow;
            int rowBegin = ( screen->width - (nMindowsInRow)*(ms->mindowWidth*2 + ms->taskbarSpacing) ) / 2 + offset;
            x = rowBegin + count * ( ms->mindowWidth * 2 + ms->taskbarSpacing ) 
                 + ((row-1)%2) * (ms->mindowWidth - ms->taskbarSpacing);
            x = x - r * sin(alpha);
         }      
         if (ms->dockSide == LEFT || ms->dockSide == RIGHT)
         {
            if (ms->dockSide == RIGHT)
            {
               x = screen->width - row * mindow->width - (row%2) * offset + hiddenOffset;
               x = x - r * (1 - cos(alpha));
               if (alpha && (group == ms->activeGroup)) x -= ms->mindowWidth;
            }
            else if (ms->dockSide == LEFT)
            {
               x = (row - 1) * mindow->width + (row%2) * offset * 1.5f - hiddenOffset;
               x = x + r * (1 - cos(alpha));
               if (alpha && (group == ms->activeGroup)) x += ms->mindowWidth;
            }
            
            int nMindowsInRow = row == nRows ? nMinis % ms->nSlotsPerRow : ms->nSlotsPerRow;
            int rowBegin = ( screen->height - (nMindowsInRow)*(ms->mindowWidth*2 + ms->taskbarSpacing) ) / 2 + offset;
            y = rowBegin + group * (ms->mindowWidth*2 + ms->taskbarSpacing ) 
                 + ((row-1)%2) * (ms->mindowWidth - ms->taskbarSpacing);
            y = y - r * sin(alpha);
         }
         
         miniwinMoveWindow (mindow, x, y+textOffset);
      }
      free (slots);
   /*   if (!current->mindow) continue;
      CompWindow *mindow = findWindowAtDisplay (screen->display, current->mindow);
      assert (mindow);
            
      MINIWIN_WINDOW (mindow);

      int y = 0;      
      int x = 0;
      
      if (ms->dockSide == BOTTOM || ms->dockSide == TOP)
      {
         if (ms->dockSide == BOTTOM)
            y = screen->height - row * mindow->height - (row%2) * offset + hiddenOffset;
         else if (ms->dockSide == TOP)
            y = (row - 1) * mindow->height + (row%2) * offset * 1.5f - hiddenOffset;

         int nMindowsInRow = row == nRows ? nMinis % ms->nSlotsPerRow : ms->nSlotsPerRow;
         int rowBegin = ( screen->width - (nMindowsInRow)*(ms->mindowWidth*2 + ms->taskbarSpacing) ) / 2 + offset;
         x = rowBegin + count * ( ms->mindowWidth * 2 + ms->taskbarSpacing ) 
              + ((row-1)%2) * (ms->mindowWidth - ms->taskbarSpacing);
      }
      if (ms->dockSide == LEFT || ms->dockSide == RIGHT)
      {
         if (ms->dockSide == RIGHT)
            x = screen->width - row * mindow->width - (row%2) * offset + hiddenOffset;
         else if (ms->dockSide == LEFT)
            x = (row - 1) * mindow->width + (row%2) * offset * 1.5f - hiddenOffset;
         
         int nMindowsInRow = row == nRows ? nMinis % ms->nSlotsPerRow : ms->nSlotsPerRow;
         int rowBegin = ( screen->height - (nMindowsInRow)*(ms->mindowWidth*2 + ms->taskbarSpacing) ) / 2 + offset;
         y = rowBegin + count * (ms->mindowWidth*2 + ms->taskbarSpacing ) 
              + ((row-1)%2) * (ms->mindowWidth - ms->taskbarSpacing);
      }*/
      
      //XMoveWindow (screen->display->display, current->mindow, x, y+textOffset);
      
      if (++count == ms->nSlotsPerRow)
      {
         count = 0;
         row ++;
      } 
   }    
      
   //miniwinRaiseMindows (ms->taskbar, screen->display);
}

static void
miniwinRearrangeSlots (MindowSlot *first)
{
   do
   {
      if (!first->mindow && first->next)
      {
         first->mindow = ((MindowSlot*)first->next)->mindow;
         first->window = ((MindowSlot*)first->next)->window;
         ((MindowSlot*)first->next)->mindow = 0;
         ((MindowSlot*)first->next)->window = 0;
      }
   } while (first = first->next);
}


static void
miniwinCheckDock (CompWindow *dock)
{        
   CompScreen *screen = dock->screen;
   MINIWIN_SCREEN (screen);
   
   Display *dpy = dock->screen->display->display;
      
   XSelectInput (dpy, dock->id, EnterWindowMask | LeaveWindowMask | KeymapStateMask | ButtonPressMask);
   XFlush (dpy);
   
   dock->state = CompWindowStateHiddenMask;
   dock->state |= CompWindowStateAboveMask;
   dock->state |= CompWindowStateStickyMask;
   dock->paint.opacity = (OPAQUE * ms->dockOpacity) / 100.0f;
   dock->type = CompWindowTypeSplashMask;  
}


static Visual *
findArgbVisual (Display *dpy, int scr)
{
    XVisualInfo		*xvi;
    XVisualInfo		template;
    int			nvi;
    int			i;
    XRenderPictFormat	*format;
    Visual		*visual;

    template.screen = scr;
    template.depth  = 32;
    template.class  = TrueColor;

    xvi = XGetVisualInfo (dpy,
			  VisualScreenMask |
			  VisualDepthMask  |
			  VisualClassMask,
			  &template,
			  &nvi);
    if (!xvi)
	return 0;

    visual = 0;
    for (i = 0; i < nvi; i++)
    {
	format = XRenderFindVisualFormat (dpy, xvi[i].visual);
	if (format->type == PictTypeDirect && format->direct.alphaMask)
	{
	    visual = xvi[i].visual;
	    break;
	}
    }

    XFree (xvi);

    return visual;
}

static void
miniwinCreateDock (CompScreen *s)
{
   MINIWIN_SCREEN (s);
   
   Atom mwmHintsAtom;
	MwmHints mwmHints;
	 
   Display *dpy = XOpenDisplay(DisplayString(s->display->display));

   int width = ms->dockSize * 2.0f;                     //multiply the desired size with two. don't know why, but other-
   int height = ms->dockSize * 2.0f; //wise the icons get too small, which one wouldn't expect.   
   
	Atom state[4];
	int nState = 0;
	XWMHints xwmh;

	xwmh.flags = InputHint;
	xwmh.input = 0;
			 
   Visual *visual = findArgbVisual (dpy, s->screenNum);
	if (!visual)
	  return;
	  
   XSetWindowAttributes attr;
	attr.background_pixel = ms->dockColour;
	attr.border_pixel     = 0xFF000000;
	attr.colormap	      = XCreateColormap (dpy, s->root, visual, AllocNone);    
	    
	    
   Window win = XCreateWindow (dpy, s->root, 0, 0, width, height, 1, 32, InputOutput, visual,
                               CWBackPixel | CWBorderPixel | CWColormap, &attr);
            
	mwmHintsAtom = XInternAtom (dpy, "_MOTIF_WM_HINTS", 0);

	memset (&mwmHints, 0, sizeof (mwmHints));

	mwmHints.flags = MwmHintsDecorations;
	mwmHints.decorations = 0;

	XChangeProperty (dpy, win, mwmHintsAtom, mwmHintsAtom,
			 8, PropModeReplace, (unsigned char *) &mwmHints,
			 sizeof (mwmHints));

	state[nState++] = s->display->winStateAboveAtom;
	state[nState++] = s->display->winStateStickyAtom;
	state[nState++] = s->display->winStateSkipTaskbarAtom;
	state[nState++] = s->display->winStateSkipPagerAtom;

	XChangeProperty (dpy, win,
			 XInternAtom (dpy, "_NET_WM_STATE", 0),
			 XA_ATOM, 32, PropModeReplace,
			 (unsigned char *) state, nState);
        
   Atom type = XInternAtom (dpy, "_NET_WM_WINDOW_TYPE_SPLASH", 0);
   XChangeProperty (dpy, win,
                    XInternAtom (dpy, "_NET_WM_WINDOW_TYPE", 0),
                    XA_ATOM, 32, PropModeReplace,
                    (unsigned char *) &type, 1);
                    
   XMapWindow (s->display->display, win);
   XSelectInput (dpy, win, StructureNotifyMask);
   
   ms->dock = win;
   
}

static void
miniwinInitializeDock (CompWindow *dock)
{
   if (!dock) return;
   CompScreen *screen = dock->screen;
   MINIWIN_SCREEN (screen);   
   MINIWIN_WINDOW (dock);   
   mw->state = MINIWIN_WINDOW_STATE_DOCK_NONE;

   miniwinChangeState (dock, MINIWIN_WINDOW_STATE_DOCK_NONE);
   miniwinRearrangeTaskbar (dock->screen);
   miniwinRaiseMindows (ms->taskbar, dock->screen->display);
   lowerWindow (dock);
}


static Bool 
miniwinCreateMindow (CompWindow *from_parent)
{  
   CompWindow *orig = from_parent;
   CompScreen *screen = orig->screen;

   MINIWIN_SCREEN (screen);
   MINIWIN_WINDOW (from_parent);
   
   Atom type;   
 	Atom		     mwmHintsAtom;
	MwmHints	     mwmHints;
   
   Display *dpy = XOpenDisplay(DisplayString(screen->display->display));
   if (!dpy) return FALSE;
  
   int whiteColor = WhitePixel(dpy, DefaultScreen(dpy));

   int width = 0, height = 0;
   
   if (ms->dockSide == BOTTOM || ms->dockSide == TOP)
   {
      width = ms->mindowWidth * 2;                     //multiply the desired size with two. don't know why, but other-
      height = (width * orig->height) / orig->width; //wise the icons get too small, which one wouldn't expect.

      //don't let the mindow get too long.
      if (height > ms->mindowWidth * 2)
      {
         height = ms->mindowWidth * 2;
         width = (height * orig->width) / orig->height;
      }
   }
   else
   {
      height = ms->mindowWidth * 2;
      width = (height * orig->width) / orig->height;
      
      if (width > ms->mindowWidth * 2)
      {
         width = ms->mindowWidth * 2;
         height = (width * orig->height) / orig->width;
      }
   }
  
   DEBUG ("Trying to create a simple mindow\n"); 
   Window win = XCreateSimpleWindow(dpy, DefaultRootWindow(dpy), 0, 0,
                                    width, height, 1, whiteColor, 0xFF000000);

   
   mwmHintsAtom = XInternAtom (dpy, "_MOTIF_WM_HINTS", 0);
	memset (&mwmHints, 0, sizeof (mwmHints));  
   mwmHints.flags	= MwmHintsDecorations;
	mwmHints.decorations = 0;

	XChangeProperty (dpy, win, mwmHintsAtom, mwmHintsAtom,
			 8, PropModeReplace, (unsigned char *) &mwmHints,
			 sizeof (mwmHints));

   type = XInternAtom (dpy, "_NET_WM_WINDOW_TYPE_SPLASH", 0);
   XChangeProperty (dpy, win, XInternAtom (dpy, "_NET_WM_WINDOW_TYPE", 0),
                    XA_ATOM, 32, PropModeReplace, (unsigned char *) &type, 1);
                      
   XMapWindow (screen->display->display, win);
   XSelectInput (dpy, win, StructureNotifyMask | PointerMotionMask);
 
   mw->state = MINIWIN_WINDOW_STATE_NORMAL_WAITING;
   mw->originalX = orig->serverX;
   mw->originalY = orig->serverY;
 
   miniwinRearrangeSlots (ms->mindows);
   MindowSlot *new_mindow = getFreeSlot(ms->mindows);
   new_mindow->mindow = win;
   new_mindow->window = from_parent;
   
   return TRUE;
}

static void
miniwinInitializeMindow (CompWindow *mini)
{
   CompScreen *screen = mini->screen;
   
   Atom mwmHintsAtom = XInternAtom (screen->display->display, "_MOTIF_WM_HINTS", 0);
         
   MwmHints mwmHints;
   memset (&mwmHints, 0, sizeof (mwmHints));  
   mwmHints.flags	= MwmHintsDecorations;
   mwmHints.decorations = 0;

   XChangeProperty (screen->display->display, mini->id, mwmHintsAtom, mwmHintsAtom,
                    8, PropModeReplace, (unsigned char *) &mwmHints, sizeof (mwmHints));
      	                 
   Atom type = XInternAtom (screen->display->display, "_NET_WM_WINDOW_TYPE_SPLASH", 0);
   XChangeProperty (screen->display->display, mini->id, 
                    XInternAtom (screen->display->display, "_NET_WM_WINDOW_TYPE", 0),
                    XA_ATOM, 32, PropModeReplace, (unsigned char *) &type, 1);
                          
   XSelectInput (screen->display->display, mini->id, 
                 EnterWindowMask | LeaveWindowMask | FocusChangeMask | StructureNotifyMask
                 | VisibilityChangeMask | PointerMotionMask);
   
   
   MINIWIN_WINDOW (mini);   
   MINIWIN_SCREEN (mini->screen);
   
   mw->originalX = -1;
   mw->originalY = -1;
   
   miniwinHide (mini);   
   mw->originalX = -1;
   mw->originalY = -1;
      
   
   if (ms->showMaximized)
   {
      MindowSlot *slot = getFreeSlot (ms->taskbar);
      slot->mindow = mini->id;
      slot->window = mw->parent;
      miniwinRearrangeTaskbar (mini->screen);
      miniwinChangeState (mini, MINIWIN_WINDOW_STATE_MINDOW_TASKBAR);
   }
   miniwinCheckMindow (mini);
   
   disableTexture (mini->screen, &mini->texture);
   glXDestroyGLXPixmap (mini->screen->display->display, mini->texture.pixmap);
	mini->texture.pixmap = None;
}

static void
miniwinChangeState (CompWindow *window, MiniwinWindowState state)
{
   MINIWIN_WINDOW (window);
   mw->state = state;
}

static Bool
miniwinCheckState (CompWindow *window, MiniwinWindowState state)
{
   MINIWIN_WINDOW (window);
   return mw->state == state;
}

static void
miniwinHide (CompWindow *window)
{
   MINIWIN_WINDOW (window);
   XMoveWindow (window->screen->display->display, window->id, window->screen->width/2, window->screen->height*4);   
   addWindowDamage (window);
   mw->hover = FALSE;
   MINIWIN_SCREEN (window->screen);
   ms->hover = FALSE;
}

static void
miniwinShow (CompWindow *window)
{
   MINIWIN_WINDOW (window);
   XMoveWindow (window->screen->display->display, window->id, mw->originalX, mw->originalY);   
   raiseWindow (window);
   addWindowDamage (window);
   updateWindowAttributes (window,TRUE);   
   moveInputFocusToWindow (window);
}

static void
miniwinDestroyMindow (CompWindow *mindow)
{
   MINIWIN_SCREEN (mindow->screen);
   
   DEBUG ("destroying a mindow!\n");

   MindowSlot *slot = getSlotOf (ms->taskbar, mindow->id);
   
   if (!slot || !mindow || slot->mindow != mindow->id)
      return;
      
   slot->mindow = 0;
   miniwinRearrangeTaskbar (mindow->screen);
   XDestroyWindow (mindow->screen->display->display, mindow->id);
}

static void
miniwinCheckMindow (CompWindow *mini)
{
   MINIWIN_SCREEN (mini->screen);
   MINIWIN_WINDOW (mini);
   mini->state = CompWindowStateHiddenMask; //TODO: find another way - this hacks around the trailfocus plugin.
   mini->state |= CompWindowStateAboveMask;
   
   //if (ms->mindowsOnAllWorkspaces)  //TODO URGENT: this is expected to work for _ALL_ mindows, currently it only works
   //for mindows with a maximized window
      mini->state |= CompWindowStateStickyMask;
      
   mini->paint.saturation = OPAQUE;
   mini->paint.brightness = OPAQUE;

   if (mw->hover)
      mini->paint.opacity = (OPAQUE * ms->opacityOnHover) / 100.0f;
   else
      mini->paint.opacity = (OPAQUE * ms->mindowOpacity) / 100.0f;
      
  CompWindow *parent = mw->parent;
         
   int width = ms->mindowWidth * 2;                  
   int height = (width * parent->height) / parent->width;
   
   if (height > ms->mindowWidth * 2)
   {
      height = ms->mindowWidth * 2;
      width = (height * parent->width) / parent->height;
   }
   XResizeWindow (mini->screen->display->display, mini->id, width, height); 
}

static void
miniwinHideDock (CompWindow *dock)
{
   miniwinChangeState (dock, MINIWIN_WINDOW_STATE_DOCK_HIDDEN);
   miniwinRearrangeTaskbar (dock->screen);
}

static void
miniwinShowDock (CompWindow *dock)
{
   MINIWIN_SCREEN (dock->screen);
   miniwinChangeState (dock, MINIWIN_WINDOW_STATE_DOCK_NONE);
   miniwinRearrangeTaskbar (dock->screen);
}

static void
miniwinMaximize (CompWindow *mini)
{
   MINIWIN_SCREEN (mini->screen);
   MINIWIN_WINDOW (mini);        
             
   MindowSlot *slot = getSlotOf (ms->taskbar, mini->id);
   if (slot)
   {
      if (!ms->showMaximized)
      {
         slot->mindow = 0;
         slot->window = 0;
      }

      if (ms->showMaximized)
      {
         mw->originalX = mini->serverX;
         mw->originalY = mini->serverY;
      }
      else
      {
         miniwinHideMindow (mini);        
         miniwinHoverOff (mini);
      }
         
      miniwinShowWindow (mw->parent);
      miniwinRearrangeTaskbar (mini->screen);
   }
}

static void
miniwinShowWindow (CompWindow *window)
{
   MINIWIN_WINDOW (window);
   if (miniwinCheckState (window, MINIWIN_WINDOW_STATE_NORMAL_NONE))
   {
      mw->originalX = window->serverX;
      mw->originalY = window->serverY;
      miniwinShowWindowComplete (window);
   }
   else
   {
      miniwinChangeState (window, MINIWIN_WINDOW_STATE_NORMAL_UNHIDING);
      addWindowDamage (window);
      mw->step = 100;
   }
   raiseWindow (window); 
   moveInputFocusToWindow (window);
}

static void
miniwinShowMindow (CompWindow *mini)
{
   miniwinCheckMindow (mini);
   miniwinShow (mini);
   raiseWindow (mini);
}

static void
miniwinHideWindow (CompWindow *window)
{
   miniwinChangeState (window, MINIWIN_WINDOW_STATE_NORMAL_HIDING);
   
   if (window->inShowDesktopMode)
      leaveShowDesktopMode(window->screen, window);
   
   MINIWIN_WINDOW (window);
   mw->step = 100;
   addWindowDamage (window);
   mw->originalX = window->serverX;
   mw->originalY = window->serverY;
}

static void
miniwinHideMindow (CompWindow *mini)
{
   MINIWIN_SCREEN (mini->screen);
   if (!ms->showMaximized) //don't ever hide mindows
   {
      miniwinChangeState (mini, MINIWIN_WINDOW_STATE_MINDOW_HIDDEN);
      miniwinHide (mini);

      MINIWIN_WINDOW (mini);
      mw->originalX = mini->serverX;
      mw->originalY = mini->serverY;
   }
}

static void
miniwinUnstressWindow (CompWindow *window);

static void
miniwinMinimize (CompWindow *window)
{
   MINIWIN_SCREEN (window->screen);
   MINIWIN_WINDOW (window);

   if (mw->stressed)
      miniwinUnstressWindow (window);
                  
   if (!ms->showMaximized)
   {
      MindowSlot *slot = getFreeSlot (ms->taskbar);
      slot->mindow = mw->icon->id;
      slot->window = window;
   }
         
   miniwinCheckMindow (mw->icon);
   miniwinHideWindow (window);
   miniwinRearrangeTaskbar (window->screen);

   if (!ms->showMaximized) miniwinChangeState (mw->icon, MINIWIN_WINDOW_STATE_MINDOW_TASKBAR);  
}

static void
miniwinStressHelper (CompWindow *window, void *closure)
{
   if (window->destroyed) return;
   MINIWIN_WINDOW (window);
   MINIWIN_SCREEN (window->screen);
   if (mw && !mw->parent && window->id != ms->dock)
   {
      if (!mw->stressed)
      {
         mw->savedOpacity = window->paint.opacity;
         window->paint.opacity = OPAQUE * 0.1;
      }
      else
      {
      }
   }
}

static void
miniwinUnstressHelper (CompWindow *window, void *closure)
{
   if (window->destroyed) return;
   MINIWIN_WINDOW (window);
   MINIWIN_SCREEN (window->screen);
   if (mw && !mw->parent && window->id != ms->dock)
      window->paint.opacity = mw->savedOpacity;
}

static void
miniwinStressWindow (CompWindow *window)
{
   MINIWIN_WINDOW (window);
   MINIWIN_SCREEN (window->screen);
   if (!mw) return;
   mw->stressed = TRUE;
   mw->stressCountdown = ms->stressTimeout;
   
   damageScreen (window->screen);
}

static void
miniwinUnstressWindow (CompWindow *window)
{
   forEachWindowOnScreen (window->screen, miniwinUnstressHelper, 0);
   
   if (!window) return;

   MINIWIN_WINDOW (window);
   if (mw)
   {
      mw->stressed = FALSE;
      mw->stressCountdown = 0;
   }
   
   damageScreen (window->screen);
}

static void
miniwinHoverOn (CompWindow *mini)
{
   MINIWIN_SCREEN (mini->screen);
   MINIWIN_WINDOW (mini);
   raiseWindow (mini);

   ms->resetGroupTimeout = ms->groupTimeout;         
   {
      CompWindow *parent = mw->parent;
      MINIWIN_WINDOW (parent);
      if (mw->state == MINIWIN_WINDOW_STATE_NORMAL_NONE)
         miniwinStressWindow (parent);
   }
   if (!mw->hover)
   {         
      if (mw->urgent) mw->max_angle = 20.0f;
      
      mw->originalX = mini->serverX;
      mw->originalY = mini->serverY;                   
                
      mw->hover = TRUE;
      ms->hover = TRUE;
      
      ms->activeGroup = miniwinGetMindowGroupOf (mw->parent, ms->taskbar);
            
      free (ms->hoveredWindow);
      ms->hoveredWindow = malloc (sizeof(XTextItem));
      ms->hoveredWindow->chars = mw->parent->resClass;
      ms->hoveredWindow->nchars = strlen (mw->parent->resClass);
      ms->hoveredWindow->delta = 5;
      ms->hoveredWindow->font = None;

      miniwinRearrangeTaskbar (mini->screen);
  
      mini->paint.opacity = (OPAQUE * ms->opacityOnHover) / 100;
   }
}

static void
setWindowMatrix (CompWindow *w)
{
    w->matrix = w->texture.matrix;
    w->matrix.x0 -= (w->attrib.x * w->matrix.xx);
    w->matrix.y0 -= (w->attrib.y * w->matrix.yy);
}

static void
miniwinHoverOff (CompWindow *mini)
{
   MINIWIN_WINDOW (mini);
   MINIWIN_SCREEN (mini->screen);
   if (mw->parent) miniwinUnstressWindow (mw->parent);
   if (mw->hover)
   {         
      (*mini->screen->setWindowScale)(mini, 1.0f, 1.0f);
      mini->paint.opacity = (OPAQUE * ms->mindowOpacity) / 100;
      mw->hover = FALSE;
      ms->hover = FALSE;
      free (ms->hoveredWindow);
      ms->hoveredWindow = 0;
   }
   setWindowMatrix (mini);
   CompWindow *dock = findWindowAtScreen (mini->screen, ms->dock);
   if (dock) miniwinGrowDock (dock, miniwinNumberOfSlots(mini->screen), miniwinNumberOfRows(mini->screen));
   //miniwinRearrangeTaskbar (mini->screen);
}

static void
miniwinShowMindowOnViewport (CompWindow *mini)
{
   MINIWIN_SCREEN (mini->screen);
   MINIWIN_WINDOW (mini);
   MindowSlot *slot = getFreeSlot (ms->taskbar);
   slot->mindow = mini->id;
   slot->window = mw->parent;
          
   miniwinCheckMindow (mini);
   miniwinRearrangeTaskbar (mini->screen);
}

static void
miniwinHideMindowFromViewport (CompWindow *mini)
{
   MINIWIN_SCREEN (mini->screen);
   MINIWIN_WINDOW (mini);
   
   MindowSlot *slot = getSlotOf (ms->taskbar, mini->id);
   if (slot)
   {
      slot->mindow = 0;
      slot->window = 0;
      miniwinHoverOff (mini);
      miniwinCheckMindow (mini);
      miniwinHideMindow (mini);
      if (ms->showMaximized) miniwinHide (mini);
   }      
}

static void 
miniwinHideWindowComplete (CompWindow *window)
{
   miniwinChangeState (window, MINIWIN_WINDOW_STATE_NORMAL_HIDDEN);
   (*window->screen->setWindowScale) (window, 1.0f, 1.0f);
   miniwinHide (window);
}

static void
miniwinShowWindowComplete (CompWindow *window)
{
   miniwinChangeState (window, MINIWIN_WINDOW_STATE_NORMAL_NONE);
   miniwinShow (window);
   (*window->screen->setWindowScale) (window, 1.0f, 1.0f);
   setWindowState (window->screen->display, window->state, window->id);
}

//now, this is getting interesting, because we are doing a lot of initialization stuff in here.
static Bool
miniwinPaintWindow (CompWindow *w, const WindowPaintAttrib *attrib, Region region, unsigned int mask)
{
   MINIWIN_WINDOW (w);
   Bool status;
   CompScreen *screen = w->screen;
   MINIWIN_SCREEN (screen);  
        
   switch (mw->state)
   {
   case MINIWIN_WINDOW_STATE_DOCK_CREATED:
      miniwinInitializeDock (w);
      miniwinRearrangeTaskbar (w->screen);
      break;
   case MINIWIN_WINDOW_STATE_DOCK_NONE: break;
   case MINIWIN_WINDOW_STATE_NORMAL_NONE:
   if (!ms->mindowsOnAllWorkspaces && ms->showMaximized)
   {
     Bool wasOnViewport = mw->onViewport;
     mw->onViewport = w->serverX >= 0-w->width && w->serverX < screen->width;
     if (wasOnViewport != mw->onViewport)
     {
        if (mw->onViewport && mw->icon)
        {
           miniwinShowMindowOnViewport (mw->icon);
        }
        else if (mw->icon)
        {
           miniwinHideMindowFromViewport (mw->icon);
        }  
     }
   }
   case MINIWIN_WINDOW_STATE_NORMAL_HIDDEN:
     if (mw->icon) 
     {
        CompWindow *icon = mw->icon;
        addWindowDamage(icon);
        XWMHints *hints = XGetWMHints (w->screen->display->display, w->id);
        if (hints && hints->flags & XUrgencyHint)
        {
           MINIWIN_WINDOW (icon);
           if (mw && !mw->urgent)
           {
            mw->urgent = TRUE;
            mw->max_angle = 20.0f;
           }
        }
        else 
        {
           MINIWIN_WINDOW (icon);
           if (mw)
           {
            mw->urgent = FALSE;
           }
        }
     }
     if (mw->state == MINIWIN_WINDOW_STATE_NORMAL_HIDDEN) return TRUE;
     else break;
   case MINIWIN_WINDOW_STATE_MINDOW_HIDDEN: return TRUE;
   case MINIWIN_WINDOW_STATE_NORMAL_HIDING:
   case MINIWIN_WINDOW_STATE_NORMAL_UNHIDING:
      mask |= PAINT_WINDOW_TRANSFORMED_MASK;
   default:
      break;
   }
      
   if (mw->hover || mw->stressed)
   {
      mask |= PAINT_WINDOW_TRANSFORMED_MASK;
      ms->hover = TRUE;
   }

   if (ms->dock == 0)
   {
      miniwinCreateDock (screen);
   }
   
   if (mw->state == MINIWIN_WINDOW_STATE_NORMAL_INITIALIZING)
   {
      mw->destinationX = -1;
      mw->destinationY = -1;
   }
   
   UNWRAP (ms, screen, paintWindow);
   status = (*screen->paintWindow) (w, attrib, region, mask);
   WRAP (ms, screen, paintWindow, miniwinPaintWindow);

   //for each newly created window there will be a mindow: it's "icon"
   //which is available until the window gets destroyed.
   if (mw->state == MINIWIN_WINDOW_STATE_NORMAL_INITIALIZING)
   {
      MindowSlot *minSlot = getSlotOf (ms->mindows, w->id);
      if (minSlot) //it's a mindow
      {
         DEBUG ("Trying to initialize a mindow\n");
         mw->state = MINIWIN_WINDOW_STATE_MINDOW_CREATED;
         mw->parent = minSlot->window;
         miniwinInitializeMindow (w);
                  
         CompWindow *parent = mw->parent;
         {
            MINIWIN_WINDOW (parent);
            mw->icon = w;     
            mw->state = MINIWIN_WINDOW_STATE_NORMAL_NONE;
         }         
      }
      else //it's a window
      {
         if (ms->wMask & w->type)
         {
            mw->state = MINIWIN_WINDOW_STATE_NORMAL_NONE;
            DEBUG ("Trying to create a mindow from a window\n");
            miniwinCreateMindow (w);
         }
      }
   }
   
   return status;
}

static void 
miniwinPreparePaintScreen (CompScreen *screen, int msSinceLastPaint)
{
   MINIWIN_SCREEN (screen);
   
   if (ms->resetGroupTimeout < 0 && ms->activeGroup != -1)
   {
      ms->activeGroup = -1;
      miniwinRearrangeTaskbar (screen);
   }
   else ms->resetGroupTimeout -= msSinceLastPaint;
   if (ms->resetGroupTimeout <= -1) ms->resetGroupTimeout = -1;

   CompWindow *w;
   ms->animation = FALSE;
   for (w = screen->windows; w; w = w->next)
	{
	  MINIWIN_WINDOW (w);
	  if (!mw) continue;
	  
	  if (mw->stressCountdown)
	  {
	     mw->stressCountdown -= msSinceLastPaint;
	     if (mw->stressCountdown <= 0)
	     {
	       forEachWindowOnScreen (screen, miniwinStressHelper, 0);
	       mw->stressCountdown = 0;
	     }
	  }
	  
	  if (mw->step)
	  {	               
	     ms->animation = TRUE;
        mw->step -= (100*msSinceLastPaint)/ms->animationTime ? (100*msSinceLastPaint)/ms->animationTime : 1;
        
        if (mw->step <= 0)
	     {
	       mw->step = 0;          
	     }
	       
	     if (mw->state == MINIWIN_WINDOW_STATE_NORMAL_HIDING)
	     {     
          if (mw->step == 0)
          {
             miniwinHideWindowComplete (w);
             return;
          }
	             
          double step = mw->step / 100.0f;

          int diffX = abs (w->width - mw->icon->width);
          int width = mw->icon->width + step * diffX;
          double scaleX = width / (w->width * 1.0f);
        
          int diffY = abs (w->height - mw->icon->height);
          int height = mw->icon->height + step * diffY;
          double scaleY = height / (w->height * 1.0f);
        
          (*screen->setWindowScale) (w, scaleX, scaleY);
          
          int distX = (1.0f - step) * (mw->icon->serverX - mw->originalX);
          int distY = (1.0f - step) * (mw->icon->serverY - mw->originalY);
        
          int dX = (mw->originalX + distX) - w->serverX;
          int dY = (mw->originalY + distY) - w->serverY;
          
          moveWindow (w, dX, dY, TRUE, TRUE);     
          
          w->serverX += dX;
          w->serverY += dY;   
	    }
	    else if (mw->state == MINIWIN_WINDOW_STATE_NORMAL_UNHIDING)
	    {
	       if (mw->step == 0)
          {
             miniwinShowWindowComplete (w);
             return;
          }          
          double step = 1.0f - (mw->step / 100.0f);

          int diffX = abs (w->width - mw->icon->width);
          int width = mw->icon->width + step * diffX;
          double scaleX = width / (w->width * 1.0f);
        
          int diffY = abs (w->height - mw->icon->height);
          int height = mw->icon->height + step * diffY;
          double scaleY = height / (w->height * 1.0f);
        
          (*screen->setWindowScale) (w, scaleX, scaleY);
          
          MiniwinWindow *tmp = mw;
          {
            MINIWIN_WINDOW (tmp->icon);
            tmp = mw;
          }
          
          int distX = step * (mw->originalX - tmp->originalX);
          int distY = step * (mw->originalY - tmp->originalY);
        
          int dX = (tmp->originalX + distX) - w->serverX;
          int dY = (tmp->originalY + distY) - w->serverY;
          
          moveWindow (w, dX, dY, TRUE, TRUE);     
          
          w->serverX += dX;
          w->serverY += dY;
	    } 
       addWindowDamage (w);
       damageScreen (w->screen);
     }
     else if (mw->urgent)
     {         
         if (mw->max_angle > 0.0f)
         {
            if (mw->angle >= mw->max_angle)
               mw->rotate_speed = -msSinceLastPaint / 4.0f;
            else if (mw->angle <= -mw->max_angle)
               mw->rotate_speed = msSinceLastPaint / 4.0f;
            if (mw->rotate_speed == 0.0f) mw->rotate_speed = msSinceLastPaint;
            
            mw->max_angle -= msSinceLastPaint / 200.0f;
            mw->angle += mw->rotate_speed;         
         }
         else
         {
            mw->max_angle = 0.0f;
            mw->angle = 0.0f;
         }
      }
     
      if (mw->destinationX != -1 && mw->destinationY != -1)
      {
         int moveX=0, moveY=0;
         if (mw->destinationX != w->attrib.x)
         {
            moveX = (mw->destinationX - w->attrib.x) / msSinceLastPaint * (ms->animationTime / 50.0f);
            if (mw->destinationX > w->attrib.x) moveX+=1;
            else if (mw->destinationX <w->attrib.x) moveX-=1;
         }
         if (mw->destinationY != w->attrib.y)
         {
            moveY = (mw->destinationY - w->attrib.y) / msSinceLastPaint * (ms->animationTime / 50.0f);
            if (mw->destinationY > w->attrib.y) moveY+=1;
            else if (mw->destinationY < w->attrib.y) moveY-=1;
         }
         if (moveX || moveY)
         {
            moveWindow (w, moveX, moveY, TRUE, TRUE);
            syncWindowPosition (w);
         }
         else
            mw->destinationX = mw->destinationY = -1;
      }
   }
  
   

   UNWRAP (ms, screen, preparePaintScreen);
   (*screen->preparePaintScreen) (screen, msSinceLastPaint);
   WRAP (ms, screen, preparePaintScreen, miniwinPreparePaintScreen);
}

static Bool
miniwinPaintScreen (CompScreen              *s,
                      const ScreenPaintAttrib *sAttrib,
                      Region                  region,
                      int                     output,
                      unsigned int            mask)
{
  Bool status;
  
  MINIWIN_SCREEN (s);

  if (ms->hover || ms->animation)
    mask |= PAINT_SCREEN_WITH_TRANSFORMED_WINDOWS_MASK;
    
  if (ms->animation)
  {
     damagePendingOnScreen (s);
     mask |= PAINT_SCREEN_WITH_TRANSFORMED_WINDOWS_MASK;
  }	  

  UNWRAP (ms, s, paintScreen);
  status = (*s->paintScreen) (s, sAttrib, region, output, mask);
  WRAP (ms, s, paintScreen, miniwinPaintScreen);

  return status;
}

static Bool
miniwinBindPixmap (CompWindow *mindow, CompWindow *window)
{
   CompScreen *screen = window->screen;
   
   disableTexture (screen, &mindow->texture);
   glXDestroyGLXPixmap (screen->display->display, mindow->texture.pixmap);
	mindow->texture.pixmap = None;
   //bindWindow (window);
   if (!bindPixmapToTexture (screen, &mindow->texture, window->pixmap, 
                             mindow->width, mindow->height, mindow->attrib.depth))
   {
      return FALSE;
   }
   /*unsigned int target;
   CompFBConfig *config = &screen->glxPixmapFBConfigs[window->attrib.depth];
   int          attribs[] = {
     GLX_TEXTURE_FORMAT_EXT, config->textureFormat,
     GLX_MIPMAP_TEXTURE_EXT, config->mipmap,
     None
   };

   bindWindow(window);
   releasePixmapFromTexture (mindow->screen, &mindow->texture);
   Pixmap old_pixmap = mindow->texture.pixmap;
   mindow->texture.pixmap = (*screen->createPixmap) (screen->display->display, config->fbConfig, window->pixmap, attribs);
   screen->queryDrawable (screen->display->display, window->texture.pixmap, GLX_TEXTURE_TARGET_EXT, &target);
   
   if ((!mindow->mapNum && mindow->attrib.map_state == IsViewable)) return FALSE;
      
   switch (target) {
   case GLX_TEXTURE_2D_EXT:
      mindow->texture.target = GL_TEXTURE_2D;
      DEBUG("GL: GL_TEXTURE_2D\n");
      mindow->texture.matrix.xx = 1.0f / mindow->width;
      if (config->yInverted)
      {
        mindow->texture.matrix.yy = 1.0f / mindow->height;
        mindow->texture.matrix.y0 = 0.0f;
      }
      else
      {
        mindow->texture.matrix.yy = -1.0f / mindow->height;
        mindow->texture.matrix.y0 = 1.0f;
      }
      break;
   case GLX_TEXTURE_RECTANGLE_EXT:
      mindow->texture.target = GL_TEXTURE_RECTANGLE_ARB;
      DEBUG("GL: GL_TEXTURE_RECTANGLE_ARB\n");
      mindow->texture.matrix.xx = (float)window->width/(float)mindow->width;
            
      {
         if (config->yInverted)
         { 
            mindow->texture.matrix.yy = ((float)window->height/(float)mindow->height);
            mindow->texture.matrix.y0 = 0.0f;
         }
         else
         {
            mindow->texture.matrix.yy = -((float)window->height/(float)mindow->height);
            mindow->texture.matrix.y0 = window->height;
         }
      }

      break;
   default:   
      DEBUG("GL: missing GL_TEXTURE_TARGET_EXT -- miniwin won't work :(\n");
      fprintf(stderr,"Return:%d\n",target);
      glXDestroyGLXPixmap (screen->display->display, mindow->texture.pixmap);
      mindow->texture.pixmap = old_pixmap;    
      return FALSE;
   }

   glGenTextures (1, &mindow->texture.name);
   glBindTexture (mindow->texture.target, mindow->texture.name);
   screen->bindTexImage (screen->display->display, mindow->texture.pixmap, GLX_FRONT_LEFT_EXT, NULL);

   glXDestroyGLXPixmap (screen->display->display, old_pixmap);*/
   
   return TRUE;
}

static Bool
miniwinDamageWindowRect (CompWindow *w, Bool initial, BoxPtr rect)
{
   Bool status;
   MINIWIN_WINDOW (w);
   MINIWIN_SCREEN (w->screen);
   

    
   if (mw->icon)
   {
      /*w->screen->releaseTexImage (w->screen->display->display, mw->icon->texture.pixmap, GLX_FRONT_LEFT_EXT);
      glBindTexture (w->texture.target, mw->icon->texture.name);
      w->screen->bindTexImage (w->screen->display->display, w->texture.pixmap, GLX_FRONT_LEFT_EXT, NULL);*/
      miniwinBindPixmap (mw->icon, w);
      mw->icon->texture.oldMipmaps = TRUE;
   }
   else if (mw->parent)
   {
      w->texture.oldMipmaps = TRUE;
   }

   UNWRAP (ms, w->screen, damageWindowRect);
   status = (*w->screen->damageWindowRect) (w, initial, rect);
   WRAP (ms, w->screen, damageWindowRect, miniwinDamageWindowRect);

   return status;
}

static void
miniwinDrawWindowTexture (CompWindow *w, CompTexture *texture, const WindowPaintAttrib *attrib, unsigned int mask)
{
   MINIWIN_SCREEN (w->screen);
   MINIWIN_WINDOW (w);
   
   Bool damage = FALSE;
		
   glPushMatrix ();        
      
   if (mw->parent && !mw->parent->destroyed)
   {
      if (w->texture.oldMipmaps)
      {
         glBindTexture (w->texture.target, w->texture.name);
         if (w->texture.target == GL_TEXTURE_2D)
         {
            glTexParameteri (w->texture.target, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
            (*w->screen->generateMipmap) (w->texture.target);
            texture->oldMipmaps = FALSE;
         }
      }
    
      if (!mw->parent->pixmap)
      { //try to repair
         bindWindow (mw->parent);
         if (!mw->parent->pixmap)
         { //and if there is still no pixmap, signal the user that something is wrong
            w->paint.saturation = COLOR * 0.1;
         }
      }
      else
      {
         if (mw->parent->texture.target == GL_TEXTURE_RECTANGLE_ARB)
         {
            w->texture.matrix.xx = mw->parent->attrib.width / w->attrib.width;
            w->texture.matrix.yy = -mw->parent->attrib.height / w->attrib.height;
            w->texture.matrix.y0 = mw->parent->attrib.height;
         }
         int px = - (w->attrib.width * attrib->xScale - w->attrib.width) / 2.0f;
         int py = - (w->attrib.height * attrib->yScale - w->attrib.height) / 2.0f;         
         glTranslatef (px, py, 0.0f);
         if (px || py)
            damage = TRUE;
      }
   }
   
   if (mw->urgent)
   {
      int filter;
      
      glTranslatef (w->attrib.x, w->attrib.y, 0.0f);
      glTranslatef (w->attrib.width*attrib->xScale/2.0f, w->attrib.height*attrib->yScale/2.0f, 0.0f);
      glRotatef (mw->angle, 0.0f, 0.0f, 1.0f);
      glTranslatef (-w->attrib.width*attrib->xScale/2.0f, -w->attrib.height*attrib->yScale/2.0f, 0.0f);
      glScalef (attrib->xScale, attrib->yScale, 0.0f);
      glTranslatef (-w->attrib.x, -w->attrib.y, 0.0f);

      filter = w->screen->filter[WINDOW_TRANS_FILTER];

      enableTexture (w->screen, texture, filter);
         
      glEnable (GL_BLEND);
      if (attrib->opacity != OPAQUE || attrib->brightness != BRIGHT)
      {
         GLushort color;

         color = (attrib->opacity * attrib->brightness) >> 16;

         screenTexEnvMode (w->screen, GL_MODULATE);
         glColor4us (color, color, color, attrib->opacity);

         (*w->screen->drawWindowGeometry) (w);

         glColor4usv (defaultColor);
         screenTexEnvMode (w->screen, GL_REPLACE);
      }
      else
      {
         (*w->screen->drawWindowGeometry) (w);
      }

      glDisable (GL_BLEND);
         
      disableTexture (w->screen, texture);
   }
   else
   {      
      UNWRAP (ms, w->screen, drawWindowTexture);
      (*w->screen->drawWindowTexture) (w, texture, attrib, mask);
      WRAP (ms, w->screen, drawWindowTexture, miniwinDrawWindowTexture);
   }
  
   glPopMatrix ();
   
   if (damage) damageScreen (w->screen);   
}

static void
miniwinDrawWindowGeometry (CompWindow *w)
{
   MINIWIN_WINDOW (w);
   MINIWIN_SCREEN (w->screen);
   
/*   static double angle;
   if (angle >= 2*360.0f) angle = -2*360.0f;
   
   if (mw->parent)
   {
      glMatrixMode (GL_MODELVIEW);
      glTranslatef (w->attrib.x + w->attrib.width, w->attrib.y, 0.0f);
      glRotatef (angle, 0.0f, 1.0f, 0.0f);
      glTranslatef (-w->attrib.x - w->attrib.width, -w->attrib.y, 0.0f);      
      
      glMatrixMode (GL_PROJECTION);
      glPushMatrix ();
      glTranslatef (0.0f, 0.0f, 0.0f);
//      glTranslatef (w->attrib.x + w->attrib.width, w->attrib.y, -100.0f);
//      glRotatef (angle, 0.0f, 0.0f, 1.0f);
//      glTranslatef (-w->attrib.x - w->attrib.width, -w->attrib.y, 0.0f);
      
      angle += 0.2f;
   }*/

   
	UNWRAP (ms, w->screen, drawWindowGeometry);
	(*w->screen->drawWindowGeometry) (w);
	WRAP (ms, w->screen, drawWindowGeometry, miniwinDrawWindowGeometry);
	
/*	if (mw->parent)
	{
	  glPopMatrix();
	  glMatrixMode (GL_MODELVIEW);
	}*/

}

//TODO: add some code in order to be able to move iconified windows more easily.
static void
miniwinHandleEvent (CompDisplay* d, XEvent* event)
{
   CompWindow* w;
   MINIWIN_DISPLAY (d);
   
   switch (event->type) {
   case MotionNotify:
      w = findWindowAtDisplay (d, event->xmotion.window);
      if (w)
      {
         MINIWIN_SCREEN (w->screen);
         
         int length, eventpos;
         if (ms->dockSide == BOTTOM || ms->dockSide == TOP)
         {
            length = w->width;
            eventpos = event->xmotion.x;
         }
         else
         {
            length = w->height;
            eventpos = event->xmotion.y;
         }
         
         int center = length / 2;
         int diff = 0;         
         if (eventpos <= center)
            diff = eventpos;
         else
            diff = length - eventpos;

         diff = 100 * diff / length;         
         if (diff > 100) diff = 100;
         else if (diff < 0) diff = 0;
         
         double scale = 1 + (ms->scaleOnHover-1.0)*diff/100.0f;
         ms->hoverFactor = scale;
         ms->resetGroupTimeout = ms->groupTimeout;
                  
         (*w->screen->setWindowScale)(w, scale, scale);
         CompWindow *dock = findWindowAtScreen (w->screen, ms->dock);
         if (dock) miniwinGrowDock (dock, miniwinNumberOfSlots(w->screen), miniwinNumberOfRows(w->screen));
      }
      break;
   case VisibilityNotify:
      w = findWindowAtDisplay (d, event->xvisibility.window);
      if (w)
      {
         MINIWIN_WINDOW (w);
         if (mw->parent)
         {
            miniwinRearrangeTaskbar (w->screen);
         }
      }
      break;
   case FocusOut:
      w = findWindowAtDisplay (d, event->xfocus.window);
      if (w)
      {
         MINIWIN_SCREEN (w->screen);
         Window focus;
         int tmp;
         XGetInputFocus (d->display, &focus, &tmp);
         if (focus == ms->dock)
         {
            XSetInputFocus (d->display, w->id, RevertToParent, CurrentTime);  
         }
      }
      break;
   case FocusIn:
      w = findWindowAtDisplay (d, event->xfocus.window);
      if (w)
      {
         MINIWIN_WINDOW (w);
         if (mw->state == MINIWIN_WINDOW_STATE_NORMAL_HIDDEN)
         {
            miniwinMaximize (mw->icon);
         }
      }
      break;
   //Iconify on minimize:
   case UnmapNotify:
      w = findWindowAtDisplay (d, event->xunmap.window);
      if (w)
      {
         MINIWIN_SCREEN (w->screen);
         MINIWIN_WINDOW (w);
         if (mw->parent)
         {
            if (w->inShowDesktopMode)
            {
               DEBUG ("minimizing the dock (or a mindow)? no way!\n");
               leaveShowDesktopMode(w->screen, w);
               break;
            }
         }
         if (w->minimized || w->inShowDesktopMode)
         {
            if (mw->state == MINIWIN_WINDOW_STATE_NORMAL_NONE)
            {
               XMapWindow (w->screen->display->display, w->id);
               miniwinMinimize (w);
               miniwinRearrangeTaskbar (w->screen);
            }
         }
      }
      break;
   case EnterNotify:
      w = findWindowAtDisplay (d, event->xcrossing.window);
      if (w)
      {
         MINIWIN_WINDOW (w);
         if (mw->state == MINIWIN_WINDOW_STATE_MINDOW_TASKBAR)
         {
            miniwinHoverOn (w);
         }
         else if (mw->state == MINIWIN_WINDOW_STATE_DOCK_HIDDEN)
         {
            miniwinShowDock (w);
         }
      }
      break;
   case LeaveNotify:
      w = findWindowAtDisplay (d, event->xcrossing.window);
      if (w)
      {
         MINIWIN_WINDOW (w);
         if (mw->state == MINIWIN_WINDOW_STATE_MINDOW_TASKBAR)
         {
            miniwinHoverOff (w);
         }
      }
      break;
   case DestroyNotify:
      w = findWindowAtDisplay (d, event->xdestroywindow.window);
      if (w)
      {
         MINIWIN_WINDOW (w); 
         if (mw && mw->icon)
         {
            miniwinDestroyMindow (mw->icon);
            {
               CompWindow *icon = mw->icon;
               MINIWIN_WINDOW (icon);
               if (mw) mw->parent = 0;
            }
            mw->icon = 0;
         }
      }   
      break;
   case ButtonPress:
   case ButtonRelease:       
      w = findWindowAtDisplay (d, event->xkey.window);
      if (w)
      {         
         CompScreen *s = w->screen;      
         MINIWIN_SCREEN (s);           
         MINIWIN_WINDOW (w);
                  
         if (mw->state == MINIWIN_WINDOW_STATE_DOCK_NONE)
         {
            if (ms->dockHideOnClick)
            {
               miniwinHideDock (w);
            }
            else
            {
               miniwinRearrangeTaskbar (s);
            }
         }
            
         if (!(ms->wMask & w->type)) break;
      }
   case KeyPress:
   case KeyRelease:
      w = findWindowAtDisplay (d, event->xkey.window);
      if (w)
      {  
         CompScreen *s = w->screen;
         MINIWIN_SCREEN (s);       
         MINIWIN_WINDOW (w);
                    
         if (!(ms->wMask & w->type)) break;
         
         /*if (eventMatches (d,event,&ms->opt[MINIWIN_SCREEN_OPTION_DEMINDOW_KEY]))
         {
            if (mw->state == MINIWIN_WINDOW_STATE_MINDOW_TASKBAR)
            {
               CompWindow *parent = mw->parent;
               MINIWIN_WINDOW (parent);
               if (mw->state == MINIWIN_WINDOW_STATE_NORMAL_HIDDEN)
               {
                  miniwinMaximize (w);
                  ms->resetGroupTimeout = 0;
               }
               else if (parent == findWindowAtScreen (s, getActiveWindow(s->display, event->xkey.root)))
               {
                  miniwinMinimize (parent);
               }
               else
               {
                  miniwinMaximize (w);
                  ms->resetGroupTimeout = 0;
               }
            }
         }
         else if (eventMatches (d,event,&ms->opt[MINIWIN_SCREEN_OPTION_CLOSE_WINDOW_KEY]))
         {
            if (mw->state == MINIWIN_WINDOW_STATE_MINDOW_TASKBAR)
            {
               if (mw->parent)
               {
                  miniwinUnstressWindow (mw->parent);
                  closeWindow (mw->parent, event->xkey.time);
               }
            }
         }*/
      }
      break;
   default:
      break;
   }
   
   UNWRAP (md, d, handleEvent);
   (*d->handleEvent) (d, event);
   WRAP (md, d, handleEvent, miniwinHandleEvent);
}

//From here on, it will get boring. I don't mind if you just close the file now.
//What follows is only compiz initialization stuff...
 
static CompOption*
miniwinGetScreenOptions (CompScreen* screen, int* count)
{
   DEBUG ("Getting a screen option\n");
   MINIWIN_SCREEN (screen); //hate this way of getting a miniwin object. have to change that.
   *count = NUM_OPTIONS (ms);
   return ms->opt;
}

static Bool
miniwinSetScreenOption (CompScreen* screen, char* name, CompOptionValue* value)
{
   DEBUG ("Setting a screen option\n");
   CompOption* o;
   int index;
   
   MINIWIN_SCREEN (screen);
   
   o = compFindOption (ms->opt, NUM_OPTIONS (ms), name, &index);
   if (!o) return FALSE;
   
   switch (index) {
   case MINIWIN_SCREEN_OPTION_SCALE_ON_HOVER:
      if (compSetFloatOption (o, value))
      {
         ms->scaleOnHover = o->value.f;
         return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_DOCK_OPACITY:
      if (compSetIntOption (o, value))
      {
         ms->dockOpacity = o->value.i;
         return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_TASKBAR_SPACING:
      if (compSetIntOption (o, value))
      {
         ms->taskbarSpacing = o->value.i;
         if (ms->dockSide == BOTTOM || ms->dockSide == TOP)
            ms->nSlotsPerRow = screen->width / ( ms->mindowWidth * 2 + ms->taskbarSpacing );
         else
            ms->nSlotsPerRow = screen->height / ( ms->mindowWidth * 2 + ms->taskbarSpacing );
         return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_ANIMATION_TIME:
      if (compSetIntOption (o, value))
      {
         ms->animationTime = o->value.i;
         return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_OPACITY_ON_HOVER:
      if (compSetIntOption (o, value))
      {
         ms->opacityOnHover = o->value.i;
         return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_MINDOW_OPACITY:
      if (compSetIntOption (o, value))
      {
         ms->mindowOpacity = o->value.i;
         return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_DOCK_SIZE:
      if (compSetIntOption (o, value))
      {
         ms->dockSize = o->value.i;
         return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_HIDDEN_DOCK_SIZE:
      if (compSetIntOption (o, value))
      {
         ms->hiddenDockSize = o->value.i;
         return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_DOCK_COLOUR:
      if (compSetIntOption (o, value))
      {
         ms->dockColour = o->value.i;
         XSetWindowAttributes attr;
         attr.background_pixel = ms->dockColour;
       	XChangeWindowAttributes (screen->display->display, ms->dock, CWBackPixel, &attr);
         return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_MINDOW_WIDTH:
      if (compSetIntOption (o, value))
      {
         ms->mindowWidth = o->value.i;
         if (ms->dockSide == TOP || ms->dockSide == BOTTOM)
         {
            ms->nSlotsPerRow = screen->width / ( ms->mindowWidth * 2 + ms->taskbarSpacing );
         }
         else
         {
            ms->nSlotsPerRow = screen->height / ( ms->mindowWidth * 2 + ms->taskbarSpacing );
         }
         return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_DOCK_SIDE:
      if (compSetIntOption (o, value))
      {
         ms->dockSide = o->value.i;
         return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_HIGHLIGHT_TIMEOUT:
      if (compSetIntOption (o, value))
      {
         ms->stressTimeout = o->value.i;
         return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_HIDE_GROUP_TIMEOUT:
      if (compSetIntOption (o, value))
      {
         ms->resetGroupTimeout = o->value.i;
         return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_MINDOWS_ON_ALL_WORKSPACES:
      if (compSetBoolOption (o, value))
      {
         ms->mindowsOnAllWorkspaces = o->value.b;
         return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_SHOW_MAXIMIZED_WINDOWS:
      if (compSetBoolOption (o, value))
      {
         ms->showMaximized = o->value.b;
         return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_DOCK_HIDE_ON_CLICK:
      if (compSetBoolOption (o, value))
      {
         ms->dockHideOnClick = o->value.b;
         return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_RUNS_ON_AIGLX:
      if (compSetBoolOption (o, value))
      {
         ms->runsOnAiglx = o->value.b;
         return TRUE;
      }
      break;
   /*case MINIWIN_SCREEN_OPTION_DEMINDOW_KEY:
      {
         removeScreenBinding (screen, &o->value.bind);
         if (compSetBindingOption (o, value)) return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_CLOSE_WINDOW_KEY:
      {
         removeScreenBinding (screen, &o->value.bind);
         if (compSetBindingOption (o, value)) return TRUE;
      }
      break;*/
   case MINIWIN_SCREEN_OPTION_WINDOW_TYPE:
      if (compSetOptionList (o, value))
      {
         ms->wMask = compWindowTypeMaskFromStringList (&o->value);
         ms->wMask &= ~CompWindowTypeDesktopMask;
         return TRUE;
      }
   case MINIWIN_SCREEN_OPTION_DOCK_FONT:
      /*if (compSetStringOption (o, value))
      {
         free (ms->dockFont);
         ms->dockFont = malloc (strlen(value->s)+1);
         strcpy (ms->dockFont, value->s);
         return TRUE;
      }*/
   default: 
      break;
   }
   
   return FALSE;
}

static void
miniwinScreenInitOptions (MiniwinScreen* ms)
{
   CompOption* o;
   int i;
   
   DEBUG ("Initializing options...\n");
   
   o = &ms->opt[MINIWIN_SCREEN_OPTION_DOCK_FONT];
   o->name = "dock_font";
   o->shortDesc = "Font used for the dock";
   o->longDesc = "Font used for the dock";
   o->type = CompOptionTypeString;
   o->value.s = strdup(MINIWIN_DOCK_FONT_DEFAULT);
   o->rest.s.string = 0;
   o->rest.s.nString = 0;
   
   o = &ms->opt[MINIWIN_SCREEN_OPTION_SCALE_ON_HOVER];
   o->name = "icon_scale_on_hover_factor";
   o->shortDesc = "Scale icons on hover";
   o->longDesc = "Scale icons on hover (factor)";
   o->type = CompOptionTypeFloat;
   o->value.f = MINIWIN_SCALE_ON_HOVER_DEFAULT;
   o->rest.f.min = MINIWIN_SCALE_ON_HOVER_MIN;
   o->rest.f.max = MINIWIN_SCALE_ON_HOVER_MAX;
   o->rest.f.precision = MINIWIN_SCALE_ON_HOVER_STEP;
   
   o = &ms->opt[MINIWIN_SCREEN_OPTION_MINDOW_WIDTH];
   o->name = "icon_width";
   o->shortDesc = "Standard width of icons";
   o->longDesc = "Standard width of icons";
   o->type = CompOptionTypeInt;
   o->value.i = MINIWIN_MINDOW_WIDTH_DEFAULT;
   o->rest.i.min = MINIWIN_MINDOW_WIDTH_MIN;
   o->rest.i.max = MINIWIN_MINDOW_WIDTH_MAX;

   o = &ms->opt[MINIWIN_SCREEN_OPTION_DOCK_SIDE];
   o->name = "dock_position";
   o->shortDesc = "Where the dock shall appear";
   o->longDesc = "Position of the dock (0 = bottom, 1 = right, 2 = top, 3 = left)";
   o->type = CompOptionTypeInt;
   o->value.i = MINIWIN_DOCK_SIDE_DEFAULT;
   o->rest.i.min = MINIWIN_DOCK_SIDE_MIN;
   o->rest.i.max = MINIWIN_DOCK_SIDE_MAX;
   
   o = &ms->opt[MINIWIN_SCREEN_OPTION_ANIMATION_TIME];
   o->name = "animation_time";
   o->shortDesc = "Animation duration";
   o->longDesc = "Duration of mindowing animations (in miliseconds)";
   o->type = CompOptionTypeInt;
   o->value.i = MINIWIN_ANIMATION_TIME_DEFAULT;
   o->rest.i.min = MINIWIN_ANIMATION_TIME_MIN;
   o->rest.i.max = MINIWIN_ANIMATION_TIME_MAX;
   
   o = &ms->opt[MINIWIN_SCREEN_OPTION_MINDOW_OPACITY];
   o->name = "icon_opacity";
   o->shortDesc = "Default opacity";
   o->longDesc = "Default opacity for all icons";
   o->type = CompOptionTypeInt;
   o->value.i = MINIWIN_MINDOW_OPACITY_DEFAULT;
   o->rest.i.min = MINIWIN_MINDOW_OPACITY_MIN;
   o->rest.i.max = MINIWIN_MINDOW_OPACITY_MAX;
   
   o = &ms->opt[MINIWIN_SCREEN_OPTION_OPACITY_ON_HOVER];
   o->name = "icon_opacity_on_hover";
   o->shortDesc = "Opacity of hovered icons";
   o->longDesc = "Opacity of hovered icons";
   o->type = CompOptionTypeInt;
   o->value.i = MINIWIN_OPACITY_ON_HOVER_DEFAULT;
   o->rest.i.min = MINIWIN_OPACITY_ON_HOVER_MIN;
   o->rest.i.max = MINIWIN_OPACITY_ON_HOVER_MAX;
   
   o = &ms->opt[MINIWIN_SCREEN_OPTION_TASKBAR_SPACING];
   o->name = "dock_spacing";
   o->shortDesc = "Space between mindows in the taskbar";
   o->longDesc = "Space between mindows in the taskbar";
   o->type = CompOptionTypeInt;
   o->value.i = MINIWIN_TASKBAR_SPACING_DEFAULT;
   o->rest.i.min = MINIWIN_TASKBAR_SPACING_MIN;
   o->rest.i.max = MINIWIN_TASKBAR_SPACING_MAX;
   
   o = &ms->opt[MINIWIN_SCREEN_OPTION_DOCK_OPACITY];
   o->name = "dock_opacity";
   o->shortDesc = "Opacity of the dock";
   o->longDesc = "Opacity of the dock";
   o->type = CompOptionTypeInt;
   o->value.i = MINIWIN_DOCK_OPACITY_DEFAULT;
   o->rest.i.min = MINIWIN_DOCK_OPACITY_MIN;
   o->rest.i.max = MINIWIN_DOCK_OPACITY_MAX;

   o = &ms->opt[MINIWIN_SCREEN_OPTION_DOCK_COLOUR];
   o->name = "dock_color";
   o->shortDesc = "Color of the dock";
   o->longDesc = "Color of the dock";
   o->type = CompOptionTypeInt;
   o->value.i = MINIWIN_DOCK_COLOUR_DEFAULT;
   o->rest.i.min = MINIWIN_DOCK_COLOUR_MIN;
   o->rest.i.max = MINIWIN_DOCK_COLOUR_MAX;

   o = &ms->opt[MINIWIN_SCREEN_OPTION_DOCK_SIZE];
   o->name = "dock_size";
   o->shortDesc = "Size of the dock";
   o->longDesc = "Size of the dock";
   o->type = CompOptionTypeInt;
   o->value.i = MINIWIN_DOCK_SIZE_DEFAULT;
   o->rest.i.min = MINIWIN_DOCK_SIZE_MIN;
   o->rest.i.max = MINIWIN_DOCK_SIZE_MAX;

   o = &ms->opt[MINIWIN_SCREEN_OPTION_HIDDEN_DOCK_SIZE];
   o->name = "dock_hidden_size";
   o->shortDesc = "Size of the hidden dock";
   o->longDesc = "Size of the hidden dock";
   o->type = CompOptionTypeInt;
   o->value.i = MINIWIN_HIDDEN_DOCK_SIZE_DEFAULT;
   o->rest.i.min = MINIWIN_HIDDEN_DOCK_SIZE_MIN;
   o->rest.i.max = MINIWIN_HIDDEN_DOCK_SIZE_MAX;

   o = &ms->opt[MINIWIN_SCREEN_OPTION_HIGHLIGHT_TIMEOUT];
   o->name = "highlight_time";
   o->shortDesc = "Highlight windows after specific time";
   o->longDesc = "Highlight the window after this amount of time (in ms)";
   o->type = CompOptionTypeInt;
   o->value.i = MINIWIN_STRESS_TIMEOUT_DEFAULT;
   o->rest.i.min = MINIWIN_STRESS_TIMEOUT_MIN;
   o->rest.i.max = MINIWIN_STRESS_TIMEOUT_MAX;
   
   o = &ms->opt[MINIWIN_SCREEN_OPTION_HIDE_GROUP_TIMEOUT];
   o->name = "group_timeout";
   o->shortDesc = "Hide a mindow group";
   o->longDesc = "Hide a mindow group after a specific time (in ms)";
   o->type = CompOptionTypeInt;
   o->value.i = MINIWIN_HIDE_GROUP_TIMEOUT_DEFAULT;
   o->rest.i.min = MINIWIN_HIDE_GROUP_TIMEOUT_MIN;
   o->rest.i.max = MINIWIN_HIDE_GROUP_TIMEOUT_MAX;

   o = &ms->opt[MINIWIN_SCREEN_OPTION_MINDOWS_ON_ALL_WORKSPACES];
   o->name = "from_all_workspaces";
   o->shortDesc = "Shows mindows from windows on all workspaces";
   o->longDesc = "Shows mindows from windows on all workspaces (only in show_maximized_windows mode)";
   o->type = CompOptionTypeBool;
   o->value.b = MINIWIN_MINDOWS_ON_ALL_WORKSPACES_DEFAULT;
   
   o = &ms->opt[MINIWIN_SCREEN_OPTION_DOCK_HIDE_ON_CLICK];
   o->name = "dock_hide_on_click";
   o->shortDesc = "Hide the dock on a left-click";
   o->longDesc = "Hide the dock on a left-click";
   o->type = CompOptionTypeBool;
   o->value.b = MINIWIN_DOCK_HIDE_ON_CLICK_DEFAULT;
   
   o = &ms->opt[MINIWIN_SCREEN_OPTION_RUNS_ON_AIGLX];
   o->name = "runs_on_aiglx";
   o->shortDesc = "Is it AIGLX?";
   o->longDesc = "Is it AIGLX?";
   o->type = CompOptionTypeBool;
   o->value.b = MINIWIN_RUNS_ON_AIGLX_DEFAULT;
   
   o = &ms->opt[MINIWIN_SCREEN_OPTION_SHOW_MAXIMIZED_WINDOWS];
   o->name = "show_maximized_windows";
   o->shortDesc = "Keep maximized windows in the dock";
   o->longDesc = "Don\'t hide mindows, even when their windows aren't minimized";
   o->type = CompOptionTypeBool;
   o->value.b = MINIWIN_SHOW_MAXIMIZED_WINDOWS_DEFAULT;
      
   /*o = &ms->opt[MINIWIN_SCREEN_OPTION_DEMINDOW_KEY];
   o->name = "uniconify_key";
   o->shortDesc = "Uniconify with this key";
   o->longDesc = "Uniconify a window";
   o->type = CompOptionTypeBinding;
   o->value.bind.type = CompBindingTypeButton;
   o->value.bind.u.button.modifiers = MINIWIN_DEMINDOW_MODIFIERS_DEFAULT;
   o->value.bind.u.button.button = MINIWIN_DEMINDOW_BUTTON_DEFAULT;

   o = &ms->opt[MINIWIN_SCREEN_OPTION_CLOSE_WINDOW_KEY];
   o->name = "close_window_key";
   o->shortDesc = "Close a mindow\'s window";
   o->longDesc = "Close the parent window of the mindow";
   o->type = CompOptionTypeBinding;
   o->value.bind.type = CompBindingTypeButton;
   o->value.bind.u.button.modifiers = MINIWIN_CLOSE_WINDOW_MODIFIERS_DEFAULT;
   o->value.bind.u.button.button = MINIWIN_CLOSE_WINDOW_BUTTON_DEFAULT;*/
   
   o = &ms->opt[MINIWIN_SCREEN_OPTION_WINDOW_TYPE];
   o->name = "window_type";
   o->shortDesc = "Window Types";
   o->longDesc = "Window types that can be iconified (be rather conservative with that)";
   o->type = CompOptionTypeList;
   o->value.list.type = CompOptionTypeString;
   o->value.list.nValue = N_WIN_TYPE;
   o->value.list.value = malloc (sizeof (CompOptionValue) * N_WIN_TYPE);
   for (i = 0; i < N_WIN_TYPE; ++i)
      o->value.list.value[i].s = strdup (winType[i]);
   o->rest.s.string = windowTypeString;
   o->rest.s.nString = nWindowTypeString;
   
   ms->wMask = compWindowTypeMaskFromStringList (&o->value);      
}

static Bool
miniwinInit (CompPlugin* p)
{
   displayPrivateIndex = allocateDisplayPrivateIndex ();
   if (displayPrivateIndex < 0)
   {
      return FALSE;
   }
   return TRUE;
}

static void
miniwinFini (CompPlugin *p)
{
   if (displayPrivateIndex >= 0)
      freeDisplayPrivateIndex (displayPrivateIndex);
}

static Bool
miniwinInitDisplay (CompPlugin* p, CompDisplay* d)
{
   MiniwinDisplay *md;
   
   md = malloc (sizeof (MiniwinDisplay));
   if (!md) return FALSE;
   
   md->screenPrivateIndex = allocateScreenPrivateIndex (d);
   if (md->screenPrivateIndex < 0)
   {
      free (md);
      return FALSE;
   }
   
   WRAP (md, d, handleEvent, miniwinHandleEvent);
   d->privates[displayPrivateIndex].ptr = md;
   
   return TRUE;
}

static void
miniwinFiniDisplay (CompPlugin* p, CompDisplay* d)
{
   MINIWIN_DISPLAY(d);
   
   freeScreenPrivateIndex (d, md->screenPrivateIndex);
   
   UNWRAP (md, d, handleEvent);
   
   free (md);
}



static Bool
miniwinInitScreen (CompPlugin* p, CompScreen* s)
{
   MiniwinScreen* ms;
   MINIWIN_DISPLAY (s->display);
   
   ms = malloc (sizeof (MiniwinScreen));
   if (!ms) return FALSE;
   
   ms->windowPrivateIndex = allocateWindowPrivateIndex (s);
   if (ms->windowPrivateIndex < 0)
   {
      free (ms);
      return FALSE;
   }   
   
   ms->hoveredWindow = 0;
   
   //General mindow settings
   ms->dockColour = MINIWIN_DOCK_COLOUR_DEFAULT;
   ms->dockSide = MINIWIN_DOCK_SIDE_DEFAULT;
   ms->mindowWidth = MINIWIN_MINDOW_WIDTH_DEFAULT;
   ms->animationTime = MINIWIN_ANIMATION_TIME_DEFAULT;
   ms->mindowOpacity = MINIWIN_MINDOW_OPACITY_DEFAULT;
   ms->scaleOnHover = MINIWIN_SCALE_ON_HOVER_DEFAULT;
   ms->opacityOnHover = MINIWIN_OPACITY_ON_HOVER_DEFAULT;
   ms->mindowsOnAllWorkspaces = MINIWIN_MINDOWS_ON_ALL_WORKSPACES_DEFAULT;
   ms->showMaximized = MINIWIN_SHOW_MAXIMIZED_WINDOWS_DEFAULT;
   ms->animation = FALSE;
   ms->dockSize = MINIWIN_DOCK_SIZE_DEFAULT;
   ms->hiddenDockSize = MINIWIN_HIDDEN_DOCK_SIZE_DEFAULT;
   ms->groupTimeout = MINIWIN_HIDE_GROUP_TIMEOUT_DEFAULT;
   ms->hoverFactor = 1.0;
   
   ms->dockFont = strdup(MINIWIN_DOCK_FONT_DEFAULT);
   
   //Taskbar settings
   ms->taskbarSpacing = MINIWIN_TASKBAR_SPACING_DEFAULT;
   ms->dockOpacity = MINIWIN_DOCK_OPACITY_DEFAULT;
   ms->dockHideOnClick = MINIWIN_DOCK_HIDE_ON_CLICK_DEFAULT;
   
   //Internal settings
   ms->runsOnAiglx = MINIWIN_RUNS_ON_AIGLX_DEFAULT;
   ms->pendingMindows = 0;

   ms->dock = 0;   
   ms->mindows = malloc (sizeof(MindowSlot));
   if (!ms->mindows) return FALSE;
   ms->mindows->next = 0;
   ms->mindows->mindow = 0;
   ms->mindows->window = 0;
   
   ms->taskbar = malloc (sizeof(MindowSlot));
   if (!ms->taskbar) return FALSE;
   ms->taskbar->next = 0;
   ms->taskbar->mindow = 0;
   ms->taskbar->window = 0;
   
   ms->activeGroup = -1;
   ms->resetGroupTimeout = 0;
   ms->stressTimeout = MINIWIN_STRESS_TIMEOUT_DEFAULT;
   
   if (ms->dockSide == BOTTOM || ms->dockSide == TOP)
      ms->nSlotsPerRow = s->width / ( ms->mindowWidth * 2 + ms->taskbarSpacing );
   else
      ms->nSlotsPerRow = s->height / ( ms->mindowWidth * 2 + ms->taskbarSpacing );
   
   miniwinScreenInitOptions (ms);
   s->privates[md->screenPrivateIndex].ptr = ms;

   WRAP (ms, s, paintWindow, miniwinPaintWindow);
   WRAP (ms, s, preparePaintScreen, miniwinPreparePaintScreen);
   WRAP (ms, s, paintScreen, miniwinPaintScreen);
   WRAP (ms, s, drawWindowTexture, miniwinDrawWindowTexture);
   WRAP (ms, s, drawWindowGeometry, miniwinDrawWindowGeometry);
   WRAP (ms, s, damageWindowRect, miniwinDamageWindowRect);
      
   return TRUE;
}

static void
miniwinFiniScreen (CompPlugin* p, CompScreen* s)
{
   MINIWIN_SCREEN(s);
   
   free (ms->dockFont);   
   removeAllSlots (ms->taskbar);
   freeWindowPrivateIndex (s, ms->windowPrivateIndex);
   
   UNWRAP (ms, s, paintWindow);
   UNWRAP (ms, s, preparePaintScreen);
   UNWRAP (ms, s, paintScreen);
   UNWRAP (ms, s, drawWindowTexture);
   UNWRAP (ms, s, drawWindowGeometry);
   UNWRAP (ms, s, damageWindowRect);
   
   free (ms->hoveredWindow);
   free (ms);
}

static Bool
miniwinInitWindow (CompPlugin *p, CompWindow *w)
{
   DEBUG ("Created a window!\n");
   
   MiniwinWindow *mw;

   MINIWIN_SCREEN (w->screen);

   mw = malloc (sizeof (MiniwinWindow));
   if (!mw) return FALSE;

   mw->state = MINIWIN_WINDOW_STATE_NORMAL_INITIALIZING;      
   mw->hover = FALSE;
   mw->urgent = FALSE;
   mw->angle = 0.0f;
   mw->max_angle = 20.0f;
   mw->rotate_speed = 0.0f;
   mw->icon = 0;
   mw->parent = 0;
   mw->step = 0;
   mw->onViewport = TRUE;
   mw->destinationX = -1;
   mw->destinationY = -1;
   mw->stressCountdown = 0;
   mw->stressed = FALSE;
   mw->savedOpacity = OPAQUE;
   
   if (ms->dock == w->id)
   {
      mw->state = MINIWIN_WINDOW_STATE_DOCK_CREATED;
   }
    
   w->privates[ms->windowPrivateIndex].ptr = mw;
   return TRUE;
}

static void
miniwinFiniWindow (CompPlugin *p, CompWindow *w)
{
    MINIWIN_WINDOW (w);
    free (mw);
}

static int
miniwinGetVersion (CompPlugin *plugin,
		  int	     version)
{
    return ABIVERSION;
}

CompPluginDep dockDeps[] = {
    { CompPluginRuleAfter, "decoration" }
};

static CompPluginVTable miniwinVTable = {
   "dock",
   "Have a MacOS like dock",
   "Or, if you prefer: a gdesklet style dock ;)",
   miniwinGetVersion,
   miniwinInit,
   miniwinFini,
   miniwinInitDisplay,
   miniwinFiniDisplay,
   miniwinInitScreen,
   miniwinFiniScreen,
   miniwinInitWindow,
   miniwinFiniWindow,
   0, /* GetDisplayOptions */
   0, /* SetDisplayOption */
   miniwinGetScreenOptions,
   miniwinSetScreenOption,
   dockDeps,
   sizeof (dockDeps) / sizeof (dockDeps[0])
};

CompPluginVTable *
getCompPluginInfo (void)
{
   return &miniwinVTable;
}
