/*
 * Copyright © 2006 Mike Dransfield
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * Author: Mike Dransfield <mike@blueroot.co.uk>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <compiz.h>


#define MOUSEGESTURES_INITIATE_MODIFIERS_DEFAULT (ShiftMask | CompSuperMask)
#define MOUSEGESTURES_INITIATE_BUTTON_DEFAULT    3

#define MOUSEGESTURES_DISPLAY_OPTION_INITIATE      0
#define MOUSEGESTURES_DISPLAY_OPTION_ACTIONS       1
#define MOUSEGESTURES_DISPLAY_OPTION_NUM           2


#define MOUSEGESTURE_MAX_VARIANCE                30
#define MOUSEGESTURE_MIN_DELTA                   80
#define MOUSEGESTURES_MAX_STROKE		 32
#define MOUSEGESTURES_MAX_GESTURES		 128


#define M_UP    'u'
#define M_RIGHT 'r'
#define M_DOWN  'd'
#define M_LEFT  'l'
#define M_9     '9'
#define M_3     '3'
#define M_1     '1'
#define M_7     '7'


#define GET_MOUSEGESTURES_DISPLAY(d) ((MousegesturesDisplay *) (d)->privates[displayPrivateIndex].ptr)
#define MOUSEGESTURES_DISPLAY(d) MousegesturesDisplay *md = GET_MOUSEGESTURES_DISPLAY (d)
#define GET_MOUSEGESTURES_SCREEN(s, md) ((MousegesturesScreen *) (s)->privates[(md)->screenPrivateIndex].ptr)
#define MOUSEGESTURES_SCREEN(s) MousegesturesScreen *ms = GET_MOUSEGESTURES_SCREEN (s, GET_MOUSEGESTURES_DISPLAY (s->display))

#define NUM_OPTIONS(s) (sizeof ((s)->opt) / sizeof (CompOption))

static char
*defaultGestures[] = {};

#define N_GESTURES (sizeof (defaultGestures) / sizeof (defaultGestures[0]))

static int displayPrivateIndex;

typedef struct _Mousegesture {
    char		 stroke[MOUSEGESTURES_MAX_STROKE];
    int     		 nStrokes;
    char		 *pluginName;
    char		 *actionName;
} Mousegesture;


typedef struct _MousegesturesDisplay {
    int		                screenPrivateIndex;
    CompOption                  opt[MOUSEGESTURES_DISPLAY_OPTION_NUM];
    HandleEventProc             handleEvent;
    Bool 			active;
    int 			originalX;
    int 			originalY;
    int 			deltaX;
    int 			deltaY;
    char			currentStroke;
    char			lastStroke;
    char			currentGesture[MOUSEGESTURES_MAX_STROKE];
    int				nCurrentGesture;
    Mousegesture		gesture[MOUSEGESTURES_MAX_GESTURES];
    int 			nGesture;
} MousegesturesDisplay;


static void
mousegesturesDisplayActionInitiate (CompDisplay *d, char *pluginName, char *actionName)
{
    CompOption *argument = NULL;
    int nArgument = 0;
    CompPlugin *p;
    CompOption *options, *option;
    int        nOptions;

    if (!strlen(pluginName) || !strlen(actionName))
	return;

    p = findActivePlugin (pluginName);
    if (p && p->vTable->getDisplayOptions)
    {
	options = (*p->vTable->getDisplayOptions) (d, &nOptions);
	option = compFindOption (options, nOptions, actionName, 0);
    }
    else if (strcmp(pluginName, "core")==0)
    {
	options = compGetDisplayOptions (d, &nOptions);
	option = compFindOption (options, nOptions, actionName, 0);
    }
    else
    {
	fprintf (stderr, "ERROR: Could not find plugin called '%s', check that you have " \
				"it loaded and that you spelt it correctly in the option.\n\n", pluginName);
	return;
    }


    if (option && option->value.action.initiate)
    {
        CompOption *a;

	a = realloc (argument,
	    	     sizeof (CompOption) * (nArgument + 2));
	if (a)
	{
	    argument = a;

	    argument[nArgument].name    = "root";
	    argument[nArgument].type    = CompOptionTypeInt;
	    argument[nArgument].value.i = currentRoot;
	    nArgument++;

	    argument[nArgument].name    = "window";
	    argument[nArgument].type    = CompOptionTypeInt;
	    argument[nArgument].value.i = d->activeWindow;
	    nArgument++;
	}

	(*option->value.action.initiate)  (d,
					   &option->value.action,
					   0,
					   argument,
					   nArgument);
    }
    else
    {
	fprintf (stderr, "ERROR: I found the plugin called '%s' but I could not" \
			 " find the action called '%s', check that you spelt " \
			 "it correctly in the option.\n\n", pluginName, actionName);
    }
}

static Bool
mousegesturesInitiate (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    MOUSEGESTURES_DISPLAY (d);

    md->originalX = pointerX;
    md->originalY = pointerY;
    md->deltaX = 0;
    md->deltaY = 0;
    md->currentStroke = '\0';
    md->lastStroke = '\0';
    md->nCurrentGesture = 0;
    md->active = TRUE;

	if (state & CompActionStateInitButton)
	    action->state |= CompActionStateTermButton;


    return TRUE;
}

static Bool
mousegesturesTerminate (CompDisplay *d,
			CompAction *action,
			CompActionState state,
			CompOption *option,
			int nOption)
{
    int ng, ns;
    Mousegesture mg;
    Bool match = TRUE;

    MOUSEGESTURES_DISPLAY (d);

    ng = md->nGesture;

    while (ng--)
    {
        mg = md->gesture[ng];

	if (mg.nStrokes == md->nCurrentGesture)
	{
	    match = TRUE;

	    for (ns = 0;ns < mg.nStrokes;ns++)
	    {
	        if (md->currentGesture[ns] != mg.stroke[ns])
		    match = FALSE;
	    }

	    if (match)
		mousegesturesDisplayActionInitiate (d, mg.pluginName, mg.actionName);
	}
    }

    return TRUE;
}

static void
mousegesturesHandleEvent (CompDisplay *d, XEvent *event)
{
    MOUSEGESTURES_DISPLAY (d);

    char currentStroke = '\0';
    float diff = 0.0;

    switch (event->type)
    {
	case MotionNotify:
	    md->deltaX += (lastPointerX - pointerX);
	    md->deltaY += (lastPointerY - pointerY);

	    if (abs(md->deltaY) > MOUSEGESTURE_MAX_VARIANCE )
		diff = (float) abs(md->deltaX) / (float) abs(md->deltaY);
	    else if (abs(md->deltaX) > MOUSEGESTURE_MAX_VARIANCE )
		diff = (float) abs(md->deltaY) / (float) abs(md->deltaX);

		// dx and dy are roughly the same so they are going diagonally
	    if (diff > 0.1 && diff < 1.9)
	    {
		if (md->deltaY > 0 && (abs(md->deltaY) > MOUSEGESTURE_MAX_VARIANCE)) // UP
		{
		    if (abs(md->deltaX) > MOUSEGESTURE_MAX_VARIANCE)
		    {
			if (md->deltaX < 0)
			    currentStroke = M_9;
		        else if (md->deltaX > 0)
			    currentStroke = M_7;
		    }
		}
		else if (md->deltaY < 0 && (abs(md->deltaY) > MOUSEGESTURE_MAX_VARIANCE))  //DOWN
		{
		    if (abs(md->deltaX) > MOUSEGESTURE_MAX_VARIANCE)
		    {
			if (md->deltaX < 0)
			    currentStroke = M_3;
			else if (md->deltaX > 0)
			    currentStroke = M_1;
		    }
		}
	    }

	    else if (abs(md->deltaX) > MOUSEGESTURE_MAX_VARIANCE)
	    {
		if (md->deltaX < 0)
		    currentStroke = M_RIGHT;
		else
		    currentStroke = M_LEFT;
		
	    }
	    else if (abs(md->deltaY) > MOUSEGESTURE_MAX_VARIANCE)
	    {
		if (md->deltaY > 0)
		    currentStroke = M_UP;
		else
		    currentStroke = M_DOWN;
	    }
	    break;
	default:
	    break;
    }

    if (currentStroke != '\0' && md->currentStroke != currentStroke)
    {
	md->currentStroke = currentStroke;
	if (md->currentStroke != md->lastStroke)
	{
		md->lastStroke = md->currentStroke;
		if (md->nCurrentGesture<MOUSEGESTURES_MAX_STROKE)
		{
		    md->currentGesture[md->nCurrentGesture] = md->currentStroke;
		    md->nCurrentGesture++;
		}
	}
	md->deltaX = 0;
	md->deltaY = 0;
    }
    else if (md->currentStroke != '\0' && md->currentStroke == currentStroke)
    {
	md->deltaX = 0;
	md->deltaY = 0;
    }

    UNWRAP (md, d, handleEvent);
    (*d->handleEvent) (d, event);
    WRAP (md, d, handleEvent, mousegesturesHandleEvent);
}

static void
loadMouseGestures(CompDisplay *d)
{
    int             nGestures, g;
    CompOptionValue *gestureVal;
    char            *gestureStr, *sep1;
    Mousegesture   *gesture;
    char *pluginName, *actionName;
    char stroke[MOUSEGESTURES_MAX_STROKE];
    int glen = 0;

    MOUSEGESTURES_DISPLAY (d);

    nGestures = md->opt[MOUSEGESTURES_DISPLAY_OPTION_ACTIONS].value.list.nValue;
    gestureVal = md->opt[MOUSEGESTURES_DISPLAY_OPTION_ACTIONS].value.list.value;

    int gnum = 0;
    while (nGestures-- && (gnum < MOUSEGESTURES_MAX_GESTURES))
    {
	gestureStr = gestureVal->s;
	pluginName = NULL;
	glen = 0;
	while (gestureStr[0]!=':')
	{
	    if (glen < MOUSEGESTURES_MAX_STROKE)
	    {
		stroke[glen] = gestureStr[0];
		glen++;
	    }

	    gestureStr++;
	}

	sep1 = strchr (gestureStr, ':');
	sep1++;
	actionName = strchr (sep1, ':');
	actionName++;
	pluginName = realloc (pluginName, sizeof(char) * (actionName - sep1));
	strncpy (pluginName, sep1, actionName - sep1 - 1);
	pluginName[actionName - sep1 - 1] = '\0';

	gesture = &md->gesture[gnum];
	for (g=0;g<glen;g++)
	    gesture->stroke[g] = stroke[g];
	gesture->nStrokes = glen;
	gesture->pluginName = pluginName;
	gesture->actionName = actionName;

	gnum++;
	gestureVal++;
    }

    md->nGesture = gnum;
}

static CompOption *
mousegesturesGetDisplayOptions (CompDisplay *d, int *count)
{
    MOUSEGESTURES_DISPLAY (d);

    *count = NUM_OPTIONS (md);
    return md->opt;
}

static Bool
mousegesturesSetDisplayOption (CompDisplay *d, char *name, CompOptionValue *value)
{
    CompOption *o;
    int         index;

    MOUSEGESTURES_DISPLAY (d);

    o = compFindOption (md->opt, NUM_OPTIONS (md), name, &index);
    if (!o)
        return FALSE;

    switch (index) 
    {

        case MOUSEGESTURES_DISPLAY_OPTION_INITIATE:
            if (setDisplayAction (d, o, value))
	        return TRUE;
            break;
        case MOUSEGESTURES_DISPLAY_OPTION_ACTIONS:
            if (compSetOptionList (o, value))
	    {
		loadMouseGestures(d);
	        return TRUE;
	    }
            break;
    
        default:
            break;
    }

    return FALSE;
}

static void
mousegesturesDisplayInitOptions (MousegesturesDisplay *md, Display *d)
{
    CompOption *o;
    int i;

    o = &md->opt[MOUSEGESTURES_DISPLAY_OPTION_INITIATE];
    o->name			        = "initiate";
    o->shortDesc		        = N_("Initiate");
    o->longDesc			        = N_("Initiate the gesture");
    o->type                             = CompOptionTypeAction;
    o->value.action.initiate            = mousegesturesInitiate;
    o->value.action.terminate           = mousegesturesTerminate;
    o->value.action.bell                = FALSE;
    o->value.action.edgeMask            = 0;
    o->value.action.state               = CompActionStateInitButton;
    o->value.action.type                = CompBindingTypeButton;
    o->value.action.button.modifiers    = MOUSEGESTURES_INITIATE_MODIFIERS_DEFAULT;
    o->value.action.button.button       = MOUSEGESTURES_INITIATE_BUTTON_DEFAULT;

    o = &md->opt[MOUSEGESTURES_DISPLAY_OPTION_ACTIONS];
    o->name	         = "actions";
    o->shortDesc         = N_("List of mouse gestures");
    o->longDesc	         = N_("List of mouse gestures");
    o->type	         = CompOptionTypeList;
    o->value.list.type   = CompOptionTypeString;
    o->value.list.nValue = N_GESTURES;
    o->value.list.value  = malloc (sizeof (CompOptionValue) * N_GESTURES);
    for (i = 0; i < N_GESTURES; i++)
	o->value.list.value[i].s = strdup (defaultGestures[i]);
    o->rest.s.string     = 0;
    o->rest.s.nString    = 0;

}

static Bool
mousegesturesInitDisplay (CompPlugin *p, CompDisplay *d)
{
    MousegesturesDisplay *md;

    md = malloc (sizeof (MousegesturesDisplay));
    if (!md)
        return FALSE;

    md->screenPrivateIndex = allocateScreenPrivateIndex (d);
    if (md->screenPrivateIndex < 0)
    {
        free (md);
        return FALSE;
    }

    mousegesturesDisplayInitOptions (md, d->display);

    md->active = FALSE;

    md->deltaX = 0;
    md->deltaY = 0;

    md->currentStroke = '\0';
    md->lastStroke = '\0';

    md->nGesture = 0;

    WRAP (md, d, handleEvent, mousegesturesHandleEvent);

    d->privates[displayPrivateIndex].ptr = md;

    return TRUE;
}

static void
mousegesturesFiniDisplay (CompPlugin *p, CompDisplay *d)
{
    MOUSEGESTURES_DISPLAY (d);

    freeScreenPrivateIndex (d, md->screenPrivateIndex);

    UNWRAP (md, d, handleEvent);

    free (md);
}

static Bool
mousegesturesInit (CompPlugin *p)
{
    displayPrivateIndex = allocateDisplayPrivateIndex ();
    if (displayPrivateIndex < 0)
        return FALSE;

    return TRUE;
}

static void
mousegesturesFini (CompPlugin *p)
{
    if (displayPrivateIndex >= 0)
        freeDisplayPrivateIndex (displayPrivateIndex);
}

static int
mousegesturesGetVersion (CompPlugin *plugin, int version)
{
    return ABIVERSION;
}

CompPluginVTable mousegesturesVTable = {
    "mousegestures",
    "Mouse gestures",
    "Use quick strokes of your mouse to initiate any compiz action",
    mousegesturesGetVersion,
    mousegesturesInit,
    mousegesturesFini,
    mousegesturesInitDisplay,
    mousegesturesFiniDisplay,
    0,
    0,
    0,
    0,
    mousegesturesGetDisplayOptions,
    mousegesturesSetDisplayOption,
    0,
    0,
    0,
    0,
    0,
    0
};

CompPluginVTable *
getCompPluginInfo (void)
{
    return &mousegesturesVTable;
}

