/*
 * Copyright (c) 2006 Darryll Truchan 
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this 
 * software and associated documentation files (the "Software"), to deal in the Software 
 * without restriction, including without limitation the rights to use, copy, modify, 
 * merge, publish, distribute, sublicense, and/or sell copies of the Software, and to 
 * permit persons to whom the Software is furnished to do so, subject to the following 
 * conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies 
 * or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A 
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT 
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF 
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR 
 * THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */


#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <X11/Xatom.h>
#include <X11/extensions/Xrender.h>

#include <compiz.h>

#define NEG_TOGGLE_KEY_DEFAULT              "n"
#define NEG_TOGGLE_MODIFIERS_DEFAULT        CompSuperMask

#define NEG_TOGGLE_ALL_KEY_DEFAULT          "m"
#define NEG_TOGGLE_ALL_MODIFIERS_DEFAULT    CompSuperMask

#define NEG_DISPLAY_OPTION_WINDOW_TOGGLE     0
#define NEG_DISPLAY_OPTION_SCREEN_TOGGLE     1
#define NEG_DISPLAY_OPTION_NUM               2

static int displayPrivateIndex;

typedef struct _NEGDisplay {
    int		    screenPrivateIndex;
    CompOption opt[NEG_DISPLAY_OPTION_NUM];
} NEGDisplay;


typedef struct _NEGSCreen {
    int                     windowPrivateIndex;
    DrawWindowTextureProc   drawWindowTexture;
    DamageWindowRectProc    damageWindowRect;
    PaintWindowProc         paintWindow;
    Bool                    isNeg;
} NEGScreen;

typedef struct _NEGWindow {
    Bool isNeg;
} NEGWindow;

#define GET_NEG_DISPLAY(d) ((NEGDisplay *) (d)->privates[displayPrivateIndex].ptr)
#define NEG_DISPLAY(d) NEGDisplay *nd = GET_NEG_DISPLAY (d)
#define GET_NEG_SCREEN(s, nd) ((NEGScreen *) (s)->privates[(nd)->screenPrivateIndex].ptr)
#define NEG_SCREEN(s) NEGScreen *ns = GET_NEG_SCREEN (s, GET_NEG_DISPLAY (s->display))
#define GET_NEG_WINDOW(w, ns) ((NEGWindow *) (w)->privates[(ns)->windowPrivateIndex].ptr)
#define NEG_WINDOW(w) NEGWindow *nw = GET_NEG_WINDOW  (w, GET_NEG_SCREEN  (w->screen, GET_NEG_DISPLAY (w->screen->display)))

#define NUM_OPTIONS(s) (sizeof ((s)->opt) / sizeof (CompOption))

static void
NEGToggle (CompWindow *w)
{
    NEG_WINDOW (w);
    nw->isNeg = !nw->isNeg;
    addWindowDamage (w);
}

static void
NEGToggleScreen (CompScreen *s)
{
    CompWindow *w;
    NEG_SCREEN (s);

    ns->isNeg = !ns->isNeg;
    for (w = s->windows; w; w = w->next)
        if (w) NEGToggle (w);
}

static Bool
negToggle (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompWindow *w;
    Window xid;

    xid = getIntOptionNamed (option, nOption, "window", 0);

    w = findWindowAtDisplay (d, xid);

    if (w)
        NEGToggle (w);

    return TRUE;
}

static Bool
negToggleAll (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompScreen *s;
    Window xid;

    xid = getIntOptionNamed (option, nOption, "root", 0);

    s = findScreenAtDisplay (d, xid);

    if (s)
        NEGToggleScreen (s);

    return TRUE;
}

static void
NEGDrawWindowTexture (CompWindow		*w,
  		      CompTexture		*texture,
		      const WindowPaintAttrib   *attrib,
		      unsigned int		mask)
{
    int filter;

    NEG_SCREEN (w->screen);
    NEG_WINDOW (w);

    if (nw->isNeg && (texture->name == w->texture->name))
    {
        glPushMatrix ();

        if (mask & PAINT_WINDOW_TRANSFORMED_MASK)
        {
	    glTranslatef (w->attrib.x, w->attrib.y, 0.0f);
	    glScalef (attrib->xScale, attrib->yScale, 0.0f);
            glTranslatef (-w->attrib.x, -w->attrib.y, 0.0f);

            filter = w->screen->filter[WINDOW_TRANS_FILTER];
        }
        else if (mask & PAINT_WINDOW_ON_TRANSFORMED_SCREEN_MASK)
        {
            filter = w->screen->filter[SCREEN_TRANS_FILTER];
        }
        else
        {
            filter = w->screen->filter[NOTHING_TRANS_FILTER];
        }


        if (w->screen->canDoSaturated && attrib->saturation != COLOR)
        {
            GLfloat constant[4];

            if (mask & PAINT_WINDOW_TRANSLUCENT_MASK)
                glEnable (GL_BLEND);

            enableTexture (w->screen, texture, filter);

            glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE);

            glTexEnvf (GL_TEXTURE_ENV, GL_COMBINE_RGB, GL_INTERPOLATE);
            glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE0_RGB, GL_TEXTURE);
            glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE1_RGB, GL_PRIMARY_COLOR);
            glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE2_RGB, GL_PRIMARY_COLOR);
            glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND0_RGB, GL_ONE_MINUS_SRC_COLOR);
            glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND1_RGB, GL_SRC_COLOR);
            glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND2_RGB, GL_SRC_ALPHA);

            glTexEnvf (GL_TEXTURE_ENV, GL_COMBINE_ALPHA, GL_REPLACE);
            glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE0_ALPHA, GL_TEXTURE);
            glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND0_ALPHA, GL_SRC_ALPHA);

            glColor4f (1.0f, 1.0f, 1.0f, 0.5f);

            w->screen->activeTexture (GL_TEXTURE1_ARB);

            enableTexture (w->screen, texture, filter);

            glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE);

            glTexEnvf (GL_TEXTURE_ENV, GL_COMBINE_RGB, GL_DOT3_RGB);
            glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE0_RGB, GL_PREVIOUS);
            glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE1_RGB, GL_CONSTANT);
            glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND0_RGB, GL_SRC_COLOR);
            glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND1_RGB, GL_SRC_COLOR);

            if (w->screen->canDoSlightlySaturated && attrib->saturation > 0)
            {
                glTexEnvf (GL_TEXTURE_ENV, GL_COMBINE_ALPHA, GL_REPLACE);
                glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE0_ALPHA, GL_PREVIOUS);
                glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND0_ALPHA, GL_SRC_ALPHA);

                constant[0] = 0.5f + 0.5f * RED_SATURATION_WEIGHT;
                constant[1] = 0.5f + 0.5f * GREEN_SATURATION_WEIGHT;
                constant[2] = 0.5f + 0.5f * BLUE_SATURATION_WEIGHT;
                constant[3] = 1.0;

                glTexEnvfv (GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, constant);

                w->screen->activeTexture (GL_TEXTURE2_ARB);

                enableTexture (w->screen, texture, filter);

                glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE);

                glTexEnvf (GL_TEXTURE_ENV, GL_COMBINE_RGB, GL_INTERPOLATE);
                glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE0_RGB, GL_TEXTURE0);
                glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE1_RGB, GL_PREVIOUS);
                glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE2_RGB, GL_CONSTANT);
                glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND0_RGB, GL_ONE_MINUS_SRC_COLOR);
                glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND1_RGB, GL_SRC_COLOR);
                glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND2_RGB, GL_SRC_ALPHA);

                glTexEnvf (GL_TEXTURE_ENV, GL_COMBINE_ALPHA, GL_REPLACE);
                glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE0_ALPHA, GL_PREVIOUS);
                glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND0_ALPHA, GL_SRC_ALPHA);

                constant[3] = attrib->saturation / 65535.0f;

                glTexEnvfv (GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, constant);

                if (attrib->opacity < OPAQUE || attrib->brightness != BRIGHT)
                {
                    w->screen->activeTexture (GL_TEXTURE3_ARB);

                    enableTexture (w->screen, texture, filter);

                    constant[3] = attrib->opacity / 65535.0f;
                    constant[0] = constant[1] = constant[2] = constant[3] * attrib->brightness / 65535.0f;

                    glTexEnvfv (GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, constant);

                    glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE);

                    glTexEnvf (GL_TEXTURE_ENV, GL_COMBINE_RGB, GL_MODULATE);
                    glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE0_RGB, GL_PREVIOUS);
                    glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE1_RGB, GL_CONSTANT);
                    glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND0_RGB, GL_SRC_COLOR);
                    glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND1_RGB, GL_SRC_COLOR);

                    glTexEnvf (GL_TEXTURE_ENV, GL_COMBINE_ALPHA, GL_MODULATE);
                    glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE0_ALPHA, GL_PREVIOUS);
                    glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE1_ALPHA, GL_CONSTANT);
                    glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND0_ALPHA, GL_SRC_ALPHA);
                    glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND1_ALPHA, GL_SRC_ALPHA);

                    (*w->screen->drawWindowGeometry) (w);

                    disableTexture (w->screen, texture);

                    glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

                    w->screen->activeTexture (GL_TEXTURE2_ARB);
                }
                else
                {
                    (*w->screen->drawWindowGeometry) (w);
                }

                disableTexture (w->screen, texture);

                glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

                w->screen->activeTexture (GL_TEXTURE1_ARB);
            }
            else
            {
                glTexEnvf (GL_TEXTURE_ENV, GL_COMBINE_ALPHA, GL_MODULATE);
                glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE0_ALPHA, GL_PREVIOUS);
                glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE1_ALPHA, GL_CONSTANT);
                glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND0_ALPHA, GL_SRC_ALPHA);
                glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND1_ALPHA, GL_SRC_ALPHA);

                constant[3] = attrib->opacity / 65535.0f;
                constant[0] = constant[1] = constant[2] = constant[3] * attrib->brightness / 65535.0f;

                constant[0] = 0.5f + 0.5f * RED_SATURATION_WEIGHT   * constant[0];
                constant[1] = 0.5f + 0.5f * GREEN_SATURATION_WEIGHT * constant[1];
                constant[2] = 0.5f + 0.5f * BLUE_SATURATION_WEIGHT  * constant[2];

                glTexEnvfv (GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, constant);

                (*w->screen->drawWindowGeometry) (w);
            }

            disableTexture (w->screen, texture);

            glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

            w->screen->activeTexture (GL_TEXTURE0_ARB);

            disableTexture (w->screen, texture);

            glColor4usv (defaultColor);
            screenTexEnvMode (w->screen, GL_REPLACE);

            if (mask & PAINT_WINDOW_TRANSLUCENT_MASK)
                glDisable (GL_BLEND);
        }
        else
        {
            enableTexture (w->screen, texture, filter);

            glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE);
            glTexEnvf (GL_TEXTURE_ENV, GL_COMBINE_RGB, GL_REPLACE);
            glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE0_RGB, GL_TEXTURE);
            glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND0_RGB, GL_ONE_MINUS_SRC_COLOR);

            if (attrib->opacity < OPAQUE || attrib->brightness != BRIGHT)
            {
                GLfloat constant[4];

                glEnable (GL_BLEND);

                constant[3] = attrib->opacity / 65535.0f;
                constant[0] = constant[3] * attrib->brightness / 65535.0f;
                constant[1] = constant[3] * attrib->brightness / 65535.0f;
                constant[2] = constant[3] * attrib->brightness / 65535.0f;
                glTexEnvfv (GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, constant);

                glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE);
                glTexEnvf (GL_TEXTURE_ENV, GL_COMBINE_RGB, GL_MODULATE);
                glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE0_RGB, GL_TEXTURE);
                glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE1_RGB, GL_CONSTANT);
                glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND0_RGB, GL_ONE_MINUS_SRC_COLOR);
                glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND1_RGB, GL_SRC_COLOR);

                glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE);
                glTexEnvf (GL_TEXTURE_ENV, GL_COMBINE_ALPHA, GL_MODULATE);
                glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE0_ALPHA, GL_TEXTURE);
                glTexEnvf (GL_TEXTURE_ENV, GL_SOURCE1_ALPHA, GL_CONSTANT);
                glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND0_ALPHA, GL_SRC_ALPHA);
                glTexEnvf (GL_TEXTURE_ENV, GL_OPERAND1_ALPHA, GL_SRC_ALPHA);

                (*w->screen->drawWindowGeometry) (w);

                glDisable (GL_BLEND);
                
            }
            else
            {
                (*w->screen->drawWindowGeometry) (w);
            }

            disableTexture (w->screen, texture);
            screenTexEnvMode (w->screen, GL_REPLACE);
        }

        glPopMatrix ();
    }
    else    
    {
        UNWRAP (ns, w->screen, drawWindowTexture);
        (*w->screen->drawWindowTexture) (w, texture, attrib, mask);
        WRAP (ns, w->screen, drawWindowTexture, NEGDrawWindowTexture);
    }
}
static Bool 
NEGDamageWindowRect (CompWindow *w, Bool initial, BoxPtr rect)
{
    int status;

    NEG_SCREEN (w->screen);
    NEG_WINDOW (w);

    if (initial)
    {
        if (ns->isNeg && !nw->isNeg)
            NEGToggle (w);
    }

    UNWRAP (ns, w->screen, damageWindowRect);
    status = (*w->screen->damageWindowRect) (w, initial, rect);
    WRAP (ns, w->screen, damageWindowRect, NEGDamageWindowRect);

    return status;
}

static Bool
NEGPaintWindow (CompWindow *w, const WindowPaintAttrib *attrib, Region region, unsigned int mask)
{
    int status;

    NEG_SCREEN (w->screen);

    UNWRAP (ns, w->screen, paintWindow);
    status = (*w->screen->paintWindow) (w, attrib, region, mask);
    WRAP (ns, w->screen, paintWindow, NEGPaintWindow);

    return status;
}

static CompOption *
NEGGetDisplayOptions (CompDisplay *display, int *count)
{
    NEG_DISPLAY (display);

    *count = NUM_OPTIONS (nd);
    return nd->opt;
}

static Bool
NEGSetDisplayOption (CompDisplay *display, char *name, CompOptionValue *value)
{
    CompOption *o;
    int	       index;

    NEG_DISPLAY (display);

    o = compFindOption (nd->opt, NUM_OPTIONS (nd), name, &index);
    if (!o)
	return FALSE;

    switch (index) {
        case NEG_DISPLAY_OPTION_WINDOW_TOGGLE:
        case NEG_DISPLAY_OPTION_SCREEN_TOGGLE:
            if (setDisplayAction (display, o, value))
                return TRUE;
	    break;
 
        default:
            break;
    }
    return FALSE;
}

static void
NEGDisplayInitOptions (NEGDisplay *nd, Display *display)
{
    CompOption *o;

    o = &nd->opt[NEG_DISPLAY_OPTION_WINDOW_TOGGLE];
    o->name                       = "window_toggle";
    o->shortDesc                  = N_("Toggle Negative");
    o->longDesc                   = N_("Toggle Window Negative");
    o->type                       = CompOptionTypeAction;
    o->value.action.initiate      = negToggle;
    o->value.action.terminate     = 0;
    o->value.action.bell          = FALSE;
    o->value.action.edgeMask      = 0;
    o->value.action.type          = CompBindingTypeKey;
    o->value.action.state         = CompActionStateInitKey;
    o->value.action.state        |= CompActionStateInitButton;
    o->value.action.key.modifiers = NEG_TOGGLE_MODIFIERS_DEFAULT;
    o->value.action.key.keycode   = XKeysymToKeycode (display, XStringToKeysym (NEG_TOGGLE_KEY_DEFAULT));

    o = &nd->opt[NEG_DISPLAY_OPTION_SCREEN_TOGGLE];
    o->name                       = "screen_toggle";
    o->shortDesc                  = N_("Toggle Negative");
    o->longDesc                   = N_("Toggle Screen Negative");
    o->type                       = CompOptionTypeAction;
    o->value.action.initiate      = negToggleAll;
    o->value.action.terminate     = 0;
    o->value.action.bell          = FALSE;
    o->value.action.edgeMask      = 0;
    o->value.action.type          = CompBindingTypeKey;
    o->value.action.state         = CompActionStateInitKey;
    o->value.action.state        |= CompActionStateInitButton;
    o->value.action.key.modifiers = NEG_TOGGLE_ALL_MODIFIERS_DEFAULT;
    o->value.action.key.keycode   = XKeysymToKeycode (display, XStringToKeysym (NEG_TOGGLE_ALL_KEY_DEFAULT));
}

static Bool
NEGInitDisplay (CompPlugin  *p, CompDisplay *d)
{
    NEGDisplay *nd;

    nd = malloc (sizeof (NEGDisplay));
    if (!nd)
	return FALSE;

    nd->screenPrivateIndex = allocateScreenPrivateIndex (d);
    if (nd->screenPrivateIndex < 0)
    {
	free (nd);
	return FALSE;
    }

    NEGDisplayInitOptions (nd, d->display);

    d->privates[displayPrivateIndex].ptr = nd;

    return TRUE;
}

static void
NEGFiniDisplay (CompPlugin  *p, CompDisplay *d)
{
    NEG_DISPLAY (d);
    freeScreenPrivateIndex (d, nd->screenPrivateIndex);
    free (nd);
}

static Bool
NEGInitScreen (CompPlugin *p, CompScreen *s)
{
    NEGScreen *ns;
    NEG_DISPLAY (s->display);

    ns = malloc (sizeof (NEGScreen));
    if (!ns)
	return FALSE;

    ns->windowPrivateIndex = allocateWindowPrivateIndex (s);
    if (ns->windowPrivateIndex < 0)
    {
	free (ns);
	return FALSE;
    }

    ns->isNeg = FALSE;

    addScreenAction (s, &nd->opt[NEG_DISPLAY_OPTION_WINDOW_TOGGLE].value.action);
    addScreenAction (s, &nd->opt[NEG_DISPLAY_OPTION_SCREEN_TOGGLE].value.action);
  
    WRAP (ns, s, drawWindowTexture, NEGDrawWindowTexture);
    WRAP (ns, s, damageWindowRect, NEGDamageWindowRect);
    WRAP (ns, s, paintWindow, NEGPaintWindow);

    s->privates[nd->screenPrivateIndex].ptr = ns;

    return TRUE;
}


static void
NEGFiniScreen (CompPlugin *p, CompScreen *s)
{
    NEG_SCREEN (s);
    freeWindowPrivateIndex (s, ns->windowPrivateIndex);
    UNWRAP (ns, s, drawWindowTexture);
    UNWRAP (ns, s, damageWindowRect);
    UNWRAP (ns, s, paintWindow);
    free (ns);
}

static Bool
NEGInitWindow (CompPlugin *p, CompWindow *w)
{
    NEGWindow *nw;

    NEG_SCREEN (w->screen);

    nw = malloc (sizeof (NEGWindow));
    if (!nw)
	return FALSE;

    nw->isNeg = FALSE;

    w->privates[ns->windowPrivateIndex].ptr = nw;

    return TRUE;
}

static void
NEGFiniWindow (CompPlugin *p, CompWindow *w)
{
    NEG_WINDOW (w);
    free (nw);
}

static Bool
NEGInit (CompPlugin *p)
{
    displayPrivateIndex = allocateDisplayPrivateIndex ();
    if (displayPrivateIndex < 0)
	return FALSE;

    return TRUE;
}

static void
NEGFini (CompPlugin *p)
{
    if (displayPrivateIndex >= 0)
	freeDisplayPrivateIndex (displayPrivateIndex);
}

static int
negGetVersion (CompPlugin *plugin, int version)
{
    return ABIVERSION;
}

CompPluginVTable NEGVTable = {
    "neg",
    N_("Negative"),
    N_("Negative"),
    negGetVersion,
    NEGInit,
    NEGFini,
    NEGInitDisplay,
    NEGFiniDisplay,
    NEGInitScreen,
    NEGFiniScreen,
    NEGInitWindow,
    NEGFiniWindow,
    NEGGetDisplayOptions,
    NEGSetDisplayOption,
    0, /* GetScreenOptions */
    0, /* SetScreenOption */
    NULL,
    0,
    0,
    0
};

CompPluginVTable *
getCompPluginInfo (void)
{
    return &NEGVTable;
}


