#
# Copyright (C) 2005  Robert Collins  <robertc@squid-cache.org>
# 
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import tests
import unittest

# if this fails to import, then the config_manager module is bust
# and this test script will never be reached to be imported
import config_manager
from config_manager.tests import FunctionalTestCase

import os
from StringIO import StringIO

class TestConfig(unittest.TestCase):
    def test_imports(self):
        from config_manager import Config

    def test_default_init(self):
        config_manager.Config()

    def test_default_entries(self):
        self.assertEqual({}, config_manager.Config().get_entries())

    def test_add_entry(self):
        config = config_manager.Config()
        entry = config_manager.ConfigEntry("./path", "url")
        config.add_entry(entry)
        self.assertEqual(config.get_entries(), {"./path": entry})

    def test_new_from_stream(self):
        stream = StringIO("./project    arch://foo\n"
                          "./project/lib    file:///foo/bar\n")
        entry1 = config_manager.ConfigEntry("./project", "arch://foo")
        entry2 = config_manager.ConfigEntry("./project/lib", "file:///foo/bar")
        config = config_manager.Config(fromStream=stream)
        self.assertEqual(config.get_entries(), {"./project": entry1,
                                                "./project/lib": entry2})

    def test_new_from_stream_with_comments(self):
        stream = StringIO("#./project    arch://foo\n"
                          "./project/lib    file:///foo/bar\n"
                          "\n"
                          "# a config\n"
                          )
        entry = config_manager.ConfigEntry("./project/lib", "file:///foo/bar")
        config = config_manager.Config(fromStream=stream)
        self.assertEqual(config.get_entries(), {"./project/lib": entry})

class FunctionalTestConfig(FunctionalTestCase):

    def test_build_fake(self):
        """Test that doing a build works."""
        config = config_manager.Config()
        config.add_entry (config_manager.ConfigEntry("nested", 
            "fake://anarchive@example.com/foo--0"))
        config.build(self.tempdir)

        from config_manager import fake_builds
        self.assertEqual(fake_builds, [("fake://anarchive@example.com/foo--0",
                                        os.path.join(
                                            self.tempdir,
                                            "nested"))])
        
    def test_update_fake(self):
        """Test that doing an update."""
        config = config_manager.Config()
        config.add_entry (config_manager.ConfigEntry("nested", 
            "fake://anarchive@example.com/foo--0"))
        config.build(self.tempdir)
        config.update(self.tempdir)

        from config_manager import fake_builds, fake_updates
        self.assertEqual(fake_updates, [("fake://anarchive@example.com/foo--0",
                                         os.path.join(
                                            self.tempdir,
                                            "nested"))])


def test_suite():
    loader = tests.TestLoader()
    result = loader.loadTestsFromName(__name__)
    return result
