/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */

/*
 * plugin-dtd.c
 *
 * Plugin for manipulating DTD files
 *
 * Copyright (C) 2003 David Malcolm
 *
 * Conglomerate is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * Conglomerate is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Authors: David Malcolm <david@davemalcolm.demon.co.uk>
 */

#include "global.h"
#include "cong-plugin.h"
#include "cong-error-dialog.h"
#include "cong-dispspec.h"

#include "cong-fake-plugin-hooks.h"

#include "cong-vfs.h"
#include "cong-util.h"
#include "cong-dtd.h"

#define RELAX_NG_NS_URI ("http://relaxng.org/ns/structure/1.0")

/* Internal function declarations: */
static xmlDtdPtr 
load_dtd (const gchar *uri, 
	  GtkWindow *toplevel_window);

static xmlDocPtr
make_rng_from_dtd (xmlDtdPtr dtd,
		   GList *list_of_start_elements);

static void
element_callback_generate_rng_from_dtd (xmlElementPtr dtd_element,
					gpointer user_data);

static void
attribute_callback_generate_rng_from_dtd (xmlElementPtr dtd_element,
					  xmlAttributePtr attr,
					  gpointer user_data);

/* Plugin hooks: */
static GtkFileFilter*
dtd_importer_filter_factory_callback (CongServiceImporter *importer)
{
	GtkFileFilter *filter;

	g_return_val_if_fail (importer, NULL);

	filter = cong_service_importer_make_basic_filter (importer);

	gtk_file_filter_add_mime_type (filter, "text/x-dtd");

	return filter;
}

/**
 * dtd_to_xds_importer_action_callback:
 * @importer:
 * @uri:
 * @mime_type:
 * @user_data:
 * @toplevel_window:
 *
 * TODO: Write me
 */
void 
dtd_to_xds_importer_action_callback(CongServiceImporter *importer, const gchar *uri, const gchar *mime_type, gpointer user_data, GtkWindow *toplevel_window)
{
	xmlDtdPtr dtd;

	g_message("dtd_to_xds_importer_action_callback");

	dtd = load_dtd(uri, toplevel_window);

	if (dtd) {
		gchar *name = g_strdup_printf(_("Autogenerated document type based on %s"), uri);
		gchar *description = g_strdup_printf(_("This Conglomerate display specification was automatically generated from %s"), uri);

		CongDispspec *dispspec = cong_dispspec_new_generate_from_dtd(dtd, name, description);
		xmlDocPtr xml_doc = cong_dispspec_make_xml(dispspec);

		g_free(name);
		g_free(description);

		/* Free up the temporary dispspec: */
		cong_dispspec_delete(dispspec);

		/* Free up the DTD: */
		xmlFreeDtd(dtd);

		/* Do appropriate UI stuff: */
		cong_ui_new_document_from_imported_xml(xml_doc,
						       toplevel_window);
	}
}

/**
 * dtd_to_rng_importer_action_callback:
 * @importer:
 * @uri:
 * @mime_type:
 * @user_data:
 * @toplevel_window:
 *
 * TODO: Write me
 */
void 
dtd_to_rng_importer_action_callback(CongServiceImporter *importer, const gchar *uri, const gchar *mime_type, gpointer user_data, GtkWindow *toplevel_window)
{
	xmlDtdPtr dtd;

	g_message("dtd_to_rng_importer_action_callback");

	dtd = load_dtd(uri, toplevel_window);

	if (dtd) {
		GList *list_of_start_elements = cong_dtd_guess_start_elements (dtd);
		xmlDocPtr xml_doc = make_rng_from_dtd (dtd,
						       list_of_start_elements);

		/* Free up the DTD: */
		xmlFreeDtd(dtd);

		/* Do appropriate UI stuff: */
		cong_ui_new_document_from_imported_xml(xml_doc,
						       toplevel_window);
	}
}

/**
 * dtd_to_w3c_xml_schema_importer_action_callback:
 * @importer:
 * @uri:
 * @mime_type:
 * @user_data:
 * @toplevel_window:
 *
 * TODO: Write me
 */
void 
dtd_to_w3c_xml_schema_importer_action_callback(CongServiceImporter *importer, const gchar *uri, const gchar *mime_type, gpointer user_data, GtkWindow *toplevel_window)
{
	g_message("dtd_to_w3c_xml_schema_importer_action_callback");

	CONG_DO_UNIMPLEMENTED_DIALOG(toplevel_window, "Importing DTD as W3C XML Schema");
}

/**
 * dtd_to_schematron_importer_action_callback:
 * @importer:
 * @uri:
 * @mime_type:
 * @user_data:
 * @toplevel_window:
 *
 * TODO: Write me
 */
void 
dtd_to_schematron_importer_action_callback(CongServiceImporter *importer, const gchar *uri, const gchar *mime_type, gpointer user_data, GtkWindow *toplevel_window)
{
	g_message("dtd_to_schematron_importer_action_callback");

	CONG_DO_UNIMPLEMENTED_DIALOG(toplevel_window, "Importing DTD as Schematron Schema");
}

/**
 * dtd_to_examplotron_importer_action_callback:
 * @importer:
 * @uri:
 * @mime_type:
 * @user_data:
 * @toplevel_window:
 *
 * TODO: Write me
 */
void 
dtd_to_examplotron_importer_action_callback(CongServiceImporter *importer, const gchar *uri, const gchar *mime_type, gpointer user_data, GtkWindow *toplevel_window)
{
	g_message("dtd_to_examplotron_importer_action_callback");

	CONG_DO_UNIMPLEMENTED_DIALOG(toplevel_window, "Importing DTD as Examplotron Schema");
}

/* would be exposed as "plugin_register"? */
/**
 * plugin_dtd_plugin_register:
 * @plugin:
 *
 * TODO: Write me
 * Returns:
 */
gboolean 
plugin_dtd_plugin_register(CongPlugin *plugin)
{
	g_return_val_if_fail(plugin, FALSE);

	cong_plugin_register_importer(plugin, 
				      _("Convert DTD into a Conglomerate Display Specification"), 
				      _("Import a DTD file, creating a Conglomerate display specification file."),
				      "dtd-to-xds-import",
				      dtd_importer_filter_factory_callback,
				      NULL,
				      dtd_to_xds_importer_action_callback,
				      NULL);

	cong_plugin_register_importer(plugin, 
				      _("Convert DTD into a Relax NG schema"), 
				      _("Import a DTD file, converting it into a RELAX NG Schema."),
				      "dtd-to-rng-import",
				      dtd_importer_filter_factory_callback,
				      NULL,
				      dtd_to_rng_importer_action_callback,
				      NULL);

	cong_plugin_register_importer(plugin, 
				      _("Convert DTD into W3C XML Schema"), 
				      _("Import a DTD file, converting it into a W3C XML Schema."),
				      "dtd-to-w3c-xml-schema-import",
				      dtd_importer_filter_factory_callback,
				      NULL,
				      dtd_to_w3c_xml_schema_importer_action_callback,
				      NULL);
	
	cong_plugin_register_importer(plugin, 
				      _("Convert DTD into a Schematron file"), 
				      _("Import a DTD file, converting it into a Schematron Schema."),
				      "dtd-to-schematron-import",
				      dtd_importer_filter_factory_callback,
				      NULL,
				      dtd_to_schematron_importer_action_callback,
				      NULL);
	
	cong_plugin_register_importer(plugin, 
				      _("Convert DTD into an Examplotron file"), 
				      _("Import a DTD file, converting it into an Examplotron Schema."),
				      "dtd-to-examplotron-import",
				      dtd_importer_filter_factory_callback,
				      NULL,
				      dtd_to_examplotron_importer_action_callback,
				      NULL);
	
	return TRUE;
}

/* exposed as "plugin_configure"? legitimate for it not to be present */
/**
 * plugin_dtd_plugin_configure:
 * @plugin:
 *
 * TODO: Write me
 * Returns:
 */
gboolean 
plugin_dtd_plugin_configure(CongPlugin *plugin)
{
	g_return_val_if_fail(plugin, FALSE);

	return TRUE;
}

/* Internal function definitions: */
static xmlDtdPtr 
load_dtd (const gchar *uri, 
	  GtkWindow *toplevel_window)
{
	xmlDtdPtr dtd;
	GnomeVFSURI *vfs_uri;
	gchar *local_path;

	g_return_val_if_fail(uri, NULL);

	vfs_uri = gnome_vfs_uri_new(uri);
	local_path = cong_vfs_get_local_path_from_uri (vfs_uri);
	gnome_vfs_uri_unref(vfs_uri);

	dtd = xmlIOParseDTD(NULL, 
			    xmlParserInputBufferCreateFilename	(local_path,
								 XML_CHAR_ENCODING_NONE),
			    XML_CHAR_ENCODING_NONE);

	return dtd;
}

/* Data stored by the DTD->RNG converter: */
typedef struct CongDTD2RNGConverter CongDTD2RNGConverter;
struct CongDTD2RNGConverter
{
	xmlDtdPtr dtd;
	GList *list_of_start_elements;

	xmlDocPtr xml_doc;
	xmlNodePtr grammar_node;
	xmlNsPtr xml_ns;
	xmlNodePtr start_node;
};

CongDTD2RNGConverter*
cong_dtd2rng_converter_new (xmlDtdPtr dtd,
			    GList *list_of_start_elements);

void
cong_dtd2rng_converter_free (CongDTD2RNGConverter *converter);

gboolean
cong_dtd2rng_converter_is_start_element (CongDTD2RNGConverter *converter,
					 xmlElementPtr dtd_element);

guint
cong_dtd2rng_converter_get_element_ref_count (CongDTD2RNGConverter *converter,
					      xmlElementPtr dtd_element);

gboolean
cong_dtd2rng_converter_should_element_have_define (CongDTD2RNGConverter *converter,
						   xmlElementPtr dtd_element);

xmlDocPtr
cong_dtd2rng_converter_make_schema (CongDTD2RNGConverter *converter);

void
cong_dtd2rng_converter_add_element_or_ref (CongDTD2RNGConverter *converter,
					   xmlElementPtr dtd_element,
					   xmlNodePtr parent_node);

void
cong_dtd2rng_converter_add_rng_element_ref (CongDTD2RNGConverter *converter,
					    xmlElementPtr dtd_element,
					    xmlNodePtr parent_node);

void
cong_dtd2rng_converter_add_rng_element_inline (CongDTD2RNGConverter *converter,
					       xmlElementPtr dtd_element,
					       xmlNodePtr parent_node);

static void
add_content_subtree_to_rng (CongDTD2RNGConverter *converter,
			    CongNodePtr node_parent,
			    xmlElementContentPtr content);


/****/

static xmlDocPtr
make_rng_from_dtd (xmlDtdPtr dtd,
		   GList *list_of_root_elements)
{
	xmlDocPtr result;
	CongDTD2RNGConverter* converter = cong_dtd2rng_converter_new (dtd,
								      list_of_root_elements);

	result = cong_dtd2rng_converter_make_schema (converter);
	
	cong_dtd2rng_converter_free (converter);

	return result;
}



/* Implementation of CongDTD2RNGConverter: */

/**
 * cong_dtd2rng_converter_new:
 * @dtd:
 * @list_of_start_elements:
 *
 * TODO: Write me
 */
CongDTD2RNGConverter*
cong_dtd2rng_converter_new (xmlDtdPtr dtd,
			    GList *list_of_start_elements)
{
	CongDTD2RNGConverter *converter;

	g_return_val_if_fail (dtd, NULL);
	g_return_val_if_fail (list_of_start_elements, NULL);

	converter = g_new0 (CongDTD2RNGConverter, 1);

	converter->dtd = dtd;
	converter->list_of_start_elements = list_of_start_elements;

	return converter;
}

/**
 * cong_dtd2rng_converter_free:
 * @converter: a #CongDTD2RNGConverter
 *
 * Calls g_free on @converter
 */
void
cong_dtd2rng_converter_free (CongDTD2RNGConverter *converter)
{
	g_free (converter);
}

/**
 * cong_dtd2rng_converter_is_start_element:
 * @converter:
 * @dtd_element:
 *
 * TODO: Write me
 */
gboolean
cong_dtd2rng_converter_is_start_element (CongDTD2RNGConverter *converter,
					 xmlElementPtr dtd_element)
{
	GList *iter;

	g_return_val_if_fail (converter, FALSE);
	g_return_val_if_fail (dtd_element, FALSE);

	for (iter=converter->list_of_start_elements; iter; iter=iter->next) {
		if (iter->data==dtd_element) {
			return TRUE;
		}
	}

	return FALSE;
}

/*
  Count references; they can happen in the main DTD, or in the <start> element (if any):
*/
/**
 * cong_dtd2rng_converter_get_element_ref_count:
 * @converter:
 * @dtd_element:
 *
 * TODO: Write me
 */
guint
cong_dtd2rng_converter_get_element_ref_count (CongDTD2RNGConverter *converter,
					      xmlElementPtr dtd_element)
{
	guint num_element_references;

	g_return_val_if_fail (converter, 0);
	g_return_val_if_fail (dtd_element, 0);

	num_element_references = cong_dtd_count_references_to_element (converter->dtd,
									     dtd_element);
	if (cong_dtd2rng_converter_is_start_element (converter,
						     dtd_element)) {
		if (dtd_element->name) { g_message ("%s is start element", dtd_element->name); }
		num_element_references++;
	}

	if (dtd_element->name) { g_message ("%s has %i refs", dtd_element->name, num_element_references); }

	return num_element_references;
}


/* 
   Elements need to be wrapped with a define if they are referenced more than once (or if the
   user has requested a flat schema?)
*/
/**
 * cong_dtd2rng_converter_should_element_have_define:
 * @converter:
 * @dtd_element:
 *
 * TODO: Write me
 */
gboolean
cong_dtd2rng_converter_should_element_have_define (CongDTD2RNGConverter *converter,
						   xmlElementPtr dtd_element)
{
	guint ref_count;

	g_return_val_if_fail (converter, FALSE);
	g_return_val_if_fail (dtd_element, FALSE);

	ref_count = cong_dtd2rng_converter_get_element_ref_count (converter,
								  dtd_element);

	if (ref_count>1) {
		return TRUE;
	} else {
		return FALSE;
	}
}

/**
 * cong_dtd2rng_converter_make_schema:
 * @converter:
 *
 * TODO: Write me
 */
xmlDocPtr
cong_dtd2rng_converter_make_schema (CongDTD2RNGConverter *converter)
{
	xmlNodePtr choice_node;

	g_return_val_if_fail (converter, NULL);


	/* Build up the document and its content: */
	converter->xml_doc = xmlNewDoc((const xmlChar*)"1.0");
	
	converter->grammar_node = xmlNewDocNode (converter->xml_doc,
						 NULL, /* xmlNsPtr ns, */
						 (const xmlChar*)"grammar",
						 NULL);
	converter->xml_ns = xmlNewNs (converter->grammar_node, 
				      (const xmlChar*)RELAX_NG_NS_URI, 
				      NULL);
	xmlSetNs (converter->grammar_node, 
		  converter->xml_ns);
	
	xmlDocSetRootElement (converter->xml_doc,
			      converter->grammar_node);

	/* The <start> element: */
	{
		GList *iter;
		converter->start_node = xmlNewDocNode (converter->xml_doc,
						       converter->xml_ns,
						       (const xmlChar*)"start",
						       NULL);

		xmlAddChild (converter->grammar_node,
			     converter->start_node);

		/* List must be non-empty: */
		g_assert (converter->list_of_start_elements);

		/* If list contains more than one start, add a <choice> element: */
		if (converter->list_of_start_elements->next) {
			choice_node = xmlNewDocNode (converter->xml_doc,
						     converter->xml_ns,
						     (const xmlChar*)"choice",
						     NULL);
			xmlAddChild (converter->start_node,
				     choice_node);
		} else {
			choice_node = converter->start_node;
		}
		

		for (iter=converter->list_of_start_elements;iter;iter=iter->next) {
			xmlElementPtr dtd_element = (xmlElementPtr)(iter->data);

			/* Add elements either directly or by ref: */
			cong_dtd2rng_converter_add_element_or_ref (converter,
								   dtd_element,
								   choice_node);
		}
	}

	
	/* Add <define> tags for all the attributes, and for all element which appear more than once in the content model: */
	cong_dtd_for_each_element (converter->dtd,
				   element_callback_generate_rng_from_dtd,
				   converter);

	return converter->xml_doc;	

}

static void
element_reference_callback_add_comment (xmlDtdPtr dtd,
					xmlElementPtr dtd_element,
					xmlElementContentPtr content,
					gpointer user_data)
{
	CongNodePtr xml_node = (CongNodePtr)user_data;

	xmlAddChild (xml_node,
		     xmlNewDocComment (xml_node->doc,
				       (const xmlChar*)"element is cross-referenced"));
}

/* Adds the element either via a ref, or directly inline, depending on whether its referenced elsewhere or not: */
/**
 * cong_dtd2rng_converter_add_element_or_ref:
 * @converter:
 * @dtd_element:
 * @parent_node:
 *
 * TODO: Write me
 */
void
cong_dtd2rng_converter_add_element_or_ref (CongDTD2RNGConverter *converter,
					   xmlElementPtr dtd_element,
					   xmlNodePtr parent_node)
{
	if (cong_dtd2rng_converter_should_element_have_define (converter,
							       dtd_element)) {
		cong_dtd2rng_converter_add_rng_element_ref (converter,
							    dtd_element,
							    parent_node);
	} else {
		/* This is the only place where this element is used in the DTD; put it inline here for a Russian-Doll effect: */
		cong_dtd2rng_converter_add_rng_element_inline (converter,
							       dtd_element,
							       parent_node);
	}
}


/* Add a <ref name="foobar"> tag: */
/**
 * cong_dtd2rng_converter_add_rng_element_ref:
 * @converter:
 * @dtd_element:
 * @parent_node:
 *
 * TODO: Write me
 */
void
cong_dtd2rng_converter_add_rng_element_ref (CongDTD2RNGConverter *converter,
					    xmlElementPtr dtd_element,
					    xmlNodePtr parent_node)
{
	CongNodePtr node_ref = xmlNewDocNode(parent_node->doc,
					     converter->xml_ns, 
					     (const xmlChar*)"ref",
					     NULL);
	xmlAddChild (parent_node, 
		     node_ref);
	
	xmlSetProp (node_ref,
		    (const xmlChar*)"name",
		    dtd_element->name);
}

/* Create a RELAX NG <element>, either somewhere inside a content model, or inside a <define> element */
/**
 * cong_dtd2rng_converter_add_rng_element_inline:
 * @converter:
 * @dtd_element:
 * @parent_node:
 *
 * TODO: Write me
 */
void
cong_dtd2rng_converter_add_rng_element_inline (CongDTD2RNGConverter *converter,
					       xmlElementPtr dtd_element,
					       xmlNodePtr parent_node)
{
	CongNodePtr node_element = xmlNewDocNode(parent_node->doc,
						 converter->xml_ns,
						 (const xmlChar*)"element",
						 NULL);
	xmlAddChild (parent_node, 
		     node_element);
	
	xmlSetProp (node_element,
		    (const xmlChar*)"name",
		    dtd_element->name);
	
	/* set up the content model */
	{
		cong_dtd_for_each_attribute (dtd_element,
					     attribute_callback_generate_rng_from_dtd,
					     node_element);
		
		if (dtd_element->content) {
			add_content_subtree_to_rng (converter,
						    node_element,
						    dtd_element->content);
		}
	}
}

static void
element_callback_generate_rng_from_dtd (xmlElementPtr dtd_element,
					gpointer user_data)
{
	CongDTD2RNGConverter *converter = (CongDTD2RNGConverter*)user_data;

	/* Create a <define> for every element referenced more than once in the document: */
	if (cong_dtd2rng_converter_should_element_have_define (converter,
							       dtd_element)) {
		CongNodePtr node_define;

		node_define = xmlNewDocNode (converter->xml_doc,
					     converter->xml_ns,
					     (const xmlChar*)"define",
					     NULL);			
		xmlAddChild (converter->grammar_node, 
			     node_define);
		
		xmlSetProp (node_define,
			    (const xmlChar*)"name",
			    dtd_element->name);
		
		
		
		cong_dtd_for_each_reference_to_element (dtd_element->parent,
							dtd_element,
							element_reference_callback_add_comment,
							node_define);
		
		/* Create the <element> tag: */
		cong_dtd2rng_converter_add_rng_element_inline (converter,
							       dtd_element,
							       node_define);
	}
}

static void
attribute_callback_generate_rng_from_dtd (xmlElementPtr dtd_element,
					  xmlAttributePtr attr,
					  gpointer user_data)
{
	CongNodePtr node_element = (CongNodePtr)user_data;
	CongNodePtr node_occurrence = NULL;
	CongNodePtr node_attribute;
	xmlNsPtr xml_ns;

	xml_ns = xmlSearchNsByHref (node_element->doc,
				    node_element,
				    (const xmlChar*)RELAX_NG_NS_URI);
	g_assert (xml_ns);

	switch (attr->def) {
	default: g_assert_not_reached ();
	case XML_ATTRIBUTE_NONE:
	case XML_ATTRIBUTE_IMPLIED:
		/* Attribute is optional: */
		node_occurrence = xmlNewDocNode(node_element->doc,
						xml_ns,
						(const xmlChar*)"optional",
						NULL);
		
		xmlAddChild (node_element,
			     node_occurrence);
		break;
	case XML_ATTRIBUTE_REQUIRED:
	case XML_ATTRIBUTE_FIXED:
		/* Attribute is required; no need to wrap with an occurrence tag */
		node_occurrence = node_element;
		break;
	}

	g_assert (node_occurrence);

	node_attribute= xmlNewDocNode(node_element->doc,
				      xml_ns,
				      (const xmlChar*)"attribute",
				      NULL);			
	xmlAddChild (node_occurrence,
		     node_attribute);
	
	xmlSetProp (node_attribute,
		    (const xmlChar*)"name",
		    attr->name);

	/* FIXME: do we need to handle the default? */

	/* Define content model: */
	switch (attr->atype) {
	default: g_assert_not_reached ();
	case XML_ATTRIBUTE_CDATA:
		/* no need to add anything? */
		break;
	case XML_ATTRIBUTE_ID:
		/* FIXME */
		break;

	case XML_ATTRIBUTE_IDREF:
		/* FIXME */
		break;

	case XML_ATTRIBUTE_IDREFS:
		/* FIXME */
		break;

	case XML_ATTRIBUTE_ENTITY:
		/* FIXME */
		break;

	case XML_ATTRIBUTE_ENTITIES:
		/* FIXME */
		break;

	case XML_ATTRIBUTE_NMTOKEN:
		/* FIXME */
		break;

	case XML_ATTRIBUTE_NMTOKENS:
		/* FIXME */
		break;

	case XML_ATTRIBUTE_ENUMERATION:
		/* Add a <choice> tag: */
		{
			CongNodePtr node_choice = xmlNewDocNode(node_element->doc,
								xml_ns,
								(const xmlChar*)"choice",
								NULL);
			xmlAddChild (node_attribute, 
				     node_choice);

			/* Add the enum values: */
			{
				xmlEnumerationPtr iter;	

				for (iter = attr->tree; iter; iter=iter->next) {
					CongNodePtr node_value = xmlNewDocNode(node_element->doc,
									       xml_ns,
									       (const xmlChar*)"value",
									       iter->name);
					xmlAddChild (node_choice,
						     node_value);
					
				}
			}	
		}
		break;

	case XML_ATTRIBUTE_NOTATION:	
		/* FIXME */
		break;
	}
}

static void
add_content_subtree_to_rng (CongDTD2RNGConverter *converter,
			    CongNodePtr node_parent,
			    xmlElementContentPtr content)
{
	CongNodePtr node_occurrence = NULL;
	xmlNsPtr xml_ns;

	xml_ns = xmlSearchNsByHref (node_parent->doc,
				    node_parent,
				    (const xmlChar*)RELAX_NG_NS_URI);
	g_assert (xml_ns);

	switch (content->ocur) {
	default: g_assert_not_reached ();
	case XML_ELEMENT_CONTENT_ONCE:
		/* then we don't need to wrap the content with an occurrence tag: */
		node_occurrence = node_parent;
		break;

	case XML_ELEMENT_CONTENT_OPT:
		node_occurrence = xmlNewDocNode(node_parent->doc,
						xml_ns,
						(const xmlChar*)"optional",
						NULL);
		xmlAddChild (node_parent, 
			     node_occurrence);
		break;

	case XML_ELEMENT_CONTENT_MULT:
		node_occurrence = xmlNewDocNode(node_parent->doc,
						xml_ns,
						(const xmlChar*)"zeroOrMore",
						NULL);
		xmlAddChild (node_parent, 
			     node_occurrence);
		break;
		
	case XML_ELEMENT_CONTENT_PLUS:
		node_occurrence = xmlNewDocNode(node_parent->doc,
						xml_ns,
						(const xmlChar*)"oneOrMore",
						NULL);
		xmlAddChild (node_parent, 
			     node_occurrence);
		break;
	}
		
	g_assert (node_occurrence);

	switch (content->type) {
	default: g_assert_not_reached ();
	case XML_ELEMENT_CONTENT_PCDATA:
		/* Add an empty <text> tag: */
		{
			CongNodePtr node_text = xmlNewDocNode(node_occurrence->doc,
							      xml_ns,
							      (const xmlChar*)"text",
							      NULL);
			xmlAddChild (node_occurrence, 
				     node_text);
		}
		break;
	case XML_ELEMENT_CONTENT_ELEMENT:
		{
			xmlElementPtr dtd_element = cong_dtd_get_element_for_content (converter->dtd,
										      content);
			cong_dtd2rng_converter_add_element_or_ref (converter,
								   dtd_element,
								   node_occurrence);
		}
		break;
	case XML_ELEMENT_CONTENT_SEQ:
		/* Add a <group> tag, and recurse; optimise away the cae where the parent is a <group> tag: */
		{
			if (cong_node_is_element (node_occurrence, RELAX_NG_NS_URI, "group")) {
				add_content_subtree_to_rng (converter,
							    node_occurrence,
							    content->c1);
				add_content_subtree_to_rng (converter,
							    node_occurrence,
							    content->c2);
			} else {
				CongNodePtr node_group = xmlNewDocNode(node_occurrence->doc,
								       xml_ns,
								       (const xmlChar*)"group",
								       NULL);
				xmlAddChild (node_occurrence, 
					     node_group);
				
				add_content_subtree_to_rng (converter,
							    node_group,
							    content->c1);
				add_content_subtree_to_rng (converter,
							    node_group,
							    content->c2);
			}
		}
		break;
	case XML_ELEMENT_CONTENT_OR:
		/* The naive implementation is to add a <choice> tag, and recurse.
		   But if we have something like ( tag-a | tag-b | tag-c | ... | tag-z) in the DTD, we get a tree of ( choice tag-a (choice tag-b (choice tag-c ( ... choice tag-y tag-z))))
		   which creates a grim-looking RNG tree.

		   So we spot simple <choice> in the tag above, and merge into it if possible; <choice> is associative and hence bracketing should make no difference...
		 */
		{
			if (cong_node_is_element (node_occurrence, RELAX_NG_NS_URI, "choice")) {
				/* Optimised case: */
				add_content_subtree_to_rng (converter,
							    node_occurrence,
							    content->c1);
				add_content_subtree_to_rng (converter,
							    node_occurrence,
							    content->c2);
			} else {
				/* Non-optimised case: */
				CongNodePtr node_choice = xmlNewDocNode(node_occurrence->doc,
									xml_ns, 
									(const xmlChar*)"choice",
									NULL);
				xmlAddChild (node_occurrence, 
					     node_choice);
				
				add_content_subtree_to_rng (converter,
							    node_choice,
							    content->c1);
				add_content_subtree_to_rng (converter,
							    node_choice,
							    content->c2);
			}
		}
		break;
	}
}
