/*
 *  Author:       Serg Stoyan
 *
 *  Classes:      NetworkView : NSObject <PrefsViewProtocol>
 *
 *  Description:  Network preferences (IP, DNS, routing, proxyarp)
 */

#import <AppKit/NSView.h>
#import <AppKit/NSPanel.h>
#import <AppKit/NSBox.h>
#import <AppKit/NSButton.h>
#import <AppKit/NSTextField.h>

#import "Network.h"

@implementation NetworkView

- (id) initWithPrefs: (NSMutableDictionary *) sessionPrefs
{
  NSView *ipFrameView;
  NSView *dnsFrameView;
  NSView *routingFrameView;

  [super init];
  
  if ((prefs = [sessionPrefs objectForKey: @"Network"]) == nil)
  {
    [sessionPrefs 
      setObject: [NSMutableDictionary dictionary] forKey: @"Network"];
    prefs = [sessionPrefs objectForKey: @"Network"];
  }

  view = [[NSView alloc] initWithFrame: NSMakeRect (10, 44, 280, 393)];

  // IP adress -------------------------------------------------- 
  ipFrame = [[NSBox alloc] initWithFrame: NSMakeRect (0, 249, 280, 141)];
  [ipFrame setTitle: @"IP address assigning"];
  [view addSubview: ipFrame];
  ipFrameView = [ipFrame contentView];

  dynamicIP = [[NSButton alloc] initWithFrame: 
    NSMakeRect (105, 99, 67, 16)];
  [dynamicIP setTitle: @"Dynamic"];
  [dynamicIP setButtonType: NSRadioButton];
  [dynamicIP setShowsBorderOnlyWhileMouseInside: YES];
  [dynamicIP setImagePosition: NSImageLeft];
  [dynamicIP setAction: @selector (setIPTypeAction:)];
  [dynamicIP setTarget: self];
  [ipFrameView addSubview: dynamicIP];

  staticIP = [[NSButton alloc] initWithFrame: 
    NSMakeRect (105, 78, 67, 16)];
  [staticIP setTitle: @"Static"];
  [staticIP setButtonType: NSRadioButton];
  [staticIP setShowsBorderOnlyWhileMouseInside: YES];
  [staticIP setImagePosition: NSImageLeft];
  [staticIP setAction: @selector (setIPTypeAction:)];
  [staticIP setTarget: self];
  [ipFrameView addSubview: staticIP];

  // Local
  localIPLabel = [[NSTextField alloc] initWithFrame: 
    NSMakeRect (0, 52, 100, 21)];
  [localIPLabel setAlignment: NSRightTextAlignment];
  [localIPLabel setBackgroundColor: [NSColor windowBackgroundColor]];
  [localIPLabel setSelectable: NO];
  [localIPLabel setEditable: NO];
  [localIPLabel setBezeled: NO];
  [localIPLabel setBordered: NO];
  [localIPLabel setStringValue: @"Local:"];
  [ipFrameView addSubview: localIPLabel];
  
  localIPTF = [[NSTextField alloc] initWithFrame: 
    NSMakeRect (105, 52, 106, 21)];
  [ipFrameView addSubview: localIPTF];

  // Remote
  remoteIPLabel = [[NSTextField alloc] initWithFrame: 
    NSMakeRect (0, 26, 100, 21)];
  [remoteIPLabel setAlignment: NSRightTextAlignment];
  [remoteIPLabel setBackgroundColor: [NSColor windowBackgroundColor]];
  [remoteIPLabel setSelectable: NO];
  [remoteIPLabel setEditable: NO];
  [remoteIPLabel setBezeled: NO];
  [remoteIPLabel setBordered: NO];
  [remoteIPLabel setStringValue: @"Remote:"];
  [ipFrameView addSubview: remoteIPLabel];
  
  remoteIPTF = [[NSTextField alloc] initWithFrame: 
    NSMakeRect (105, 26, 106, 21)];
  [ipFrameView addSubview: remoteIPTF];

  // Netmask
  netmaskLabel = [[NSTextField alloc] initWithFrame: 
    NSMakeRect (0, 0, 100, 21)];
  [netmaskLabel setAlignment: NSRightTextAlignment];
  [netmaskLabel setBackgroundColor: [NSColor windowBackgroundColor]];
  [netmaskLabel setSelectable: NO];
  [netmaskLabel setEditable: NO];
  [netmaskLabel setBezeled: NO];
  [netmaskLabel setBordered: NO];
  [netmaskLabel setStringValue: @"Netmask:"];
  [ipFrameView addSubview: netmaskLabel];
  
  netmaskTF = [[NSTextField alloc] initWithFrame: 
    NSMakeRect (105, 0, 106, 21)];
  [ipFrameView addSubview: netmaskTF];

  // Linking above TFs
  [localIPTF setNextText: remoteIPTF];
  [remoteIPTF setNextText: netmaskTF];
  [netmaskTF setNextText: localIPTF];
  
  // DNS servers -------------------------------------------------- 
  dnsFrame = [[NSBox alloc] initWithFrame: NSMakeRect (0, 129, 280, 115)];
  [dnsFrame setTitle: @"DNS servers"];
  [view addSubview: dnsFrame];
  dnsFrameView = [dnsFrame contentView];

  dynamicDNS = [[NSButton alloc] initWithFrame: 
    NSMakeRect (105, 73, 67, 16)];
  [dynamicDNS setTitle: @"Dynamic"];
  [dynamicDNS setButtonType: NSRadioButton];
  [dynamicDNS setImagePosition: NSImageLeft];
  [dynamicDNS setAction: @selector (setDNSTypeAction:)];
  [dynamicDNS setTarget: self];
  [dnsFrameView addSubview: dynamicDNS];

  staticDNS = [[NSButton alloc] initWithFrame: 
    NSMakeRect (105, 52, 67, 16)];
  [staticDNS setTitle: @"Static"];
  [staticDNS setButtonType: NSRadioButton];
  [staticDNS setImagePosition: NSImageLeft];
  [staticDNS setAction: @selector (setDNSTypeAction:)];
  [staticDNS setTarget: self];
  [dnsFrameView addSubview: staticDNS];

  // Primary
  primaryDNSLabel = [[NSTextField alloc] initWithFrame: 
    NSMakeRect (0, 26, 100, 21)];
  [primaryDNSLabel setAlignment: NSRightTextAlignment];
  [primaryDNSLabel setBackgroundColor: [NSColor windowBackgroundColor]];
  [primaryDNSLabel setSelectable: NO];
  [primaryDNSLabel setEditable: NO];
  [primaryDNSLabel setBezeled: NO];
  [primaryDNSLabel setBordered: NO];
  [primaryDNSLabel setStringValue: @"Primary:"];
  [dnsFrameView addSubview: primaryDNSLabel];
  
  primaryDNSTF = [[NSTextField alloc] initWithFrame: 
    NSMakeRect (105, 26, 106, 21)];
  [dnsFrameView addSubview: primaryDNSTF];

  // Secondary
  secondaryDNSLabel = [[NSTextField alloc] initWithFrame: 
    NSMakeRect (0, 0, 100, 21)];
  [secondaryDNSLabel setAlignment: NSRightTextAlignment];
  [secondaryDNSLabel setBackgroundColor: [NSColor windowBackgroundColor]];
  [secondaryDNSLabel setSelectable: NO];
  [secondaryDNSLabel setEditable: NO];
  [secondaryDNSLabel setBezeled: NO];
  [secondaryDNSLabel setBordered: NO];
  [secondaryDNSLabel setStringValue: @"Secondary:"];
  [dnsFrameView addSubview: secondaryDNSLabel];
  
  secondaryDNSTF = [[NSTextField alloc] initWithFrame: 
    NSMakeRect (105, 0, 106, 21)];
  [dnsFrameView addSubview: secondaryDNSTF];

  [primaryDNSTF setNextText: secondaryDNSTF];
  [secondaryDNSTF setNextText: primaryDNSTF];

  // Routing -------------------------------------------------- 
  routingFrame = [[NSBox alloc] initWithFrame: NSMakeRect (0, 30, 280, 94)];
  [routingFrame setTitle: @"Routing"];
  [view addSubview: routingFrame];
  routingFrameView = [routingFrame contentView];

  // Default gateway
  defaultGateway = [[NSButton alloc] initWithFrame: 
    NSMakeRect (10, 47, 191, 21)];
  [defaultGateway setTitle: @"Default gateway to remote side"];
  [defaultGateway setButtonType: NSSwitchButton];
  [routingFrameView addSubview: defaultGateway];
  
  // Proxyarp
  proxyARP = [[NSButton alloc] initWithFrame: 
    NSMakeRect (10, 26, 191, 21)];
  [proxyARP setTitle: @"Join remote network (proxyarp)"];
  [proxyARP setButtonType: NSSwitchButton];
  [routingFrameView addSubview: proxyARP];
  
  // Advanced routing button
  advancedRouting = [[NSButton alloc] initWithFrame: 
    NSMakeRect (184, 0, 82, 21)];
  [advancedRouting setButtonType: NSMomentaryPushButton];
  [advancedRouting setTitle: @"Advanced..."];
  [advancedRouting setAction: @selector (advancedRoutingAction)];
  [advancedRouting setTarget: self];
  [routingFrameView addSubview: advancedRouting];

  RELEASE (dynamicIP);
  RELEASE (staticIP);
  RELEASE (localIPLabel);
  RELEASE (localIPTF);
  RELEASE (remoteIPLabel);
  RELEASE (remoteIPTF);
  RELEASE (netmaskLabel);
  RELEASE (netmaskTF);
  RELEASE (ipFrame);

  RELEASE (dynamicDNS);
  RELEASE (staticDNS);
  RELEASE (primaryDNSLabel);
  RELEASE (primaryDNSTF);
  RELEASE (secondaryDNSLabel);
  RELEASE (secondaryDNSTF);
  RELEASE (dnsFrame);

  RELEASE (defaultGateway);
  RELEASE (proxyARP);
  RELEASE (advancedRouting);
  RELEASE (routingFrame);

  // load settings or set defaults
  [self setDefaults];

  return self;
}

// dealloc never called for array memebers
- (void) release
{
  NSLog (@"NetworkView: release (%i)", [view retainCount]);
  if ([view superview])
  {
    [view removeFromSuperview];
  }
  else
  {
    RELEASE (view);
  }

  [super dealloc];
}

/*-------------------------------------------------------------------------*
 *  Protocol methods
 *-------------------------------------------------------------------------*/
- (id) show: (id)superView
{
  [superView addSubview: view];
  RELEASE (view);

  return nil;
}

- (void) hide 
{
  RETAIN (view);
  [view removeFromSuperview];
}

- (void) setDefaults
{
  NSButton *ipType;
  NSButton *dnsType;
  
  if ([prefs count] > 0)
  {
    if ([[prefs objectForKey: @"DynamicIP"] isEqualToString: @"YES"])
    {
      ipType = dynamicIP;
    }
    else
    {
      ipType = staticIP;
    }
    [localIPTF setStringValue: [prefs objectForKey: @"LocalIP"]];
    [remoteIPTF setStringValue: [prefs objectForKey: @"RemoteIP"]];
    [netmaskTF setStringValue: [prefs objectForKey: @"Netmask"]];
    
    if ([[prefs objectForKey: @"DynamicDNS"] isEqualToString: @"YES"])
    {
      dnsType = dynamicDNS;
    }
    else
    {
      dnsType = staticDNS;
    }
    [primaryDNSTF setStringValue: [prefs objectForKey: @"PrimaryDNS"]];
    [secondaryDNSTF setStringValue: [prefs objectForKey: @"SecondaryDNS"]];
    
    if ([[prefs objectForKey: @"DefaultGateway"] isEqualToString: @"YES"])
    {
      [defaultGateway setState: NSOnState];
    }
    else
    {
      [defaultGateway setState: NSOffState];
    }
    if ([[prefs objectForKey: @"ProxyARP"] isEqualToString: @"YES"])
    {
      [proxyARP setState: NSOnState];
    }
    else
    {
      [proxyARP setState: NSOffState];
    }
  }
  else
  {
    ipType = dynamicIP;
    dnsType = dynamicDNS;

    [defaultGateway setState: NSOnState];
    [proxyARP setState: NSOffState];
  }

  [self setIPTypeAction: ipType];
  [self setDNSTypeAction: dnsType];
}

- (NSMutableDictionary *) prefsDict
{
  NSMutableString *isDynamicIP;
  NSMutableString *isDynamicDNS;
  NSMutableString *isDefaultGateway;
  NSMutableString *isProxyARP;
 
  isDynamicIP = [NSMutableString stringWithString: @"YES"];
  isDynamicDNS = [NSMutableString stringWithString: @"YES"];
  isDefaultGateway = [NSMutableString stringWithString: @"YES"];
  isProxyARP = [NSMutableString stringWithString: @"YES"];

  if ([dynamicIP state] == NSOffState) [isDynamicIP setString: @"NO"];
  [prefs setObject: isDynamicIP forKey: @"DynamicIP"];
  [prefs setObject: [localIPTF stringValue] forKey: @"LocalIP"];
  [prefs setObject: [remoteIPTF stringValue] forKey: @"RemoteIP"];
  [prefs setObject: [netmaskTF stringValue] forKey: @"Netmask"];
  
  if ([dynamicDNS state] == NSOffState) [isDynamicDNS setString: @"NO"];
  [prefs setObject: isDynamicDNS forKey: @"DynamicDNS"];
  [prefs setObject: [primaryDNSTF stringValue] forKey: @"PrimaryDNS"];
  [prefs setObject: [secondaryDNSTF stringValue] forKey: @"SecondaryDNS"];
  
  if ([defaultGateway state] == NSOffState) [isDefaultGateway setString: @"NO"];
  [prefs setObject: isDefaultGateway forKey: @"DefaultGateway"];
    
  if ([proxyARP state] == NSOffState) [isProxyARP setString: @"NO"];
  [prefs setObject: isProxyARP forKey: @"ProxyARP"];

  return prefs;
}

/*-------------------------------------------------------------------------*
 *  Controls' action methods
 *-------------------------------------------------------------------------*/
- (void) setIPTypeAction: (id)sender
{
  if (sender == dynamicIP)
  {
    [dynamicIP setState: NSOnState];
    [staticIP setState: NSOffState];

    [localIPLabel setTextColor: [NSColor darkGrayColor]];
    [localIPTF setEditable: NO];
    [localIPTF setSelectable: NO];
    [localIPTF setTextColor: [NSColor darkGrayColor]];

    [remoteIPLabel setTextColor: [NSColor darkGrayColor]];
    [remoteIPTF setEditable: NO];
    [remoteIPTF setSelectable: NO];
    [remoteIPTF setTextColor: [NSColor darkGrayColor]];

    [netmaskLabel setTextColor: [NSColor darkGrayColor]];
    [netmaskTF setEditable: NO];
    [netmaskTF setSelectable: NO];
    [netmaskTF setTextColor: [NSColor darkGrayColor]];
  }
  else if (sender == staticIP)
  {
    [dynamicIP setState: NSOffState];
    [staticIP setState: NSOnState];

    [localIPLabel setTextColor: [NSColor blackColor]];
    [localIPTF setEditable: YES];
    [localIPTF setSelectable: YES];
    [localIPTF setTextColor: [NSColor blackColor]];

    [remoteIPLabel setTextColor: [NSColor blackColor]];
    [remoteIPTF setEditable: YES];
    [remoteIPTF setSelectable: YES];
    [remoteIPTF setTextColor: [NSColor blackColor]];

    [netmaskLabel setTextColor: [NSColor blackColor]];
    [netmaskTF setEditable: YES];
    [netmaskTF setSelectable: YES];
    [netmaskTF setTextColor: [NSColor blackColor]];
  }

  [localIPLabel setNeedsDisplay];
  [localIPTF setNeedsDisplay];
  [remoteIPLabel setNeedsDisplay];
  [remoteIPTF setNeedsDisplay];
  [netmaskLabel setNeedsDisplay];
  [netmaskTF setNeedsDisplay];
}

- (void) setDNSTypeAction: (id)sender
{
  if (sender == dynamicDNS || sender == self)
  {
    [dynamicDNS setState: NSOnState];
    [staticDNS setState: NSOffState];

    [primaryDNSLabel setTextColor: [NSColor darkGrayColor]];
    [primaryDNSTF setEditable: NO];
    [primaryDNSTF setSelectable: NO];
    [primaryDNSTF setTextColor: [NSColor darkGrayColor]];

    [secondaryDNSLabel setTextColor: [NSColor darkGrayColor]];
    [secondaryDNSTF setEditable: NO];
    [secondaryDNSTF setSelectable: NO];
    [secondaryDNSTF setTextColor: [NSColor darkGrayColor]];
  }
  else if (sender == staticDNS)
  {
    [dynamicDNS setState: NSOffState];
    [staticDNS setState: NSOnState];

    [primaryDNSLabel setTextColor: [NSColor blackColor]];
    [primaryDNSTF setEditable: YES];
    [primaryDNSTF setSelectable: YES];
    [primaryDNSTF setTextColor: [NSColor blackColor]];

    [secondaryDNSLabel setTextColor: [NSColor blackColor]];
    [secondaryDNSTF setEditable: YES];
    [secondaryDNSTF setSelectable: YES];
    [secondaryDNSTF setTextColor: [NSColor blackColor]];
  }
  [primaryDNSLabel setNeedsDisplay];
  [primaryDNSTF setNeedsDisplay];
  [secondaryDNSLabel setNeedsDisplay];
  [secondaryDNSTF setNeedsDisplay];
}

- (void) advancedRoutingAction
{
  NSRunAlertPanel(@"Session Preferences", 
      @"Sorry, adnvanced routing setting is not implemented yet",
      @"Ok", nil, nil);
}

@end
