/*
 *  Author:       Serg Stoyan
 *
 *  Classes:      SessionPrefsPanel : NSObject
 *
 *  Description:  Panel with PPP session settings.
 */

#import <Foundation/NSFileManager.h>

#import <AppKit/NSPanel.h>
#import <AppKit/NSPopUpButton.h>
#import <AppKit/NSBox.h>
#import <AppKit/NSTextField.h>

#import "AppController.h"
#import "AppDefaults.h"
#import "SessionWindow.h"

#import "Panel.h"
#import "Connection.h"
#import "Modem.h"
#import "Dialing.h"
#import "Network.h"
#import "Compression.h"
#import "Authentication.h"

@implementation SessionPrefsPanel

- (id) init
{
  [super init];
  // Panel
  panel = [[NSPanel alloc] initWithContentRect: NSMakeRect (0, 0, 300, 502)
                                     styleMask: NSTitledWindowMask |
                                                NSClosableWindowMask
                                       backing: NSBackingStoreRetained
                                         defer: YES];
  
  [panel setTitle: @"Session Inspector"];
  [panel setMinSize: NSMakeSize(300, 502)];
  [panel setMaxSize: NSMakeSize(300, 502)];
  [panel setReleasedWhenClosed: NO];
  [panel setHidesOnDeactivate: NO];
  [panel setDelegate: self];
  panelContentView = [panel contentView];

  if (![panel setFrameUsingName:@"SessionPrefs"])
  {
    [panel center];
  }

  // Sections popup
  sectionsPopup = [[NSPopUpButton alloc] initWithFrame: 
    NSMakeRect (93, 449, 112, 20)];
  [sectionsPopup setTag: 10];
  [sectionsPopup setAutoenablesItems: NO];
  [sectionsPopup addItemWithTitle: @"Connection"];
  [sectionsPopup addItemWithTitle: @"Modem"];
  [sectionsPopup addItemWithTitle: @"Dialing"];
  [sectionsPopup addItemWithTitle: @"Network"];
  [sectionsPopup addItemWithTitle: @"Compression"];
  [sectionsPopup addItemWithTitle: @"Authentication"];
  [sectionsPopup setAction: @selector (changeView:)];
  [sectionsPopup setTarget: self];
  [panelContentView addSubview: sectionsPopup];

  [[sectionsPopup itemAtIndex: 0] setKeyEquivalent: @"1"];
  [[sectionsPopup itemAtIndex: 1] setKeyEquivalent: @"2"];
  [[sectionsPopup itemAtIndex: 2] setKeyEquivalent: @"3"];
  [[sectionsPopup itemAtIndex: 3] setKeyEquivalent: @"4"];
  [[sectionsPopup itemAtIndex: 4] setKeyEquivalent: @"5"];
  [[sectionsPopup itemAtIndex: 5] setKeyEquivalent: @"6"];

  // Top Horizontal line
  topHLine = [[NSBox alloc] initWithFrame: NSMakeRect (0, 437, 300, 2)];
  [topHLine setTitlePosition: NSNoTitle];
  [panelContentView addSubview: topHLine];

  // Bottom Horizontal line
  bottomHLine = [[NSBox alloc] initWithFrame: NSMakeRect (0, 44, 300, 2)];
  [bottomHLine setTitlePosition: NSNoTitle];
  [panelContentView addSubview: bottomHLine];

  // Defaults button 
  defaultsButton = [[NSButton alloc] initWithFrame: 
    NSMakeRect (115, 10, 85, 24)];
  [defaultsButton setButtonType: NSMomentaryPushButton];
  [defaultsButton setTitle: @"Defauls"];
  [defaultsButton setAction: @selector (setDefaults)];
  [panelContentView addSubview: defaultsButton];

  // Set button
  setButton = [[NSButton alloc] initWithFrame: 
    NSMakeRect (205, 10, 85, 24)];
  [setButton setButtonType: NSMomentaryPushButton];
  [setButton setTitle: @"Set"];
  [setButton setAction: @selector (setButtonAction)];
  [setButton setTarget: self];
  [panelContentView addSubview: setButton];

  sectionViews = [[NSMutableDictionary alloc] initWithCapacity: 1];

  RELEASE (sectionsPopup);
  RELEASE (topHLine);
  RELEASE (bottomHLine);
  RELEASE (setButton);
  RELEASE (defaultsButton);

  return self;
}

- (id) openNewSessionWithPrefs: (NSMutableDictionary *)sessions
{
  NSMutableDictionary *connectionDict;

  NSLog (@"openNewSessionWithPrefs");

  [panel setTitle: @"New - Session Inspector"];

  // Creating Sessions = { "Session name" = {}; };
  currentSessionDict = RETAIN ([NSMutableDictionary dictionary]);
  // Creating Sessions = { "Session name" = { Connection = {}; }; };
  connectionDict = RETAIN ([NSMutableDictionary dictionary]);

  [connectionDict setObject: @"New" forKey: @"ConnectionName"];
  [currentSessionDict setObject: connectionDict forKey: @"Connection"];
  [sessions setObject: currentSessionDict forKey: @"New"];
  
  [self loadPrefsForSession: @"New" andPrefs: sessions];
  [self orderFront: nil];

  return self;
}

- (id) openWithSessionName: (NSString *)sessionName
                  andPrefs: (NSMutableDictionary *)sessions
{
  NSLog(@"openWithSessionName");

  [self loadPrefsForSession: sessionName andPrefs: sessions];
  [self orderFront: nil];

  return self;
}

- (id) loadPrefsForSession: (NSString *)sessionName
                  andPrefs: (NSMutableDictionary *)sessions
{
  NSLog(@"loadPrefsForSession");

  currentSessionName = sessionName;
  sessionsDict = sessions;
  [sectionViews removeAllObjects];
  
  [panel setTitle: [NSString 
    stringWithFormat: @"%@ - Session Inspector", currentSessionName]];

  currentSessionDict = [[sessionsDict objectForKey: currentSessionName] mutableCopy];
  if (currentSessionDict == nil)
  {
    NSLog (@"ALERT!!! impossible situation!!!");
  }
 
  [sectionViews 
    setObject: [[ConnectionView alloc] initWithPrefs: currentSessionDict]
    forKey: @"Connection"];
  [sectionViews 
    setObject: [[ModemView new]  initWithPrefs: currentSessionDict]
    forKey: @"Modem"];
  [sectionViews 
    setObject: [[DialingView new]  initWithPrefs: currentSessionDict]
    forKey: @"Dialing"];
  [sectionViews 
    setObject: [[NetworkView new]  initWithPrefs: currentSessionDict]
    forKey: @"Network"];
  [sectionViews 
    setObject: [[CompressionView new]  initWithPrefs: currentSessionDict]
    forKey: @"Compression"];
  [sectionViews 
    setObject: [[AuthenticationView new] initWithPrefs: currentSessionDict]
    forKey: @"Authentication"];

  currentSectionView = nil;
  [sectionsPopup selectItemWithTitle: @"Connection"];
  [self changeView: self];

  return self;
}

- (NSString *) currentSessionName
{
  return currentSessionName;
}

/*-------------------------------------------------------------------------*
 *  Window delegate methods
 *-------------------------------------------------------------------------*/
- (void) orderFront: (id)sender
{
  [panel orderFront: nil];
  [panel makeKeyWindow];
//  [NSApp runModalForWindow: panel];
} 

- (void) dealloc
{
  NSLog (@"PreferencesPanel: dealloc");
  
  RELEASE (sectionViews);
  RELEASE (panel);

  [super dealloc];
}

- (void) close 
{
  [panel close];
}

- (void) windowWillClose: (NSNotification *)notif
{
  NSLog (@"PreferencesPanel: will close");

  [ConnectApp sessionPrefsPanelWillClose];
  
  [panel saveFrameUsingName: @"SessionPrefs"];
  
  [self release];
}

/*-------------------------------------------------------------------------*
 * Controls' actions methods
 *-------------------------------------------------------------------------*/
- (void) changeView: (id)sender
{
  NSString *selectedItem = [sectionsPopup titleOfSelectedItem];
  id       firstResponder;
  
  if (selectedItem != nil)
  {
    selectedSectionView = [sectionViews objectForKey: selectedItem];

    if (selectedSectionView != nil 
        && selectedSectionView != currentSectionView)
    {
      if (currentSectionView != nil)
      {
        [currentSectionView hide];
      }
      firstResponder = [selectedSectionView show: panelContentView]; 
      if (firstResponder != nil)
      {
        [panel makeFirstResponder: firstResponder];
      }
      [defaultsButton setTarget: selectedSectionView];
      currentSectionView = selectedSectionView;
    }
  }
}

- (void) setButtonAction
{
  NSMutableDictionary *connectionDict = nil;
  NSMutableDictionary *modemDict = nil;
  NSMutableDictionary *dialingDict = nil;
  NSMutableDictionary *networkDict = nil;
  NSMutableDictionary *compressionDict = nil;
  NSMutableDictionary *authDict = nil;
  NSString            *sessionName;

  // Connection
  connectionDict = [[sectionViews objectForKey: @"Connection"] prefsDict];
  sessionName = [connectionDict objectForKey: @"ConnectionName"];

  // remove this session settings with old name
  if ([sessionName compare: currentSessionName] != NSOrderedSame)
  {
    [sessionsDict removeObjectForKey: currentSessionName];
  }
  [currentSessionDict setObject: connectionDict forKey: @"Connection"];
  NSLog (@"- Connection prefs added");

  if ([[connectionDict objectForKey: @"ConnectionType"] 
      isEqualToString: @"Connect session"])
  {
    // Modem
    if ((modemDict = [[sectionViews objectForKey: @"Modem"] prefsDict]) 
        == nil)
    {
      [sectionsPopup selectItemWithTitle: @"Modem"];
      [self changeView: self];
      return;
    }
    [currentSessionDict setObject: modemDict forKey: @"Modem"];
    NSLog (@"- Modem prefs added");
    
    // Dialing
    if ((dialingDict = [[sectionViews objectForKey: @"Dialing"] prefsDict])
        == nil)
    {
      [sectionsPopup selectItemWithTitle: @"Dialing"];
      [self changeView: self];
      return;
    }
    [currentSessionDict setObject: dialingDict forKey: @"Dialing"];
    NSLog (@"- Dialing prefs added");

    // Network
    if ((networkDict = [[sectionViews objectForKey: @"Network"] prefsDict])
        == nil)
    {
      [sectionsPopup selectItemWithTitle: @"Network"];
      [self changeView: self];
      return;
    }
    [currentSessionDict setObject: networkDict forKey: @"Network"];
    NSLog (@"- Network prefs added");

    // Compression
    if ((compressionDict = [[sectionViews objectForKey: @"Compression"] prefsDict])
        == nil)
    {
      [sectionsPopup selectItemWithTitle: @"Compression"];
      [self changeView: self];
      return;
    }
    [currentSessionDict setObject: compressionDict forKey: @"Compression"];
    NSLog (@"- Compression prefs added");

    // Authentication
    if ((authDict = [[sectionViews objectForKey: @"Authentication"] prefsDict])
        == nil)
    {
      [sectionsPopup selectItemWithTitle: @"Authentication"];
      [self changeView: self];
      return;
    }
    [currentSessionDict setObject: authDict forKey: @"Authentication"];
    NSLog (@"- Authentication prefs added");
    
    [self writeBundleFilesForSession: sessionName];

    // Remove Password object from Session Authentication prefs
    [authDict removeObjectForKey: @"Password"];
  }
  else
  {
    [currentSessionDict removeObjectForKey: @"Modem"];
    [currentSessionDict removeObjectForKey: @"Network"];
    [currentSessionDict removeObjectForKey: @"Compression"];
    [currentSessionDict removeObjectForKey: @"Authentication"];
  }

  [sessionsDict setObject: currentSessionDict forKey: sessionName];

  [[ConnectApp appDefaults] writeToFile];
  
  [[ConnectApp sessionWindow] updateSessionsList];
}

- (void) writeBundleFilesForSession: (NSString *)sessionName
{
  NSMutableString *pppdOptions;
  NSMutableString *secrets;
  NSMutableString *chatScript;
  NSDictionary    *prefsDict;

  // Name.session pack
  if (![[ConnectApp appDefaults] createSessionBundle: currentSessionDict 
    withSessionName: sessionName])
  {
    NSLog (@"error creating session bundle directory. files writing: abort");
    return;
  }

  // pppd options file ----------------------------------------------
  // Defaults
  pppdOptions = [[NSMutableString alloc] initWithCString: "lock\n"];
  [pppdOptions appendString: [NSString stringWithCString: "modem\n"]];
  [pppdOptions appendString: [NSString stringWithCString: "hide-password\n"]];
  [pppdOptions appendString: [NSString stringWithCString: "nodetach\n"]];
  [pppdOptions appendString: [NSString stringWithCString: "nopersist\n"]];
  // "noauth" requires root permissions
  // [pppdOptions appendString: [NSString stringWithCString: "noauth\n"]];

// Modem prefs
  prefsDict = [currentSessionDict objectForKey: @"Connection"];
  [pppdOptions appendString: 
    [NSString stringWithCString: "\n# Connection\n"]];
    
  [pppdOptions appendString: [NSString stringWithCString: "idle "]];
  [pppdOptions appendString: [prefsDict objectForKey: @"Idle"]];
  [pppdOptions appendString: [NSString stringWithCString: "\n"]];

  // Modem prefs
  prefsDict = [currentSessionDict objectForKey: @"Modem"];
  [pppdOptions appendString: [NSString stringWithCString: "\n# Modem\n"]];

  if ([[prefsDict objectForKey: @"DefaultMRUAndMTU"] isEqualToString: @"NO"])
  {
    [pppdOptions appendString: [NSString stringWithCString: "mru "]];
    [pppdOptions appendString: [prefsDict objectForKey: @"MRU"]];
    [pppdOptions appendString: [NSString stringWithCString: "\n"]];

    [pppdOptions appendString: [NSString stringWithCString: "mtu "]];
    [pppdOptions appendString: [prefsDict objectForKey: @"MTU"]];
    [pppdOptions appendString: [NSString stringWithCString: "\n"]];
  }

  if (![[prefsDict objectForKey: @"EscapeChars"] isEqualToString: @""])
  {
    [pppdOptions appendString: [NSString stringWithCString: "escape "]];
    [pppdOptions appendString: [prefsDict objectForKey: @"EscapeChars"]];
    [pppdOptions appendString: [NSString stringWithCString: "\n"]];
  }

  if (![[prefsDict objectForKey: @"AsyncCharmap"] isEqualToString: @""])
  {
    [pppdOptions appendString: [NSString stringWithCString: "asyncmap "]];
    [pppdOptions appendString: [prefsDict objectForKey: @"AsyncCharmap"]];
    [pppdOptions appendString: [NSString stringWithCString: "\n"]];
  }
  
  // Network prefs
  prefsDict = [currentSessionDict objectForKey: @"Network"];
  [pppdOptions appendString: [NSString stringWithCString: "\n# Network\n"]];
  
  if ([[prefsDict objectForKey: @"DynamicIP"] isEqualToString: @"NO"])
  {
    [pppdOptions appendString: [prefsDict objectForKey: @"LocalIP"]];
    [pppdOptions appendString: [NSString stringWithCString: ":"]];
    [pppdOptions appendString: [prefsDict objectForKey: @"RemoteIP"]];
    [pppdOptions appendString: [NSString stringWithCString: "\n"]];
    [pppdOptions appendString: [NSString stringWithCString: "netmask "]];
    [pppdOptions appendString: [prefsDict objectForKey: @"Netmask"]];
    [pppdOptions appendString: [NSString stringWithCString: "\n"]];
  }
  
  // TODO write DNS servers if any to /etc/resolv.conf
  if ([[prefsDict objectForKey: @"DynamicDNS"] isEqualToString: @"YES"])
  {
    [pppdOptions appendString: [NSString stringWithCString: "usepeerdns\n"]];
  }

  if ([[prefsDict objectForKey: @"DefaultGateway"] isEqualToString: @"YES"])
  {
    [pppdOptions appendString: 
      [NSString stringWithCString: "defaultroute\n"]];
  }

  if ([[prefsDict objectForKey: @"ProxyARP"] isEqualToString: @"YES"])
  {
    [pppdOptions appendString: [NSString stringWithCString: "proxyarp\n"]];
  }
  
  // Compression prefs
  prefsDict = [currentSessionDict objectForKey: @"Compression"];
  [pppdOptions appendString: 
    [NSString stringWithCString: "\n# Compression\n"]];

  if ([[prefsDict objectForKey: @"BSD"] isEqualToString: @"NO"])
  {
    [pppdOptions appendString: [NSString stringWithCString: "nobsdcomp\n"]];
  }
  if ([[prefsDict objectForKey: @"Deflate"] isEqualToString: @"NO"])
  {
    [pppdOptions appendString: [NSString stringWithCString: "nodeflate\n"]];
  }
  if ([[prefsDict objectForKey: @"AddressControl"] isEqualToString: @"NO"])
  {
    [pppdOptions appendString: [NSString stringWithCString: "noaccomp\n"]];
  }
  if ([[prefsDict objectForKey: @"CCP"] isEqualToString: @"NO"])
  {
    [pppdOptions appendString: [NSString stringWithCString: "noccp\n"]];
  }
  if ([[prefsDict objectForKey: @"ProtocolField"] isEqualToString: @"NO"])
  {
    [pppdOptions appendString: [NSString stringWithCString: "nopcomp\n"]];
  }
  if ([[prefsDict objectForKey: @"TCPIPHeader"] isEqualToString: @"NO"])
  {
    [pppdOptions appendString: [NSString stringWithCString: "novj\n"]];
  }
  if ([[prefsDict objectForKey: @"ConnectionID"] isEqualToString: @"NO"])
  {
    [pppdOptions appendString: [NSString stringWithCString: "novjccomp\n"]];
  }
  if ([[prefsDict objectForKey: @"Predictor1"] isEqualToString: @"NO"])
  {
    [pppdOptions appendString: [NSString stringWithCString: "nopredictor1\n"]];
  }
  
  // Authentication prefs
  // TODO if PAP/CHAP write into pap-secrets or chap-secrets
  prefsDict = [currentSessionDict objectForKey: @"Authentication"];
  [pppdOptions appendString: 
    [NSString stringWithCString: "\n# Authentication\n"]];

  if (![[prefsDict objectForKey: @"UserName"] isEqualToString: @""])
  {
    [pppdOptions appendString: [NSString stringWithCString: "user "]];
    [pppdOptions appendString: [prefsDict objectForKey: @"UserName"]];
    [pppdOptions appendString: [NSString stringWithCString: "\n"]];

    secrets = [[NSMutableString alloc] initWithString: 
      [prefsDict objectForKey: @"UserName"]];
    if (![[prefsDict objectForKey: @"RemoteSystemName"] isEqualToString: @""])
    {
      [secrets appendString: @"\t"];
      [secrets appendString: [prefsDict objectForKey: @"RemoteSystemName"]];
      [secrets appendString: @"\t"];
    }
    else
    {
      [secrets appendString: @"\t*\t"];
    }
    [secrets appendString: [prefsDict objectForKey: @"Password"]];

    [[ConnectApp appDefaults] writeString: secrets intoFile: @"secret.pap" 
      forSessionBundle: sessionName];
    [[ConnectApp appDefaults] writeString: secrets intoFile: @"secret.chap" 
      forSessionBundle: sessionName];

    RELEASE (secrets);
  }
  
  if (![[prefsDict objectForKey: @"LocalSystemName"] isEqualToString: @""])
  {
    [pppdOptions appendString: [NSString stringWithCString: "name "]];
    [pppdOptions appendString: [prefsDict objectForKey: @"LocalSystemName"]];
    [pppdOptions appendString: [NSString stringWithCString: "\n"]];
  }
  if (![[prefsDict objectForKey: @"RemoteSystemName"] isEqualToString: @""])
  {
    [pppdOptions appendString: [NSString stringWithCString: "remotename "]];
    [pppdOptions appendString: [prefsDict objectForKey: @"RemoteSystemName"]];
    [pppdOptions appendString: [NSString stringWithCString: "\n"]];
  }
  if (![[prefsDict objectForKey: @"Domain"] isEqualToString: @""])
  {
    [pppdOptions appendString: [NSString stringWithCString: "domain "]];
    [pppdOptions appendString: [prefsDict objectForKey: @"Domain"]];
    [pppdOptions appendString: [NSString stringWithCString: "\n"]];
  }

  // Writing pppd options file
  [[ConnectApp appDefaults] writeString: pppdOptions 
    intoFile: @"pppd.options" 
    forSessionBundle: sessionName];

  RELEASE (pppdOptions);

  // Chat file ------------------------------------------------------
  // TODO get the following from "Dialing" prefs:
  //      - Pulse/Tone dial
  //      - time waiting for init response
  //      - time waiting for "CONNECT" response
  //      - init string
  prefsDict = [currentSessionDict objectForKey: @"Dialing"];

  chatScript = [[NSMutableString alloc] initWithCString: "ECHO\tOFF\n"];
  [chatScript appendFormat: @"%@\t\"%@\"\n", @"ABORT",
    [prefsDict objectForKey: @"BusyString"]];
  [chatScript appendFormat: @"%@\t\"%@\"\n", @"ABORT",
    [prefsDict objectForKey: @"ErrorString"]];
  [chatScript appendFormat: @"%@\t\"%@\"\n", @"ABORT",
    [prefsDict objectForKey: @"VoiceString"]];
  [chatScript appendFormat: @"%@\t\"%@\"\n", @"ABORT",
    [prefsDict objectForKey: @"NoAnswerString"]];
  [chatScript appendFormat: @"%@\t\"%@\"\n", @"ABORT",
    [prefsDict objectForKey: @"NoCarrierString"]];
  [chatScript appendFormat: @"%@\t\"%@\"\n", @"ABORT",
    [prefsDict objectForKey: @"NoDialtoneString"]];
    
  [chatScript appendFormat: @"%@\t%@\n", @"TIMEOUT",
    [prefsDict objectForKey: @"InitTimeout"]];
  [chatScript appendFormat: @"%@\t%@\n", @"\"\"",
    [prefsDict objectForKey: @"InitString"]];
  [chatScript appendFormat: @"%@\t%@\n", @"OK", @"\"\"" ];

  [chatScript appendFormat: @"%@\t\"%@\"\n", @"REPORT",
    [prefsDict objectForKey: @"ConnectString"]];
  [chatScript appendFormat: @"%@\t%@\n", @"TIMEOUT",
    [prefsDict objectForKey: @"DialTimeout"]];
  [chatScript appendFormat: @"%@\t%@\\T\n", @"\"\"",
    [prefsDict objectForKey: @"DialPrefix"]];
  [chatScript appendFormat: @"%@\t\"%@\"\n", 
    [prefsDict objectForKey: @"ConnectString"], @""];

  // Writing chat file
  [[ConnectApp appDefaults] writeString: chatScript intoFile: @"chat.script" 
    forSessionBundle: sessionName];
  
  RELEASE (chatScript);
}

@end

