/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*
 *
  Copyright Ricardo Galli <gallir@uib.es>
  This is an attempt to do The Right Thing for SpeedSteps and PPC.
  And also obeys strictly to KISS.


  Based on ideas and some code in cpufreq.cc
*/

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <sys/errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <stdarg.h>
#include <time.h>
#include <sys/time.h>
#include <sys/resource.h>

#include "cpudynd.h"
#include "cpus.h"

unsigned short_interval, long_interval;
unsigned do_services = 0;
unsigned do_exit = 0;
unsigned do_cpufreq = 0;
unsigned do_min_freq = 0;
float    min_freq_float;
unsigned acpi_only = 0;
unsigned do_standby = 0;
unsigned interval = 0;			// In milliseconds
unsigned count_idle = 0;
unsigned throttling_low = THROTTLING_LOW_DEFAULT;

unsigned asus_acpi_support = 0;
unsigned asus_disable = 1;

char *standby_devs = NULL;
unsigned standby_interval = 0;
unsigned standby_timeout = 120;			// In seconds

extern float clock_up_idle;
extern float clock_down_idle;


char *str_usage =
	"Usage: cpudynd [Options]\n"
	"       Options:\n\n"
	"       -acpi                  - do not use frequency scaling, use ACPI throttling if available\n"
	"\n"
	"       -asus                  - Enable Asus laptop led display\n"
	"\n"
	"       -d                     - Daemonize process (run in background)\n"
	"\n"
	"       -h <dev0[,dev1]...>    - Specified the disks to spindown, example: -h /dev/hda,/dev/hdc\n"
	"                                If this option is specified, but not -t, the default is 120 secs)\n"
	"\n"
	"       -i <interval>          - Check period in 1/10 seconds. Alternative to -ms.\n"
	"\n"
	"       -l <state>             - Specifies to which throttling state the cpu is set when\n"
	"                                speeding down (default is 3)\n"
	"\n"
	"       -minf <min>            - Set the minimum CPU's frequency in a value\n"
	"                                between 0.0 and 1.0\n"
	"\n"
	"       -ms <interval>         - Check period in miliseconds. Alternative to -i.\n"
	"\n"
	"       -nice                  - Count nice CPU usage as load (default = no)\n"
	"\n"
	"       -p <up> <down>         - Set CPU idle:work ratios to speed up or\n"
	"                                down CPU (default is 0.5 0.9)\n"
	"\n"
	"       -t <timeout>           - Set the timeout to put the disk in standby mode\n"
	"                                if there were no io during that period\n"
	"                                _Not_ activated by default\n"
	"                                /dev/hda is assumed if not  specified option -h)\n"
	"                                If option -h is specified, but not -t, the default is 120 secs)\n"
	"\n"
	"       -X                     - Just exit after initial checkings\n"
	"\n"
	"       -V                     - just print version and exit\n"
	"\n"
	"       -?, --help             - print this help\n"
	"\n";

// display an error message and exit the program
void die(unsigned system_error, const char *fmt, ...)
{

	va_list ap;

	fprintf(stderr, "Error: ");
	va_start(ap, fmt);			// get variable argument list passed
	vfprintf(stderr, fmt, ap);	// display message passed on stderr
	va_end(ap);

	fprintf(stderr, "\n");
	if (system_error)
		fprintf(stderr, "Error: %s\n", strerror(errno));

	exit(1);
}

// handles the HUP signal (dynamically scale performance)
void term_handler(int s)
{
	if(do_cpufreq) {
		cpu_term();
	}
	if (asus_acpi_support) {
		asus_leds(99);
	}

	exit(0);
}


long get_int(const char *s)
{
	char *end;
	long r = strtol(s, &end, 10);
	if (*end != '\0')
		die(FALSE, "Not an integer: %s", s);
	if (errno == ERANGE)
		die(FALSE, "Number is out of range: %s", s);

	return r;
}

float get_float(const char *s)
{
	char *end;
	float r = (float) strtod(s, &end);
	if (*end != '\0')
		die(FALSE, "Not a floating point number: %s", s);
	if (errno == ERANGE)
		die(FALSE, "Number is out of range: %s", s);

	return r;
}


void print_version()
{
	fprintf(stderr, "cpudynd version %s Copyright: Ricardo Galli <gallir@uib.es>\n", VERSION);
}

int main(unsigned argc, char *argv[])
{
	unsigned daemonize = FALSE;
	unsigned i;
	struct sigaction signal_action;

	print_version();
	// parse argv
	for (i = 1; i < argc; i++) {
		if (!strcmp(argv[i], "-d")) {
			daemonize = TRUE;
		} else if (!strcmp(argv[i], "-nice")) {
				count_idle = 1;
		} else if (!strcmp(argv[i], "-acpi")) {
				acpi_only = 1;
		} else if (!strcmp(argv[i], "-i")) {
			if (argc <= i + 1)
				die(FALSE,
					"The -i option must be followed by an interval in tenths of a second\n");
			interval = get_int(argv[++i])*100;
			errprintf("Interval in milliseconds is %u\n", interval);
		} else if (!strcmp(argv[i], "-ms")) {
			if (argc <= i + 1)
				die(FALSE,
					"The -ms option must be followed by an interval in tenths of a second\n");
			interval = get_int(argv[++i]);
			errprintf("Interval in milliseconds is %u\n", interval);
		} else if (!strcmp(argv[i], "-minf")) {
			if (argc <= i + 1)
				die(FALSE, "The -minf option must be followed by a float value ( 0 < minf < 1)\n");
			min_freq_float = get_float(argv[++i]);
			errprintf("Min freq is %f\n", min_freq_float);
			if (min_freq_float <= 0.0 || min_freq_float >= 1.0 )
				die(FALSE, "The -minf option must be followed by a float value ( 0 < minf < 1)\n");
			do_min_freq = 1;
		} else if (!strcmp(argv[i], "-t")) {
			if (argc <= i + 1)
				die(FALSE,
					"The -t option must be followed by a timeout in seconds\n");
			standby_timeout = get_int(argv[++i]);
			do_standby = 1;
			errprintf("Standby timeout is %u\n", standby_timeout);
		} else if (!strcmp(argv[i], "-h")) {
			if (argc <= i + 1)
				die(FALSE,
					"The -h option must be followed by device names\n");
			standby_devs = argv[++i];
			do_standby = 1;
			errprintf("Disks %s\n", standby_devs);
		} else if (!strcmp(argv[i], "-p")) {
			if (argc <= i + 2)
				die(FALSE, "The -p option must be followed by 2 floats\n");

			clock_up_idle = get_float(argv[++i]);
			clock_down_idle = get_float(argv[++i]);
			errprintf("Triggers are %f %f\n", clock_up_idle,
					  clock_down_idle);
		} else if (!strcmp(argv[i], "-l")) {
			if (argc <= i + 1)
				die(FALSE,
					"The -l option must be followed by an integer\n");
			throttling_low = get_int(argv[++i]);
			errprintf("Throttling low state is T%u\n", throttling_low);
		} else if (!strcmp(argv[i], "-V")) {
			exit(0);
		} else if (!strcmp(argv[i], "-X")) {
			do_exit = 1;
		} else if ((!strcmp(argv[i], "-?"))||(!strcmp(argv[i], "--help"))) {
			printf( str_usage );
			exit(0);
		} else if (!strcmp(argv[i], "-asus")) {
			asus_disable = 0;
		} else {
			die(FALSE, "%s: Bad command line\n%s\n", argv[0], str_usage);
		}
	}

	sigemptyset(&signal_action.sa_mask);
	sigaddset(&signal_action.sa_mask, SIGALRM);
	sigaddset(&signal_action.sa_mask, SIGUSR1);
	sigaddset(&signal_action.sa_mask, SIGUSR2);
	sigaddset(&signal_action.sa_mask, SIGHUP);
	sigaddset(&signal_action.sa_mask, SIGTERM);
	sigaddset(&signal_action.sa_mask, SIGINT);
	signal_action.sa_flags = 0;

	//signal_action.sa_handler = check_cpu;
	//sigaction(SIGALRM, &signal_action, 0);

	signal_action.sa_handler = usr1_handler;
	sigaction(SIGUSR1, &signal_action, 0);

	signal_action.sa_handler = usr2_handler;
	sigaction(SIGUSR2, &signal_action, 0);

	signal_action.sa_handler = hup_handler;
	sigaction(SIGHUP, &signal_action, 0);

	signal_action.sa_handler = term_handler;
	sigaction(SIGTERM, &signal_action, 0);
	sigaction(SIGINT, &signal_action, 0);

	if (interval != 0 && (do_cpufreq = cpus_init()) == 1) {
		do_cpufreq = 1;
		do_services++;
	} else {
		fprintf(stderr, "cpudynd: CPU frequency control disabled\n");
		interval = 1000;
	}

	if (do_cpufreq && !asus_disable) {
		asus_acpi_support = get_asus_acpi_support();
	}

	if (do_standby == 1) {
		// Now we check for disks
		if (standby_devs == NULL)
			standby_devs = DEFAULT_DISK;
		do_standby = standby_init(standby_timeout, standby_devs);
		if(do_standby)
			do_services++;
	}

	if(do_services == 0) {
		die(FALSE, "Nothing to do, exiting\n");
		exit(1);
	}

	// run in background if requested
	if (daemonize)
		daemon(0, 0);

	if (do_cpufreq == 1) {
		short_interval = interval * 1000L;
		// less interrupts if we are in performance
		long_interval = short_interval * 5;
		// reset the time counters
		reset_times();
	} else {
		long_interval = short_interval = STANDBY_PERIOD * 1000000L;
	}
	errprintf("Short: %u, Long: %u\n", short_interval, long_interval);

	if (do_exit) {
		term_handler(0);
		fprintf(stderr, "Exiting due to option -X\n");
		exit(0);
	}

	// main loop
	while (1) {
		if (cpu_get_state() == PERFORMANCE) {
			usleep(long_interval);
			standby_interval += long_interval;
		} else {
			usleep(short_interval);
			standby_interval += short_interval;
		}
		if (do_cpufreq)
			check_cpu(0);
		if (standby_interval > STANDBY_PERIOD * 999999) {
			// Two seconds, at least, have elapsed
			if (do_standby) {
				check_standby();
			}
			standby_interval = 0;
		}
	}
	return 0;
}

