/***************************************************************************
 *																		 *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or	 *
 *   (at your option) any later version.								   *
 *																		 *
 ***************************************************************************/

/*
 *
  Ricardo Galli <gallir@uib.es>
  This is an attempt to do The Right Thing for SpeedSteps and PPC.
  And it also obeys strictly to KISS.
 */
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <sys/errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <stdarg.h>
#include <time.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "cpudynd.h"
#include "cpus.h"

static cpu_parser_t cpu_parser = NULL;
static cpu_writer_t cpu_writer = NULL;
static get_times_t get_times = NULL;

static char *cpufreq_dir = NULL;
static char *governor_file = NULL;
static char *asus_led_file_1 = "/proc/acpi/asus/mled";
static char *asus_led_file_2 = "/proc/acpi/asus/wled";
static long *frequencies_table = NULL;
static freq_count = 0;
static freq_min = -1; // Index to the table

unsigned current_state, mode = DYNAMIC;

extern unsigned do_cpufreq;
extern unsigned do_min_freq;
extern float    min_freq_float;
extern unsigned acpi_only;
extern unsigned count_idle;
extern unsigned throttling_low;

extern unsigned asus_acpi_support;

// cpufreq at the time the program was started
unsigned initial_state;
// if CPU idle/work ratio is below this, CPU will be set to next higher speed
float clock_up_idle = 0.50;
// if CPU idle/work ratio is above this, CPU will be set to next lower speed
float clock_down_idle = 0.90;


// read a line from a file
int read_line(const char *filename, char *line, unsigned len, char *key)
{
	int klen;
	char *str;
	int fd;
	static char buffer[4096];

	fd = open(filename, O_RDONLY);
	if (fd<0)
		die(TRUE, "Could not open file for reading: %s\nDid you enable cpufreq or ACPI throttling in your kernel?",
			filename);
	if(read(fd, buffer, 4096) <= 0) {
		close(fd);
		return 0;
	}
	close(fd);
	str = strtok(buffer, "\n\r");
	if (str == NULL)
		return 0;
	if (key != NULL ) {
		klen = strlen(key);
		while (str != NULL) {
			if(strncasecmp(str, key, klen) == 0 ) {
				strncpy(line, str, len);
				return 1;
			}
			str = strtok(NULL, "\n\r");
		}
		return 0;
	}
	strncpy(line, str, len);
	return 1;
}

// write a line to a file
void write_line(const char *filename, const char *fmt, ...)
{
	va_list ap;
	int fd, bytes;
	char line[64];

	fd = open(filename, O_WRONLY);
	if (fd < 0)
		die(TRUE, "Could not open file for writing: %s", filename);
	va_start(ap, fmt);	// get variable argument list passed
	bytes = vsnprintf(line, 64, fmt, ap);
	if (bytes <= 0 || write(fd, line, bytes) < bytes) {
		close(fd);
		die(TRUE, "Could not write to file: %s", filename);
	}
	close(fd);
	va_end(ap);
}

void asus_leds(unsigned mode){
	switch (mode){
		case POWERSAVE:{
			write_line(asus_led_file_1,"1");
			write_line(asus_led_file_2,"0");
			break;
		}
		case PERFORMANCE:{ 
			write_line(asus_led_file_1,"0");
			write_line(asus_led_file_2,"1");
			break;
		}
		default:{
			write_line(asus_led_file_1,"0");
			write_line(asus_led_file_2,"0");
			break;
		}
	}
}

// set the current CPU policy
void policy_manager(int value)
{
	static int powersave_tries = DECAY;

	if (value != current_state) {
		if (value == POWERSAVE) {
			if (!powersave_tries) {
				if(asus_acpi_support) {
					asus_leds(POWERSAVE);
				}
				cpu_writer(POWERSAVE);
				errprintf("Setting speed to: powersave\n");
				setpriority(PRIO_PROCESS, 0, -11);
			} else {
				powersave_tries--;
			}
		} else {
			if(asus_acpi_support) {
				asus_leds(PERFORMANCE);
			}
			cpu_writer(PERFORMANCE);
			// We relinquish the CPU and continue work later
			setpriority(PRIO_PROCESS, 0, 10);
			sched_yield();
			errprintf("Setting speed to: performance\n");
			powersave_tries = DECAY;
			reset_times();
		}
	} else if (current_state == PERFORMANCE) {
		powersave_tries = DECAY;
	}
}

// gets the elapsed total time and elapsed idle time since it was last called
void get_times_24(unsigned long *total_elapsed, unsigned long *idle_elapsed)
{
	char what[32];
	unsigned long user_time, nice_time, system_time, idle_time, total_time;
	static unsigned long last_total_time = 0, last_idle_time = 0;
	unsigned found;
	char line[256];

	if (read_line("/proc/stat", line, sizeof line, "cpu ")) {
		sscanf(line, "%s %lu %lu %lu %lu", what, &user_time, &nice_time,
			   &system_time, &idle_time);
	} else
		die(FALSE, "Could not find entry for cpu0 in /proc/stat!?");

	if (count_idle == 0) {
		// count nice time as idle time
		idle_time += nice_time;
	}

	total_time = user_time + system_time + idle_time;
	*total_elapsed = total_time - last_total_time;
	last_total_time = total_time;
	*idle_elapsed = idle_time - last_idle_time;
	last_idle_time = idle_time;

	errprintf("time: %lu  idle: %lu nice: %lu  idle-:%lu\n", *total_elapsed, *idle_elapsed, nice_time, idle_time);
}

// this fucntion is for kernel 2.6, which gives much more accurate values
void get_times_26(unsigned long *total_elapsed, unsigned long *idle_elapsed)
{
	char what[32];
	static unsigned long total_time, total_idle,
			user_time, nice_time, system_time, idle_time,
			iowait_time, irq_time, softirq_time;
	static unsigned long last_total_time = 0, last_idle_time = 0;
	unsigned found;
	char line[256];

	if (read_line("/proc/stat", line, sizeof line, "cpu ")) {
		sscanf(line, "%s %lu %lu %lu %lu %lu %lu %lu", what, &user_time, &nice_time,
			   &system_time, &idle_time, &iowait_time, &irq_time, &softirq_time);
	} else
		die(FALSE, "Could not find entry for cpu in /proc/stat!?");

	total_time = user_time + nice_time + idle_time + system_time + iowait_time + irq_time + softirq_time;
	total_idle = idle_time + iowait_time;

	if (count_idle == 0) {
		// count nice time as idle time
		total_idle += nice_time;
	}

	*total_elapsed = total_time - last_total_time;
	last_total_time = total_time;
	*idle_elapsed = total_idle - last_idle_time;
	last_idle_time = total_idle;

	//errprintf("time: %lu  idle: %lu\n", total_elapsed, idle_elapsed);
	//errprintf("time: %lu %lu %lu %lu %lu %lu %lu\n", user_time, nice_time,
	//		   system_time, idle_time, iowait_time, irq_time, softirq_time);
	//errprintf("%lu - %lu %f\n", *total_elapsed, *idle_elapsed, (float) *idle_elapsed / *total_elapsed);

}



// resets the elapsed total time and elapsed idle time counters
void reset_times()
{
	unsigned long dummy1, dummy2;
	get_times(&dummy1, &dummy2);
}



// handles the periodic check of idle and setting CPU speed
void check_cpu(int s)
{
	static unsigned new_policy = PERFORMANCE;
	unsigned long elapsed_time, idle_time;
	float idle_ratio;

	// figure out what our current speed should be
	switch (mode) {
	case DYNAMIC:
		{
		// get the elapsed and idle times since we last checked
		get_times(&elapsed_time, &idle_time);
		if (elapsed_time > 0) {
			idle_ratio = (float) idle_time / elapsed_time;
			if (idle_ratio <= clock_up_idle)
				new_policy = PERFORMANCE;
			else if (idle_ratio >= clock_down_idle)
				new_policy = POWERSAVE;
			//errprintf("Idle ratio: %.2f, %.2f \n", idle_ratio, clock_up_idle);
		}
		break;
	}

	case MIN:
		new_policy = POWERSAVE;
		break;

	case MAX:
		new_policy = PERFORMANCE;
		break;
	}
	policy_manager(new_policy);
}

// handles the USR1 signal (stay at maximum performance)
void usr1_handler(int s)
{
	mode = MAX;
}

// handles the USR2 signal (stay at minimum performance)
void usr2_handler(int s)
{
	mode = MIN;
}

// handles the HUP signal (dynamically scale performance)
void hup_handler(int s)
{
	reset_times();
	mode = DYNAMIC;
}

int cpus_init()
{

	if (get_cpu_functions() == 0)
		return 0;
	current_state = initial_state = cpu_parser();
	errprintf("Initial state: %d\n", current_state);
	if (initial_state < 0)
		return 0;
	return 1;
}

// Checks if the asus acpi support is enabled, by checking for the presence of the typical asus acpi files
int get_asus_acpi_support()
{
	struct stat stat_buf;
	if(stat(asus_led_file_1, &stat_buf) == 0 &&
	   stat(asus_led_file_2, &stat_buf) == 0) {
		errprintf("get_asus_acpi_support: asus files found.\n");
		asus_leds(initial_state);
		errprintf("get_asus_acpi_support: state %d set.\n", initial_state);
		return 1;
	}
	errprintf("get_asus_acpi_support: asus files not found.\n");
	return 0;
}

int get_cpu_functions()
{
	struct stat stat_buf;
	if (!acpi_only) {
		if (stat("/sys/devices/system/cpu/cpu0/cpufreq/scaling_governor", &stat_buf) == 0) {
			governor_file = "/sys/devices/system/cpu/cpu0/cpufreq/scaling_governor";
			cpu_parser = cpu_parser_25;
			cpu_writer = cpu_writer_25;
			get_times = get_times_26;
			cpufreq_dir = "/sys/devices/system/cpu/cpu0/cpufreq";
			freq_count = cpu_get_frequencies();
			fprintf(stderr, "Linux 2.6 cpufreq (%s) detected\n", governor_file);
			return 1;
		}
		if (stat("/sysfs/devices/system/cpu/cpu0/cpufreq/scaling_governor", &stat_buf) == 0 ) {
			governor_file = "/sysfs/devices/system/cpu/cpu0/cpufreq/scaling_governor";
			cpu_parser = cpu_parser_25;
			cpu_writer = cpu_writer_25;
			get_times = get_times_26;
			cpufreq_dir = "/sysfs/devices/system/cpu/cpu0/cpufreq";
			freq_count = cpu_get_frequencies();
			fprintf(stderr, "Linux 2.6 cpufreq (%s) detected\n", governor_file);
			return 1;
		}
		if (stat("/proc/cpufreq", &stat_buf) == 0) {
			governor_file = "/proc/cpufreq";
			cpu_parser = cpu_parser_24;
			cpu_writer = cpu_writer_24;
			get_times = get_times_24;
			fprintf(stderr, "Linux 2.4 cpufreq (%s) detected\n", governor_file);
			return 1;
		}
		fprintf(stderr, "Linux kernel doesn't support cpufreq\n");
	}

	fprintf(stderr, "We'll try acpi throttling support...\n");
	if (stat("/proc/acpi/processor/CPU0/throttling",&stat_buf) == 0) {
		governor_file = "/proc/acpi/processor/CPU0/throttling";
		cpu_parser = cpu_acpi_24;
		cpu_writer = cpu_acpi_writer_24;
		get_times = get_times_24;
		fprintf(stderr,"Linux 2.4 ACPI throttling detected!\n");
		return 1;
	}
	fprintf(stderr,"Kernel doesn't support ACPI throttling\n");
	return 0;
}

unsigned cpu_acpi_24()
{
	static char line[256];

	if (read_line(governor_file, line, 255, "active state:")) {
		errprintf("cpu_acpi_24: got active state\n");
		if(strstr(line, "T0") > 0) {
			errprintf("cpu_acpi_24: got T0\n");
			return PERFORMANCE;
		} else {
			errprintf("cpu_acpi_24: got anything else\n");
			return POWERSAVE;
		}
	}

	errprintf("cpu_acpi_24: guess PERFORMANCE\n");
	return PERFORMANCE;
}

unsigned cpu_acpi_writer_24(unsigned state) {
	switch (state) {
		case PERFORMANCE: {
		  /* we put processor in T0 state */
			write_line(governor_file, "0");
			break;
		}
		case POWERSAVE: {
		  /* we put processor in low state, by default T3 state, 63% */
			char new_state[3];

			if (throttling_low > 99) {
				throttling_low = THROTTLING_LOW_DEFAULT;
			}
			snprintf(new_state, 3, "%u", throttling_low);
			write_line(governor_file, new_state);
			break;
		}
	}
	current_state = state;
}




unsigned cpu_parser_24()
{
	static char line[256];

	if( read_line(governor_file, line, 255, "CPU")) {
		if (strstr(line, "performance") > 0)
			return PERFORMANCE;
		else
			return POWERSAVE;
	} else {
		fprintf(stderr, "Could not find any CPU key in %s, disabling frequency control\n",
			governor_file);
		return -1;
	}
}

unsigned cpu_writer_24(unsigned state)
{

	switch (state) {
		case PERFORMANCE: {
			write_line(governor_file, "0%%0%%100%%performance\n");
			break;
		}
		case POWERSAVE: {
			write_line(governor_file, "0%%0%%100%%powersave\n");
			break;
		}
	}
	current_state = state;
}

unsigned cpu_parser_25()
{
	static char line[16];

	if( read_line(governor_file, line, 15, NULL)) {
		errprintf("cpu_parser: %s\n", line);
		if (strstr(line, "performance") != NULL)
			return PERFORMANCE;
		else
			return POWERSAVE;
	} else {
		fprintf(stderr, "Could not find any line in %s\n", governor_file);
		return -1;
	}
}

unsigned cpu_writer_25(unsigned state)
{

	switch (state) {
		case PERFORMANCE: {
			write_line(governor_file, "performance\n");
			break;
		}
		case POWERSAVE: {
			write_line(governor_file, "powersave\n");
			break;
		}
	}
	current_state = state;
}

cpu_term()
{
	cpu_writer(initial_state);
	errprintf("Setting initial state: %d\n", initial_state);
	if(asus_acpi_support) {
		write_line(asus_led_file_1,"0");
		write_line(asus_led_file_2,"0");
	}
}

inline unsigned cpu_get_state()
{
	return current_state;
}


//Get the list of frequencies
unsigned cpu_get_frequencies()
{
	char freq_file[256], freq_line[1024], freq_line_cpy[1024];
	char *tok;
	unsigned i;
	struct stat stat_buf;

	sprintf(freq_file, "%s/%s", cpufreq_dir, "scaling_available_frequencies");
	if (stat(freq_file, &stat_buf) == 0 && read_line(freq_file, freq_line, 1024, NULL)) {
		// First count the number of frequencies
		strcpy(freq_line_cpy, freq_line);
		if(strtok(freq_line_cpy, " ") != NULL) freq_count = 1;
		while(strtok(NULL, " ") != NULL)
			freq_count++;
		if (freq_count == 0) return 0;
		// Now we assing the values to the table
		frequencies_table = (long *) malloc(sizeof(long) * freq_count);
		for (i=0; i < freq_count; i++) {
			if (i == 0) tok = strtok(freq_line, " ");
			else tok = strtok(NULL, " ");
			frequencies_table[i] = atol(tok);
		}
		if (do_min_freq) set_freq_min_index();
		return freq_count;
	} else {
		if (do_min_freq) fprintf(stderr, "Frequencies not available in %s\n", freq_file);
		return 0;
	}

}

// Get the index of the minimum frequency
int set_freq_min_index()
{
	int i = 0;
	long freq;
	char file[256], frequency[20];

	freq = (long) frequencies_table[freq_count - 1] * min_freq_float;
	while (i < freq_count && frequencies_table[i]< freq ) i++;
	sprintf(file, "%s/%s", cpufreq_dir, "scaling_min_freq");
	sprintf(frequency, "%d", frequencies_table[i]);
	fprintf(stderr,"Will set the minimum frequency to: %s \n", frequency);
	write_line(file, frequency);
	return i;
}


