/* statistics.n-grams.c - n-gram counts and various related stats
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank-interface.h"
#include "common.statistics.n-grams.h"

/* Global variables */
static float *slft_std;
static float *bift_std;
static float *trift_std;
static stats *text_stats = NULL;

static int already_running = FALSE; /* Only one copy allowed at at time */

static char *slft_display_titles[] = {"Letter", STR_FREQUENCY, STR_STANDARD_FREQUENCY};
static char *bift_display_titles[] = {"First Letter", "Second Letter", STR_FREQUENCY, STR_STANDARD_FREQUENCY};
static char *trift_display_titles[] = {"First Letter", "Second Letter", "Third Letter", STR_FREQUENCY, STR_STANDARD_FREQUENCY};

/* Fn Declarations */
static void cb_save_slft(GtkWidget *widget, gpointer gdata);
static void cb_save_bift(GtkWidget *widget, gpointer gdata);
static void cb_save_trift(GtkWidget *widget, gpointer gdata);
static void cb_sort_ft_display(GtkWidget *widget, gint col, gpointer data);
static int cb_destroy_stats_display(GtkWidget *widget, gpointer gdata);
static GtkWidget *make_stats_display_window(stats *s);

/* Plugin Interface */
/* ---------------- */

const char name[] = "statistics.n-grams";
const char version[] = VERSION;
const char author[] = "Matthew Russell";
const char description[] = "n-gram counts and various related statistics.";
const int interface_version = 1;
const int plugin_type = PLUGIN_FLOATING;
const char menu_string[] = "/Statistics/n-grams";

int boot(void) {
    /* Load the standard frequency tables */
    /* Should use global resource when implemented */
    slft_std = load_slft_std(DEFAULT_SLFT);
    bift_std = load_bift_std(DEFAULT_BIFT);
    trift_std = load_trift_std(DEFAULT_TRIFT);
    return PLUGIN_BOOT_OK;
}

GtkWidget *make_widget(char *text) {
    if (already_running) 
	return NULL;
    already_running = TRUE;
    
    if (text_stats)
	free_stats(text_stats);
    text_stats = make_stats(text, slft_std, bift_std, trift_std);
    return make_stats_display_window(text_stats);
}

/* Plugin Implementation */
/* --------------------- */

/* Make a tabular display of slft using a GtkClist */
/* Displays zero values */
static GtkWidget *make_slft_display(float *slft) {
    GtkWidget *clist;
    int i;
    char *row_data[3];
    char letter[2], value[30], value2[30];
    
    clist = gtk_clist_new_with_titles(3, slft_display_titles);
    gtk_clist_set_shadow_type(GTK_CLIST(clist), GTK_SHADOW_ETCHED_IN);
    gtk_signal_connect(GTK_OBJECT( clist),
                       "click-column",
                       GTK_SIGNAL_FUNC(cb_sort_ft_display),
                       NULL);

    for (i = 'A'; i <= 'Z'; i++) {
	sprintf(letter, "%c", (char) i);
	sprintf(value, "%.8f", slft[i]);
	sprintf(value2, "%.8f", slft_std[i]);
	row_data[0] = letter; row_data[1] = value; row_data[2] = value2;
	gtk_clist_append(GTK_CLIST(clist), row_data);
    }
    gtk_clist_columns_autosize(GTK_CLIST(clist));
    return clist;
}

/* Make a tabular display of bift using a GtkClist */
static GtkWidget *make_bift_display(float *bift) {
    GtkWidget *clist;
    int i, j;
    char *row_data[4];
    char letter1[2], letter2[2], value[30], value2[30];
    float fvalue;
    clist = gtk_clist_new_with_titles(4, bift_display_titles);
    gtk_clist_set_shadow_type(GTK_CLIST(clist), GTK_SHADOW_ETCHED_IN);
    gtk_signal_connect(GTK_OBJECT( clist),
                       "click-column",
                       GTK_SIGNAL_FUNC(cb_sort_ft_display),
                       NULL);

    for (i = 'A'; i <= 'Z'; i++) {    
	for (j = 'A'; j <= 'Z'; j++) {
	    fvalue = (bift + 26 * i)[j];
	    if (fvalue > 0.0) {
		sprintf(letter1, "%c", (char) i);
		sprintf(letter2, "%c", (char) j);
		sprintf(value, "%.8f", fvalue);
		sprintf(value2, "%.8f", (bift_std + 26 * i)[j]);
		row_data[0] = letter1; row_data[1] = letter2;  row_data[2] = value; row_data[3] = value2;
		gtk_clist_append(GTK_CLIST(clist), row_data);
	    }
	}
    }
    gtk_clist_columns_autosize(GTK_CLIST(clist));
    return clist;
}

/* Make a tabular display of trift using a GtkClist */
static GtkWidget *make_trift_display(float *trift) {
    GtkWidget *clist;
    float fvalue;
    int i, j, k;
    char *row_data[5];
    char letter1[2], letter2[2], letter3[2], value[30], value2[30];

    clist = gtk_clist_new_with_titles(5, trift_display_titles);
    gtk_clist_set_shadow_type(GTK_CLIST(clist), GTK_SHADOW_ETCHED_IN);
    gtk_signal_connect(GTK_OBJECT( clist),
                       "click-column",
                       GTK_SIGNAL_FUNC(cb_sort_ft_display),
                       NULL);

    for (i = 'A'; i <= 'Z'; i++) {    
	for (j = 'A'; j <= 'Z'; j++) {
	    for (k = 'A'; k <= 'Z'; k++) {
		fvalue = (trift + 26 * 26 * i + 26 * j)[k];
		if (fvalue > 0.0) {
		    sprintf(letter1, "%c", (char) i);
		    sprintf(letter2, "%c", (char) j);
		    sprintf(letter3, "%c", (char) k);
		    sprintf(value, "%.8f", fvalue);
		    sprintf(value2, "%.8f", (trift_std + 26 * 26 * i + 26 * j)[k]);
		    row_data[0] = letter1; row_data[1] = letter2;  row_data[2] = letter3; row_data[3] = value; row_data[4] = value2;
		    gtk_clist_append(GTK_CLIST(clist), row_data);
		}
	    }
	}
    }
    gtk_clist_columns_autosize(GTK_CLIST(clist));
    return clist;
}

/* Add a floating point entry to the stats summary table */
static void add_clist_entry_f(GtkWidget *clist, char *name, float value) {
    char str_value[BUFFER_SIZE], *row_data[2];
    sprintf(str_value, "%f", value);
    row_data[0] = name; row_data[1] = str_value;
    gtk_clist_append(GTK_CLIST(clist), row_data);
}

/* Add an integer entry to the stats summary table */
static void add_clist_entry_i(GtkWidget *clist, char *name, int value) {
    char str_value[BUFFER_SIZE], *row_data[2];
    sprintf(str_value, "%d", value);
    row_data[0] = name; row_data[1] = str_value;
    gtk_clist_append(GTK_CLIST(clist), row_data);
}

static GtkWidget *make_stats_summary(stats *the_stats) {
    GtkWidget *box;
    GtkWidget *vbox;
    GtkWidget *clist;
    GtkWidget *save_button_slft;
    GtkWidget *save_button_bift;
    GtkWidget *save_button_trift;
    int column;
    
    box = gtk_hbox_new(FALSE, 0);
    vbox = gtk_vbox_new(FALSE, 0);
    clist = gtk_clist_new(2);
    gtk_clist_set_shadow_type(GTK_CLIST(clist), GTK_SHADOW_ETCHED_IN);

    save_button_slft = gtk_button_new_with_label("Save as Default Unigram Frequencies");
    gtk_signal_connect(GTK_OBJECT(save_button_slft), "pressed", GTK_SIGNAL_FUNC(cb_save_slft), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), save_button_slft, FALSE, FALSE, 0);
    gtk_widget_show(save_button_slft);

    save_button_bift = gtk_button_new_with_label("Save as Default Bigram Frequencies");
    gtk_signal_connect(GTK_OBJECT(save_button_bift), "pressed", GTK_SIGNAL_FUNC(cb_save_bift), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), save_button_bift, FALSE, FALSE, 0);
    gtk_widget_show(save_button_bift);

    save_button_trift = gtk_button_new_with_label("Save as Default Trigram Frequencies");
    gtk_signal_connect(GTK_OBJECT(save_button_trift), "pressed", GTK_SIGNAL_FUNC(cb_save_trift), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), save_button_trift, FALSE, FALSE, 0);
    gtk_widget_show(save_button_trift);

    add_clist_entry_f(clist, "Unigram Error:", the_stats->slft_error);
    add_clist_entry_f(clist, "Bigram Error:", the_stats->bift_error);
    add_clist_entry_f(clist, "Trigram Error:", the_stats->trift_error);
    add_clist_entry_f(clist, "Total Error:", the_stats->total_error);
    add_clist_entry_i(clist, "Letter Count:", the_stats->letter_count);
    add_clist_entry_f(clist, "IC:", the_stats->ic);
    add_clist_entry_f(clist, "Entropy:", the_stats->entropy);
    add_clist_entry_f(clist, "Efficiency:", the_stats->efficiency);
    add_clist_entry_f(clist, "Redundancy:", the_stats->redundancy);

    for (column = 0; column <= 1; column++)
	gtk_clist_set_column_width(GTK_CLIST(clist), column, gtk_clist_optimal_column_width(GTK_CLIST(clist), column));


    gtk_box_pack_start(GTK_BOX(vbox), clist, TRUE, TRUE, 0);
    gtk_widget_show(clist);
    gtk_widget_show(box); gtk_widget_show(vbox);
    gtk_box_pack_start(GTK_BOX(box), vbox, TRUE, TRUE, 0);

    return box;
}

static GtkWidget *make_stats_display_window(stats *s) {
    float *slft = s->slft;
    float *bift = s->bift;
    float *trift = s->trift;
    GtkWidget *dialog;
    GtkWidget *summary_page = make_stats_summary(s);
    GtkWidget *clist_slft = make_slft_display(slft);
    GtkWidget *clist_bift = make_bift_display(bift);
    GtkWidget *clist_trift = make_trift_display(trift);
    GtkWidget *window_main;
    GtkWidget *window_slft;
    GtkWidget *window_bift;
    GtkWidget *window_trift;
    GtkWidget *button;
    GtkWidget *notebook;
    window_main = gtk_scrolled_window_new(NULL, NULL);
    window_slft = gtk_scrolled_window_new(NULL, NULL);
    window_bift = gtk_scrolled_window_new(NULL, NULL);
    window_trift = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(window_main),  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(window_slft),  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);    
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(window_bift),  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(window_trift), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(window_main), summary_page);
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(window_slft), clist_slft);
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(window_bift), clist_bift);    
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(window_trift), clist_trift);    
    gtk_widget_show(window_main);
    gtk_widget_show(window_slft);
    gtk_widget_show(window_bift);
    gtk_widget_show(window_trift);
    dialog  = gtk_dialog_new();
    gtk_widget_set_usize(dialog, 440, 500);
    
    gtk_window_set_title(GTK_WINDOW(dialog), "Text Statistics");
    button = gtk_button_new_with_label("Dismiss");
    gtk_signal_connect_object(GTK_OBJECT(button),
			      "clicked",
			      (GtkSignalFunc) gtk_widget_destroy,
			      GTK_OBJECT(dialog));
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), button, FALSE, FALSE, 0);
    gtk_widget_show(button);
    notebook = gtk_notebook_new();
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), window_main, gtk_label_new("Summary"));
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), window_slft, gtk_label_new("Unigrams"));
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), window_bift, gtk_label_new("Bigrams"));
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), window_trift, gtk_label_new("Trigrams"));
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), notebook, TRUE, TRUE, 0);
    gtk_widget_show(clist_slft);
    gtk_widget_show(clist_bift);
    gtk_widget_show(clist_trift);
    gtk_widget_show(summary_page);
    gtk_widget_show(notebook);
    gtk_signal_connect(GTK_OBJECT(dialog), "destroy", GTK_SIGNAL_FUNC(cb_destroy_stats_display), NULL);
    return dialog;
}

/* Callbacks */
/* --------- */

static void cb_save_slft(GtkWidget *widget, gpointer gdata) {
    do_save_slft(text_stats->slft);
    free(slft_std);
    slft_std = load_slft_std(DEFAULT_SLFT);
}

static void cb_save_bift(GtkWidget *widget, gpointer gdata) {
    do_save_bift(text_stats->bift);
    free(bift_std);
    bift_std = load_bift_std(DEFAULT_BIFT);
}

static void cb_save_trift(GtkWidget *widget, gpointer gdata) {
    do_save_trift(text_stats->trift);
    free(trift_std);
    trift_std = load_trift_std(DEFAULT_TRIFT);
}

static void cb_sort_ft_display(GtkWidget *widget, gint col, gpointer gdata) {
    gtk_clist_set_sort_column(GTK_CLIST(widget), col);
    /* Sort up for the frequencies and down for the letters */
    if (!strcmp(STR_FREQUENCY, gtk_clist_get_column_title(GTK_CLIST(widget), col)) ||
        !strcmp(STR_STANDARD_FREQUENCY, gtk_clist_get_column_title(GTK_CLIST(widget), col)))
	gtk_clist_set_sort_type(GTK_CLIST(widget), GTK_SORT_DESCENDING);
    else
	gtk_clist_set_sort_type(GTK_CLIST(widget), GTK_SORT_ASCENDING);
    gtk_clist_sort(GTK_CLIST(widget));
}

static int cb_destroy_stats_display(GtkWidget *widget, gpointer gdata) {
    if (already_running == FALSE)
	g_warning("Uniqueness problem with stats display.");
    already_running = FALSE;
    return TRUE;
}
