/*
 * PT
 * Copyright (C) 2002  The Printing Tool Developers
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * printsys.c
 */

#include <cups/language.h>

#include "support.h"
#include "printsys.h"


/*
 * p_get_jobs () - get print jobs
 */

gint p_get_jobs (print_job_t **j, const gchar *host, gint port,
	const gchar *username, http_encryption_t ssl, gint allocation,
	const gchar *filter_dest)
{
	ipp_t *request, *response;
	ipp_attribute_t *attr;
	cups_lang_t *language;
	http_t *http;
	gint n, rank, id, size, copies, priority;
	gchar *path, *dest, *title, *user, *format, *default_user;
	ipp_jstate_t state;
	time_t creation_time, processing_time;
	print_job_t *q = NULL;
	gpointer a;
	
	/*
	 * save default user
	 */

	default_user = g_strdup (cupsUser ());
	cupsSetUser (username);

	/*
     * connect to host
	 */
	
	http = httpConnectEncrypt (host, port, ssl);

	/*
	 * if no connection
	 */

	if (http == NULL)
	{
		cupsSetUser (default_user);
		g_free (default_user);
		*j = NULL;
		return -1;
	}
	
	/*
	 * encryption
	 */

	httpEncryption (http, ssl);

	/*
	 * initial values
	 */

	response = NULL;
	n = 0;
	rank = 0;
	
	/*
 	* build IPP_GET_JOBS request
 	*/
    
	request = ippNew ();
	request->request.op.operation_id = IPP_GET_JOBS;
	request->request.op.request_id = 1;
	language = cupsLangDefault ();
	
	ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_CHARSET,
		"attributes-charset", NULL, cupsLangEncoding (language));

	ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_LANGUAGE,
		"attributes-natural-language", NULL, language->language);

	if (strcmp (filter_dest, NO_ARG) == 0)
	{
		path = g_strdup_printf ("ipp://%s/jobs", host);
		ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_URI,
		"job-uri", NULL, path);
	}
	else
	{
		path = g_strdup_printf ("ipp://%s/printers/%s", host, filter_dest);
		ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_URI,
		"printer-uri", NULL, path);
	}
	g_free (path);

	ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_NAME,
		"requesting-user-name", NULL, username);

	/*
	 * send request and get response
	 */
	
	response = cupsDoRequest (http, request, "/");
	
	/*
	 * if no response
	 */
		
	if (response == NULL)
	{
		cupsSetUser (default_user);
		g_free (default_user);
		*j = NULL;
		httpClose (http);
		return -1;
	}
	
	/*
	 * if error
	 */
	
	if (response->request.status.status_code > IPP_OK_CONFLICT)
	{
		ippDelete(response);
		cupsSetUser (default_user);
		g_free (default_user);
		*j = NULL;
		httpClose (http);
		return -1;
	}
	
	attr = response->attrs;
	
	while (attr != NULL)
	{
		
		/*
		 * skip leading attributes
		 */
		
		while ((attr != NULL) && (attr->group_tag != IPP_TAG_JOB))
			attr = attr->next;
		
		if (attr == NULL) 
			break;				

		/*
		 * set default attributes
		 */
		
		id = 0;
		dest = NO_ARG;
		title = NO_ARG;
		user = NO_ARG;
		format = NO_ARG;
		state = IPP_JOB_PENDING;
		size = 0;
		copies = 1;
		priority = 50;
		creation_time = 0;
		processing_time = 0;

		/*
		 * get attributes
		 */
		
		while ((attr != NULL) && (attr->group_tag == IPP_TAG_JOB))
		{
			if (strcmp (attr->name, "job-priority") == 0)
			{
				if (attr->value_tag == IPP_TAG_INTEGER)
					priority = attr->values[0].integer;
			}
			else if (strcmp (attr->name, "job-id") == 0)
			{
				if (attr->value_tag == IPP_TAG_INTEGER)
					id = attr->values[0].integer;
			}
			else if (strcmp (attr->name, "job-originating-user-name") == 0)
			{
				if (attr->value_tag == IPP_TAG_NAME)
					user = attr->values[0].string.text;
			}
			else if (strcmp (attr->name, "job-name") == 0)
			{
				if (attr->value_tag == IPP_TAG_NAME)
					title = attr->values[0].string.text;
			}
			else if (strcmp (attr->name, "document-format") == 0)
			{
				if (attr->value_tag == IPP_TAG_MIMETYPE)
					format = attr->values[0].string.text;
			}
			else if (strcmp (attr->name, "copies") == 0)
			{
				if (attr->value_tag == IPP_TAG_INTEGER)
					copies = attr->values[0].integer;
			}
			else if (strcmp (attr->name, "job-printer-uri") == 0)
			{
				if (attr->value_tag == IPP_TAG_URI)
					if (strrchr (attr->values[0].string.text, '/') != NULL)
					{
						dest = strrchr (attr->values[0].string.text, '/');
						dest = &dest[1];
					}
			}
			else if (strcmp (attr->name, "job-state") == 0)
			{
				if (attr->value_tag == IPP_TAG_ENUM)
					state = attr->values[0].integer;
			}
			else if (strcmp (attr->name, "job-k-octets") == 0)
			{
				if (attr->value_tag == IPP_TAG_INTEGER)
					size = attr->values[0].integer;
			}
			else if (strcmp (attr->name, "time-at-creation") == 0)
			{
				if (attr->value_tag == IPP_TAG_INTEGER)
					creation_time = attr->values[0].integer;
			}
			else if (strcmp (attr->name, "time-at-processing") == 0)
			{
				if (attr->value_tag == IPP_TAG_INTEGER)
					processing_time = attr->values[0].integer;
			}
			
			attr = attr->next;
		}

		/*
		 * allocate memory
		 */

		if ((n == 0) || (n >= allocation))
		{
			allocation += J_PRE_ALLOCATION;
			if ((a = g_realloc (q, sizeof (print_job_t) * allocation)) == NULL)
			{
				p_free_jobs (q, n);
				cupsSetUser (default_user);
				g_free (default_user);
				*j = NULL;
				httpClose (http);
				return -1;
			}
			q = a;
		}

		/*
		 * copy arguments
		 */

		if (state == IPP_JOB_PROCESSING)
			q[n][0] = g_strdup (gettext("active"));
		else
			q[n][0] = g_strdup_printf  ("%d", ++rank);
		q[n][1] = g_strdup_printf ("%d", priority);
		q[n][2] = g_strdup_printf ("%d", id);
		q[n][3] = g_strdup (user);
		q[n][4] = g_strdup (title);
		q[n][5] = g_strdup (format);
		q[n][6] = g_strdup_printf ("%d", copies);
		q[n][7] = g_strdup (dest);
		q[n][8] = p_jstate (state);
		q[n][9] = g_strdup_printf ("%d", size);
		q[n][10] = g_strdup (p_time (&creation_time));
		q[n++][11] = g_strdup (p_time (&processing_time));
	}
	
	/*
	 * free pre-allocated memory 
	 */

	if (n != 0 )
		if (n < allocation)
			q = g_realloc (q, sizeof (print_job_t) * n);

	/*
	 * delete response
	 */

	ippDelete (response);

	/*
	 * call by reference
	 */

	*j = q;

	/*
	 * close connection
	 */
	
	httpClose (http);

	/*
	 * restore default user
	 */

	cupsSetUser (default_user);
	g_free (default_user);

	return n;
}

/*
 * p_time () - get hh:mm time string from calendar time
 */

gchar *p_time (const time_t *t)
{
	gchar *s;
	
	if (*t == 0)
		return NO_TIME;
	else
	{
		s = ctime (t);
		s[16] = 0;
		return (&s[11]);
	}
}

/*
 * p_jstate () - get job state in string format
 */

gchar *p_jstate (ipp_jstate_t state)
{
	switch (state)
	{
		case IPP_JOB_PENDING : return gettext("pending");
		case IPP_JOB_HELD : return gettext("held");
		case IPP_JOB_PROCESSING : return gettext("processing");
		case IPP_JOB_STOPPED : return gettext("stopped");
		case IPP_JOB_CANCELLED : return gettext("cancelled");
		case IPP_JOB_ABORTED : return gettext("aborted");
		case IPP_JOB_COMPLETED : return gettext("completed");
	}
	return NULL;
}

/*
 * p_free_jobs () - free memory allocated for print_job_t
 */

void p_free_jobs (print_job_t *j, gint n)
{
	gint i, w;
	
	if (j != NULL)
	{
		for (i = 0; i < n; i++)
			for (w = 0; w <= 11; w++)
				if (w != 8)
					g_free (j[i][w]);
	}
	g_free (j);
	return;
}

/*
 * p_multi_request ()
 */
 
void p_multi_request(const gint *selected_row, gint r,
	const gchar *host, gint port, const gchar *username,
	http_encryption_t ssl, ipp_op_t w,
	const multi_arg_t multi_arg)
{
	ipp_t *request, *response;
	cups_lang_t *language;
	http_t *http;
	gint i;
	gchar *path, *default_user;
	cups_option_t *options;

	/*
	 * save default user
	 */

	default_user = g_strdup (cupsUser ());
	cupsSetUser (username);

	/*
	 * connect to host
	 */

	http = httpConnectEncrypt (host, port, ssl);

	/*
	 * if no connection
	 */

	if (http == NULL)
	{
		cupsSetUser (default_user);
		g_free (default_user);
		return;
	}

	httpEncryption (http, ssl);

	/*
	 * build request
	 */
    
    for (i = 0; i < r; i++)
    {
	    request = ippNew ();
		request->request.op.operation_id = w;
		request->request.op.request_id = 2;
		language = cupsLangDefault ();

		ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_CHARSET,
			"attributes-charset", NULL, cupsLangEncoding (language));

		ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_LANGUAGE,
			"attributes-natural-language", NULL, language->language);

		path = g_strdup_printf ("ipp://%s/jobs/%d",host, selected_row[i]);
		ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_URI,
			"job-uri", NULL, path);
		g_free (path);
	
		ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_NAME,
			"requesting-user-name", NULL, username);

		if (w == CUPS_MOVE_JOB)
		{
			path = g_strdup_printf ("ipp://%s/printers/%s",host,
				multi_arg.dest);
			ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_URI,
				"job-printer-uri", NULL, path);
			g_free (path);	
		}
		else if (w == IPP_SET_JOB_ATTRIBUTES)
		{
			options = g_malloc (sizeof (cups_job_t));
			options->name = g_strdup ("job-priority");
			options->value = g_strdup_printf ("%d", multi_arg.priority[i]);
			cupsEncodeOptions (request, 1, options);
			g_free (options->name);
			g_free (options->value);
			g_free (options);
		}
		
		/*
		 * send request and get response
		 */
	
		response = cupsDoRequest (http, request, "/jobs/");

		if (response == NULL)
		{
			cupsSetUser (default_user);
			g_free (default_user);
			return;
		}

		/*
		 * delete response
		 */
	 
		ippDelete (response);
	}

	/*
	 * close connection
	 */
	
	httpClose (http);

	/*
	 * restore default user
	 */

	cupsSetUser (default_user);
	g_free (default_user);

	return;
}

/*
 * p_get_dests () - get printers and classes
 */

gint p_get_dests (dest_t **dest, const gchar *host, gint port,
	const gchar *username, http_encryption_t ssl, gint allocation)
{
	ipp_t *request, *response;
	ipp_attribute_t *attr;
	cups_lang_t *language;
	http_t *http;
	gint i, n;
	gchar *name, *default_user;
	dest_t *q = NULL;
	gpointer a;

	/*
	 * save default user
	 */

	default_user = g_strdup (cupsUser ());
	cupsSetUser (username);

	/*
     * connect to host
	 */

	http = httpConnectEncrypt (host, port, ssl);

	/*
	 * if no connection
	 */

	if (http == NULL)
	{
		cupsSetUser (default_user);
		g_free (default_user);
		*dest = NULL;
		return -1;
	}
	
	/*
	 * encryption
	 */

	httpEncryption (http, ssl);

	/*
	 * initial values
	 */

	response = NULL;
	n = 0;
	
	for (i = 0; i <= 1; i++)
	{

		/*
	 	* build IPP_GET_JOBS or IPP_GET_CLASSES request
	 	*/

		request = ippNew ();
		request->request.op.operation_id =
			((i == 0) ? CUPS_GET_PRINTERS : CUPS_GET_CLASSES);
		request->request.op.request_id = 3;
		language = cupsLangDefault ();

		ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_CHARSET,
			"attributes-charset", NULL, cupsLangEncoding (language));

		ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_LANGUAGE,
			"attributes-natural-language", NULL, language->language);

		ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_NAME,
			"requesting-user-name", NULL, username);

		ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_URI,
			"requested-attributes", NULL, "printer-name");

		/*
		 * send request and get response
		 */

		response = cupsDoRequest (http, request, "/");

		/*
		 * if no response
		 */

		if (response == NULL)
		{
			cupsSetUser (default_user);
			g_free (default_user);
			*dest = NULL;
			httpClose (http);
			return -1;
		}

		/*
		 * if error
		 */

		if (response->request.status.status_code > IPP_OK_CONFLICT)
		{
			ippDelete(response);
			cupsSetUser (default_user);
			g_free (default_user);
			*dest = NULL;
			httpClose (http);
			return -1;
		}

		attr = response->attrs;

		while (attr != NULL)
		{
		
			/*
		 	* skip leading attributes
			 */
		
			while ((attr != NULL) && (attr->group_tag != IPP_TAG_PRINTER))
				attr = attr->next;
		
			if (attr == NULL) 
				break;
		
			/*
			 * set default attribute
			 */
		
			name = NO_ARG;

			/*
			 * get attributes
			 */
		
			while ((attr != NULL) && (attr->group_tag == IPP_TAG_PRINTER))
			{
				if (strcmp (attr->name, "printer-name") == 0)
					if (attr->value_tag == IPP_TAG_NAME)
						name = attr->values[0].string.text;

				attr = attr->next;
			}
				
			/*
			 * allocate memory
			 */

			if ((n == 0) || (n >= allocation))
			{
				allocation += DEST_PRE_ALLOCATION;
				if ((a = g_realloc (q, sizeof (dest_t) * allocation)) == NULL)
				{
					for (i = 0; i < n; i++)
						g_free (q[i][0]);
					g_free (q);
					cupsSetUser (default_user);
					g_free (default_user);
					*dest = NULL;
					httpClose (http);
					return -1;
				}
				q = a;
			}

			/*
			 * copy argument
			 */

			q[n++][0] = g_strdup (name);
		}

		/*
		 * delete response
		 */

		ippDelete (response);
	}

	/*
	 * free pre-allocated memory 
	 */

	if (n != 0 )
		if (n < allocation)
			q = g_realloc (q, sizeof (dest_t) * n);

	/*
	 * call by reference
	 */

	*dest = q;

	/*
	 * close connection
	 */
	
	httpClose (http);
	
	/*
	 * restore default user
	 */

	cupsSetUser (default_user);
	g_free (default_user);

	return n;
}

