/*
 * Danpei -- a GTK+ based Image Viewer
 * Copyright (C) 2001-2003 Shinji Moiino
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/* dialog.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "config.h"
#include "dialog.h"
#include "intl.h"
#include "version.h"

/* Static variables. */
static gint  app_msg_lines[] = { 0, 3, 6, 9 };
static gchar *app_message[]  = { N_("  File will be removed.  ")          ,
                                 N_("  Are you sure ?  ")                 ,
                                 ""                                       ,
                                 N_("  File will be overwrriten.  ")      ,
                                 N_("  Are you sure ?  ")                 ,
                                 ""                                       ,
                                 N_("  Directory will be removed.  ")     ,
                                 N_("  Are you sure ?  ")                 ,
                                 ""                                       ,
                                 N_("  Copy the file, for the failure  ") ,
                                 N_("  while removing dest file.  ")      ,
                                 ""                                       
                               };

static gint  err_msg_lines[] = {  0,  1,  3,  5,  7,  9, 11, 13, 15, 17,
                                 19, 21, 23, 25, 27, 29, 31, 33, 35, 37,
                                 40
                               };
static gchar *err_message[]  = { "",
                                 N_("  File not exists.  ")                ,
                                 ""                                        ,
                                 N_("  Permission denied.  ")              ,
                                 ""                                        ,
                                 N_("  File already exists.  ")            ,
                                 ""                                        ,
                                 N_("  No space left on this device.  ")   ,
                                 ""                                        ,
                                 N_("  Read only file system.  ")          ,
                                 ""                                        ,
                                 N_("  File name is too long.  ")          ,
                                 ""                                        ,
                                 N_("  Rename failed.  ")                  ,
                                 ""                                        ,
                                 N_("  Remove failed.  ")                  ,
                                 ""                                        ,
                                 N_("  Change directory failed.  ")        ,
                                 ""                                        ,
                                 N_("  Open failed.  ")                    ,
                                 ""                                        ,
                                 N_("  Paste failed.  ")                   ,
                                 ""                                        ,
                                 N_("  Read failed.  ")                    ,
                                 ""                                        ,
                                 N_("  Write failed.  ")                   ,
                                 ""                                        ,
                                 N_("  Directory already exists.  ")       ,
                                 ""                                        ,
                                 N_("  Directory not exists.  ")           ,
                                 ""                                        ,
                                 N_("  mkdir failed.  ")                   ,
                                 ""                                        ,
                                 N_("  Directory not empty.  ")            ,
                                 ""                                        ,
                                 N_("  rmdir failed.  ")                   ,
                                 ""                                        ,
                                 N_("  Internal error.  ")                 ,
                                 N_("  -- Proccess not excuted.  ")        ,
                                 ""                                        ,
                                 N_("  Rotation failed.  ")
                               };

/* Static functions declaration. */
static void dialog_cb_ok_button                 (GtkWidget*            ,
                                                 gpointer                );

static void dialog_cb_cancel_button             (GtkWidget*            ,
                                                 gpointer                );

static void dialog_cb_skip_button               (GtkWidget*            ,
                                                 gpointer                );

static void dialog_cb_ok_all_button             (GtkWidget*            ,
                                                 gpointer                );

static void dialog_create_common                (gchar*                ,
                                                 gchar*                ,
                                                 gchar**               ,
                                                 AppDialogType         ,
                                                 AppDialogDefaultButton,
                                                 ManageDialog*           );

static void dialog_cb_progressbar_dialog_cancel (GtkWidget*            ,
                                                 gpointer                );

/* Function definitions. */
/*
 * @dialog_message_dialog_create
 *
 *  Create an message dialog.
 *
 */
AppDialogResult dialog_message_dialog_create(
                                      AppMessageNum          app_msg_no ,
                                      gchar                  *user_msg  ,
                                      AppDialogType          type       ,
                                      AppDialogDefaultButton button     ,
                                      GtkWindow              *window      ) {
  ManageDialog md;

  md.window = window;
  dialog_create_common(_("MESSAGE"), user_msg,
                       &(app_message[app_msg_lines[app_msg_no]]), 
                       type, button, &md);

  return md.is_ok;
}

/*
 * @dialog_error_dialog_create
 *
 *  Create an error dialog.
 *
 */
AppDialogResult dialog_error_dialog_create (AppErrorNum app_err_no ,
                                            gchar       *user_msg  ,
                                            GtkWindow   *window      ) {
  ManageDialog md;

  md.window = window;
  dialog_create_common(_("ERROR"), user_msg,
                       &(err_message[err_msg_lines[app_err_no]]),
                       APP_OK_ONLY_DIALOG, APP_OK_BUTTON, &md);

  return md.is_ok;
}

/*
 * @dialog_progressbar_dialog_create
 *
 *  Create a dialog with a progressbar.
 *
 */
void dialog_progressbar_dialog_create (gchar             *message[], 
                                       ProgressbarDialog *pd         ) {
  GtkWidget *label;
  GtkWidget *vbox;
  GtkWidget *frame;
  GtkWidget *button;

  pd->dialog = gtk_dialog_new();
  gtk_window_set_title (GTK_WINDOW(pd->dialog), _("MESSAGE"));
  gtk_window_set_policy(GTK_WINDOW(pd->dialog), FALSE, FALSE, FALSE);
  gtk_window_set_position(GTK_WINDOW(pd->dialog), GTK_WIN_POS_CENTER);
  gtk_window_set_transient_for(GTK_WINDOW(pd->dialog), 
                               GTK_WINDOW(pd->window) ); 
  gtk_signal_connect(GTK_OBJECT(pd->dialog), "destroy", 
                     GTK_SIGNAL_FUNC(dialog_cb_progressbar_dialog_cancel),
                     pd);
  gtk_widget_show(pd->dialog);

  frame = gtk_frame_new(NULL);
  gtk_container_border_width(GTK_CONTAINER(frame), 8);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(pd->dialog)->vbox), frame, 
                     TRUE, TRUE, 0);
  gtk_widget_show(frame);

  vbox = gtk_vbox_new(TRUE, 2);
  gtk_container_border_width(GTK_CONTAINER(vbox), 4);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_widget_show(vbox);

  while(strcmp(*message, "") != 0) {
    label = gtk_label_new(_(*message));
    gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
    gtk_widget_show(label);
    message++;
  }

  pd->bar = gtk_progress_bar_new();
  gtk_progress_set_show_text(GTK_PROGRESS(pd->bar), TRUE);
  gtk_box_pack_start(GTK_BOX(vbox), pd->bar, TRUE, TRUE, 0);
  gtk_widget_show(pd->bar);

  button = gtk_button_new_with_label(_("  Cancel  "));
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(pd->dialog)->action_area),
                     button, FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(dialog_cb_progressbar_dialog_cancel),
                     pd);
  /* Set the GtkButton GTK_CAN_DEFAULT, a boader can be around 
   * the button.
   */
  GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default(button);
  gtk_widget_grab_focus(button);
  gtk_widget_show(button);

  pd->is_cancel = FALSE;
  gtk_window_set_modal(GTK_WINDOW(pd->dialog), TRUE);

  return;
}

/*
 * @dialog_progressbar_dialog_update
 *
 *  Update the progress bar using the numerator and the denomirator.
 *  The argument numerator and denomirator will be treated as gfloat 
 *  type.
 *
 */
void dialog_progressbar_dialog_update  (ProgressbarDialog *pd,
                                        gint numerator       ,
                                        gint denomirator       ) {
  gchar buf[20];

  /* Update the progress bar. */
  snprintf(buf, 15, "%d / %d", numerator, denomirator);
  gtk_progress_set_format_string(GTK_PROGRESS(pd->bar), buf);
  gtk_progress_bar_update(GTK_PROGRESS_BAR(pd->bar),
                          (gfloat)numerator / (gfloat)denomirator);

  /* Redraw the toplevel window */
  while (gtk_events_pending()) { gtk_main_iteration(); }

  return;
}

/*
 * @dialog_progressbar_dialog_destroy
 *
 *  Destroy the progress bar dialog.
 *
 */
void dialog_progressbar_dialog_destroy(ProgressbarDialog *pd) {
  if (GTK_IS_WIDGET(pd->dialog)) { 
    gtk_window_set_modal(GTK_WINDOW(pd->dialog), FALSE);
    gtk_widget_destroy(pd->dialog); 
  }

  return;
}

/* Static function definitions. */
/*
 * @dialog_create_common
 *
 *
 *
 */
static void dialog_create_common (gchar                  *title        ,
                                  gchar                  *usr_msg      ,
                                  gchar                  **message     ,
                                  AppDialogType          type          ,
                                  AppDialogDefaultButton default_button,
                                  ManageDialog  *md                       ) {
  GtkWidget *vbox;
  GtkWidget *frame;
  GtkWidget *label;
  GtkWidget *button_box;
  GtkWidget *button;

  md->dialog = gtk_dialog_new();
  gtk_window_set_title (GTK_WINDOW(md->dialog), title);
  gtk_window_set_policy(GTK_WINDOW(md->dialog), FALSE, FALSE, FALSE);
  gtk_window_set_position(GTK_WINDOW(md->dialog), GTK_WIN_POS_CENTER);
  gtk_window_set_transient_for(GTK_WINDOW(md->dialog), 
                               GTK_WINDOW(md->window)); 
  gtk_signal_connect(GTK_OBJECT(md->dialog), "delete_event", 
                     GTK_SIGNAL_FUNC(dialog_cb_cancel_button),
                     md);
  gtk_widget_show(md->dialog);

  frame = gtk_frame_new(NULL);
  gtk_container_border_width(GTK_CONTAINER(frame), 8);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(md->dialog)->vbox), frame, 
                     TRUE, TRUE, 0);
  gtk_widget_show(frame);

  vbox = gtk_vbox_new(TRUE, 2);
  gtk_container_border_width(GTK_CONTAINER(vbox), 4);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_widget_show(vbox);

  label = gtk_label_new("    ");
  gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);
  if (usr_msg != NULL) {
    label = gtk_label_new(usr_msg);
    gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
    gtk_widget_show(label);
    label = gtk_label_new("    ");
    gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
    gtk_widget_show(label);
  }
  while(strcmp(*message, "") != 0) {
    label = gtk_label_new(_(*message));
    gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
    gtk_widget_show(label);
    message++;
  }
  label = gtk_label_new("    ");
  gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);

  button_box = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(button_box), GTK_BUTTONBOX_EDGE);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(md->dialog)->action_area),
                     button_box, TRUE, TRUE, 0);
  gtk_widget_show(button_box);

  switch (type) {
    case APP_OK_ONLY_DIALOG:
      button = gtk_button_new_with_label(_("  OK  "));
      gtk_box_pack_start(GTK_BOX(button_box), button, FALSE, FALSE, 0);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(dialog_cb_ok_button), md);

      /* Set the GtkButton GTK_CAN_DEFAULT, a boader can be around 
       * the button.
       */
      GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
      gtk_widget_grab_default(button);
      gtk_widget_grab_focus(button);
      gtk_widget_show(button);
      break;

    case APP_CANCEL_ONLY_DIALOG:
      button = gtk_button_new_with_label(_("  Cancel  "));
      gtk_box_pack_start(GTK_BOX(button_box), button, FALSE, FALSE, 0);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(dialog_cb_cancel_button), md);
      GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
      gtk_widget_grab_default(button);
      gtk_widget_grab_focus(button);
      gtk_widget_show(button);
      break;

    case APP_OK_CANCEL_DIALOG:
      button = gtk_button_new_with_label(_("  OK  "));
      gtk_box_pack_start(GTK_BOX(button_box), button, FALSE, FALSE, 0);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(dialog_cb_ok_button), md);
      GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
      if (default_button == APP_OK_BUTTON) {
        gtk_widget_grab_default(button);
        gtk_widget_grab_focus(button);
      }
      gtk_widget_show(button);

      button = gtk_button_new_with_label(_("  Cancel  "));
      gtk_box_pack_start(GTK_BOX(button_box), button, FALSE, FALSE, 0);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(dialog_cb_cancel_button), md);
      GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
      if (default_button != APP_OK_BUTTON) {
        gtk_widget_grab_default(button);
        gtk_widget_grab_focus(button);
      }
      gtk_widget_show(button);
      break;

    case APP_OK_CANCEL_ALL_DIALOG:
      button = gtk_button_new_with_label(_("  OK  "));
      gtk_box_pack_start(GTK_BOX(button_box), button, FALSE, FALSE, 0);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(dialog_cb_ok_button), md);
      GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
      if (default_button == APP_OK_BUTTON) {
        gtk_widget_grab_default(button);
        gtk_widget_grab_focus(button);
      }
      gtk_widget_show(button);

      button = gtk_button_new_with_label(_("  Cancel  "));
      gtk_box_pack_start(GTK_BOX(button_box), button, FALSE, FALSE, 0);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(dialog_cb_cancel_button), md);
      GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
      if (default_button != APP_OK_BUTTON) {
        gtk_widget_grab_default(button);
        gtk_widget_grab_focus(button);
      }
      gtk_widget_show(button);

      button = gtk_button_new_with_label(_("  Apply All  "));
      gtk_box_pack_start(GTK_BOX(button_box), button, FALSE, FALSE, 0);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(dialog_cb_ok_all_button), md);
      GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
      gtk_widget_show(button);
      break;

    case APP_OK_CANCEL_SKIP_ALL_DIALOG:
      button = gtk_button_new_with_label(_("  OK  "));
      gtk_box_pack_start(GTK_BOX(button_box), button, FALSE, FALSE, 0);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(dialog_cb_ok_button), md);
      GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
      if (default_button == APP_OK_BUTTON) {
        gtk_widget_grab_default(button);
        gtk_widget_grab_focus(button);
      }
      gtk_widget_show(button);

      button = gtk_button_new_with_label(_("  Cancel  "));
      gtk_box_pack_start(GTK_BOX(button_box), button, FALSE, FALSE, 0);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(dialog_cb_cancel_button), md);
      GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
      if ((default_button != APP_OK_BUTTON  ) &&
          (default_button != APP_SKIP_BUTTON) &&
          (default_button != APP_ALL_BUTTON )    ) {
        gtk_widget_grab_default(button);
        gtk_widget_grab_focus(button);
      }
      gtk_widget_show(button);

      button = gtk_button_new_with_label(_("  Skip  "));
      gtk_box_pack_start(GTK_BOX(button_box), button, FALSE, FALSE, 0);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(dialog_cb_skip_button), md);
      GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
      if (default_button == APP_SKIP_BUTTON) {
        gtk_widget_grab_default(button);
        gtk_widget_grab_focus(button);
      }
      gtk_widget_show(button);

      button = gtk_button_new_with_label(_("  Apply All  "));
      gtk_box_pack_start(GTK_BOX(button_box), button, FALSE, FALSE, 0);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(dialog_cb_ok_all_button), md);
      GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
      if (default_button == APP_ALL_BUTTON) {
        gtk_widget_grab_default(button);
        gtk_widget_grab_focus(button);
      }
      gtk_widget_show(button);
      break;
  }

  md->is_ok = APP_RET_CANCEL;
  gtk_window_set_modal(GTK_WINDOW(md->dialog), TRUE);

  gtk_main();

  gtk_widget_destroy(md->dialog);

  return;
}

/* Callback function definitions. */
/*
 * @dialog_cb_ok_button
 *
 *  Called when the OK button pressed.
 *
 */
static void dialog_cb_ok_button(GtkWidget *widget, 
                                gpointer  data     ) {
  ManageDialog *md;

  md = (ManageDialog*)data;

  md->is_ok = APP_RET_OK;

  gtk_main_quit();

  return;
}

/*
 * @dialog_cb_cancel_button
 *
 *  Called when the CANCEL button pressed.
 *
 */
static void dialog_cb_cancel_button(GtkWidget *widget,
                                    gpointer  data     ) {
  ManageDialog *md;

  md = (ManageDialog*)data;

  md->is_ok = APP_RET_CANCEL;

  gtk_main_quit();

  return;
}

/*
 * @dialog_cb_skip_button
 *
 *  Called when the Skip button pressed.
 *
 */
static void dialog_cb_skip_button(GtkWidget *widget, 
                                  gpointer  data     ) {
  ManageDialog *md;

  md = (ManageDialog*)data;

  md->is_ok = APP_RET_SKIP;

  gtk_main_quit();

  return;
}

/*
 * @dialog_cb_ok_all_button
 *
 *  Called when the Apply_All button pressed.
 *
 */
static void dialog_cb_ok_all_button(GtkWidget *widget, 
                                    gpointer  data     ) {
  ManageDialog *md;

  md = (ManageDialog*)data;

  md->is_ok = APP_RET_OK_ALL;

  gtk_main_quit();

  return;
}

/*
 * @dialog_cb_progressbar_dialog_cancel
 *
 *  Called when the cancel button of the progressbar dialog pushed or
 *  the window manager's 'close' button is pushed.
 *
 */
static void dialog_cb_progressbar_dialog_cancel (GtkWidget *widget,
                                                 gpointer  data     ) {
  ProgressbarDialog *pd;

  pd = (ProgressbarDialog*)data;

  pd->is_cancel = TRUE;

  return;
}


