/***************************************************************************
 *   Copyright (C) 2005 by Adam Treat                                      *
 *   treat@kde.org                                                         *
 *                                                                         *
 *   Copyright (C) 2004 by Scott Wheeler                                   *
 *   wheeler@kde.org                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef DATATABLEBOX_H
#define DATATABLEBOX_H

#include "project.h"
#include "datatable.h"

#include <kdebug.h>
#include <klistview.h>

#include <qptrdict.h>

class DataClip;
class DataTable;
class DataTableItem;
class ViewMode;
class DataTableSearch;
class SearchDataTable;

class KPopupMenu;
class KSelectAction;

/**
 * This is the data table selection box that is by default on the right side of
 * DataKiosk's main widget (DataTableSplitter).
 */

class DataTableBox : public KListView, public Project
{
    Q_OBJECT

public:
    class Item;
    typedef QValueList<Item *> ItemList;

    friend class Item;
    friend class TreeViewMode;

    DataTableBox( QWidget *parent, QWidgetStack *dataTableStack,
                  const char *name = 0 );

    virtual ~DataTableBox();

    virtual void initialize();
    //UGLY HACK
    virtual void uglyInitialize();

    virtual void newProject();
    virtual void closeProject();
    virtual void clipboard();
    virtual void raise( DataTable *dataTable );
    virtual void raise( DataReport *dataReport );
    virtual void removeTable();
    virtual void sortTables();
    virtual void removeReport();

    virtual void editSearch();
    virtual void removeSearch();
    virtual void invokeSearch( const QString &search );

    virtual void advancedSearchRequested();

    Item *dropItem() const
    {
        return m_dropItem;
    }

    bool hasSelection();

public slots:
    //UGLY HACK
    virtual void slotInitialize()
    {
        initialize();
    }
    virtual DataTable *currentDataTable() const;
    virtual DataReport *currentDataReport() const;
    virtual DataTable *rootOfCurrentDataTable();
    virtual DataTableList dataTablesInCurrentTree();
    virtual DataTableList dataTablesInDataTableTree( DataTable* dataTable );
    virtual DataTableList childrenOfDataTable( DataTable* dataTable );
    virtual void scrollTableUp();
    virtual void scrollTableDown();

protected:
    virtual void setSplashMessage( const QString &txt );
    virtual void setupDataTable( DataTable *dataTable, const QString &iconName );
    virtual void setupDataTable( DataTable *dataTable, const QString &iconName, Item *parentItem );
    virtual void setupDataTable( DataTable *dataTable, const QString &iconName, DataTable *parentTable );
    virtual void setIconName( DataTable* dataTable, const QString &iconName );
    virtual void setupDataReport( DataReport *dataReport, const QString &iconName, Item *parentItem );
    virtual void setupDataReport( DataReport *dataReport, const QString &iconName, DataTable *parentTable );
    virtual void setIconName( DataReport* dataReport, const QString &iconName );

signals:
    void searchEdited();
    void searchRemoved();
    void searchInvoked( const QString &search );
    void advancedSearchClicked();
    void splashMessage( const QString &txt, int alignment, const QColor &color );

private:
    void readConfig();
    void saveConfig();

    virtual void contentsMousePressEvent( QMouseEvent *e );
    virtual void contentsMouseReleaseEvent( QMouseEvent *e );
    virtual void keyPressEvent( QKeyEvent *e );
    virtual void keyReleaseEvent( QKeyEvent *e );

    QValueList<Item *> selectedItems();
    void setSingleItem( QListViewItem *item );

    void setupItem( Item *item );
    void performTreeViewSetup();
    int viewModeIndex() const
    {
        return m_viewModeIndex;
    }
    ViewMode *viewMode() const
    {
        return m_viewModes[ m_viewModeIndex ];
    }

private slots:
    /**
     * Catches QListBox::currentChanged(QListBoxItem *), does a cast and then re-emits
     * the signal as currentChanged(Item *).
     */
    void slotItemDestroyed( Item* item );
    void slotWidgetChanged();
    void slotShowContextMenu( QListViewItem *item, const QPoint &point, int );
    void slotSetViewMode( int index );
    void clipOnWindowShown();

private:
    KPopupMenu *m_listMenu;
    DataClip *m_clipboard;
    KPopupMenu *m_reportMenu;
    QPtrDict<Item> m_dataTableDict;
    int m_viewModeIndex;
    QValueList<ViewMode *> m_viewModes;
    KAction *m_k3bAction;
    bool m_hasSelection;
    bool m_doingMultiSelect;
    bool m_treeViewSetup;
    Item *m_dropItem;
    bool m_clipShown;
};



class DataTableBox::Item : public QObject, public KListViewItem
{
    friend class DataTableBox;
    friend class ViewMode;
    friend class CompactViewMode;
    friend class TreeViewMode;

    Q_OBJECT

    // moc won't let me create private QObject subclasses and Qt won't let me
    // make the destructor protected, so here's the closest hack that will
    // compile.

public:
    virtual ~Item();

protected:
    Item( DataTableBox *listBox, const QString &icon, const QString &text, QWidget *l = 0 );
    Item( Item *parent, const QString &icon, const QString &text, QWidget *l = 0 );

    QWidget *itemWidget() const
    {
        return m_widget;
    }
    DataTableBox *listView() const
    {
        return static_cast<DataTableBox *>( KListViewItem::listView() );
    }
    QString iconName() const
    {
        return m_iconName;
    }

    void setIconName( const QString &iconName )
    {
        m_iconName = iconName;
    }

    QString text() const
    {
        return m_text;
    }
    virtual int compare( QListViewItem *i, int col, bool ) const;
    virtual void paintCell( QPainter *p, const QColorGroup &colorGroup, int column, int width, int align );
    virtual void paintFocus( QPainter *, const QColorGroup &, const QRect & )
    {}
    virtual void setText( int column, const QString &text );

    virtual QString text( int column ) const
    {
        return KListViewItem::text( column );
    }

    virtual void setup();

public slots:
    void repaintRequested();
    void slotSetName( const QString &name );

signals:
    void itemDestroyed( Item *item );

private:
    // setup() was already taken.
    void init();

    QWidget *m_widget;
    QString m_text;
    QString m_iconName;
    static Item *m_collectionItem;
};

#endif
