/***************************************************************************
*   Copyright (C) 2005 by Adam Treat                                      *
*   treat@kde.org                                                         *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/

#include "dataclip.h"

#include "actioncollection.h"

#include <qgrid.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpainter.h>
#include <qgroupbox.h>
#include <qcheckbox.h>
#include <qclipboard.h>
#include <qvaluelist.h>

#include <kapplication.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kaction.h>
#include <kdebug.h>

#define widget (kapp->mainWidget())

using namespace ActionCollection;

PlainButton::PlainButton( const QPixmap &icon, int frameStyle, QWidget *parent )
        : QFrame( parent, "PlainButton" ),
        m_frameStyle( frameStyle )
{
    setFrameStyle( m_frameStyle );
    QVBoxLayout *topLayout = new QVBoxLayout( this, 1, 0 );
    m_label = new QLabel( this );
    m_label->setPixmap( icon );
    m_label->setAlignment( Qt::AlignCenter );
    m_background = paletteBackgroundColor();
    topLayout->add
    ( m_label );
}

PlainButton::~PlainButton()
{}

void PlainButton::mousePressEvent( QMouseEvent *ev )
{
    QFrame::mousePressEvent( ev );
    if ( ev->button() != Qt::LeftButton )
        return ;

    setFrameStyle( QFrame::Panel | QFrame::Sunken );
    setPaletteBackgroundColor( Qt::lightGray );
}

void PlainButton::mouseReleaseEvent( QMouseEvent *ev )
{
    QFrame::mouseReleaseEvent( ev );
    if ( ev->button() != Qt::LeftButton )
        return ;

    setFrameStyle( m_frameStyle );
    setPaletteBackgroundColor( m_background );
    emit clicked();
}

CopyItem::CopyItem( QWidget *parent, const QString &text )
        : PlainButton( DesktopIcon( "txt", 32 ), QFrame::Panel | QFrame::Raised, parent ),
        m_text( text )
{
    setFrameStyle( QFrame::Panel | QFrame::Sunken );
    setMinimumSize( 46, 46 );
    setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
    connect( this, SIGNAL( clicked() ),
             this, SLOT( paste() ) );
}

CopyItem::~CopyItem()
{}

QString CopyItem::text() const
{
    return m_text;
}

void CopyItem::setText( const QString &text )
{
    m_text = text;
}

void CopyItem::paste()
{
    QClipboard * clip = KApplication::clipboard();
    clip->blockSignals( true );
    const QString cache = clip->text( QClipboard::Clipboard );
    clip->setText( m_text, QClipboard::Clipboard );
    kapp->paste();
    clip->setText( cache, QClipboard::Clipboard );
    clip->blockSignals( false );
}

DataClip::DataClip( QWidget *parent )
        : QFrame( parent, "DataClip",
                  Qt::WStyle_Customize | Qt::WX11BypassWM )
{
    QVBoxLayout * topLayout = new QVBoxLayout( this, 1, 0 );

    QGroupBox *groupBox = new QGroupBox( this );
    QVBoxLayout *groupLayout = new QVBoxLayout( groupBox, 1, 0 );

    groupBox->setFrameShadow( QFrame::Raised );
    groupBox->setSizePolicy( QSizePolicy::Minimum, QSizePolicy::Minimum );
    groupBox->setMinimumSize( 200, 194 );

    TitleBar *titleBar = new TitleBar( groupBox, this );
    titleBar->setSizePolicy( QSizePolicy::Minimum, QSizePolicy::Minimum );
    titleBar->setMaximumHeight( 16 );

    QGroupBox *toolBar = new QGroupBox( groupBox );
    toolBar->setSizePolicy( QSizePolicy::Minimum, QSizePolicy::Minimum );
    toolBar->setMaximumHeight( 24 );
    toolBar->setFrameStyle( QFrame::ToolBarPanel | QFrame::Plain );
    QHBoxLayout *toolLayout = new QHBoxLayout( toolBar, 3, 0 );

    PlainButton *clear = new PlainButton( SmallIcon( "edittrash" ),
                                          QFrame::MenuBarPanel | QFrame::Plain, toolBar );
    clear->setMaximumSize( 18, 18 );
    clear->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
    connect( clear, SIGNAL( clicked() ),
             this, SLOT( clear() ) );

    QSpacerItem *left = new QSpacerItem( 2, 0, QSizePolicy::Fixed, QSizePolicy::Minimum );
    QSpacerItem *right = new QSpacerItem( 0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum );
    toolLayout->addItem( left );
    toolLayout->add
    ( clear );
    toolLayout->addItem( right );

    m_grid = new QGrid( 4, groupBox );
    m_grid->setSizePolicy( QSizePolicy::Minimum, QSizePolicy::Minimum );
    m_grid->setMargin( 5 );
    m_grid->setSpacing( 2 );

    connect( KApplication::clipboard(), SIGNAL( dataChanged() ),
             this, SLOT( appendCopy() ) );

    QSpacerItem *spacer = new QSpacerItem( 0, 0, QSizePolicy::Minimum, QSizePolicy::Expanding );

    groupLayout->add
    ( titleBar );
    groupLayout->add
    ( toolBar );
    groupLayout->add
    ( m_grid );
    groupLayout->addItem( spacer );
    topLayout->add
    ( groupBox );
}

DataClip::~DataClip()
{}

void DataClip::appendCopy()
{
    if ( !isVisible() || m_copies.count() == 12 )
        return ;

    QClipboard *clip = KApplication::clipboard();
    CopyItem *item = new CopyItem( m_grid, clip->text( QClipboard::Clipboard ) );
    item->setFrameStyle( QFrame::Panel | QFrame::Raised );
    item->show();
    m_copies.append( item );
}

void DataClip::clear()
{
    QValueList<CopyItem*>::Iterator it = m_copies.begin();
    for ( ; it != m_copies.end(); ++it )
    {
        delete ( *it );
        ( *it ) = 0L;
    }
    m_copies.clear();
}

TitleBar::TitleBar( QWidget *parent, DataClip *clip )
        : QFrame( parent, "TitleBar" ),
        m_clip( clip )
{
    //dataKiosk splash color
    //also part of the new kde color swatch :)
    setPaletteBackgroundColor( QColor( 35, 114, 253 ) );
    QHBoxLayout *topLayout = new QHBoxLayout( this, 2, 0 );

    QLabel *caption = new QLabel( i18n( "Clipboard" ), this );
    caption->setAlignment( Qt::AlignCenter );
    caption->setPaletteForegroundColor( Qt::white );
    QSpacerItem *spacer = new QSpacerItem( 0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum );

    m_close = new QCheckBox( this );
    m_close->setChecked( true );
    connect( m_close, SIGNAL( clicked() ),
             this, SLOT( close() ) );

    topLayout->add
    ( caption );
    topLayout->addItem( spacer );
    topLayout->add
    ( m_close );
}

TitleBar::~TitleBar()
{}

void TitleBar::close()
{
    m_clip->hide();
    m_close->setChecked( true );
    if ( KToggleAction *toggle = ::qt_cast<KToggleAction*>( action( "clipboard_manager" ) ) )
        toggle->setChecked( false );
}

void TitleBar::mousePressEvent( QMouseEvent * ev )
{
    m_x = ev->x();
    m_y = ev->y();
}

void TitleBar::mouseMoveEvent( QMouseEvent * ev )
{
    const int x = ev->globalX() - m_x;
    const int y = ev->globalY() - m_y;
    m_clip->move( x, y );
}

#include "dataclip.moc"
