/* Distributed Checksum Clearinghouse
 *
 * Copyright (c) 2005 by Rhyolite Software, LLC
 *
 * This agreement is not applicable to any entity which sells anti-spam
 * solutions to others or provides an anti-spam solution as part of a
 * security solution sold to other entities, or to a private network
 * which employs the DCC or uses data provided by operation of the DCC
 * but does not provide corresponding data to other users.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * Parties not eligible to receive a license under this agreement can
 * obtain a commercial license to use DCC and permission to use
 * U.S. Patent 6,330,590 by contacting Commtouch at http://www.commtouch.com/
 * or by email to nospam@commtouch.com.
 *
 * A commercial license would be for Distributed Checksum and Reputation
 * Clearinghouse software.  That software includes additional features.  This
 * free license for Distributed ChecksumClearinghouse Software does not in any
 * way grant permision to use Distributed Checksum and Reputation Clearinghouse
 * software
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND RHYOLITE SOFTWARE, LLC DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL RHYOLITE SOFTWARE, LLC
 * BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Rhyolite Software DCC 1.3.42-1.16 $Revision$
 */

#include "dcc_defs.h"


/* because tv.tv_sec is not a time_t on all systems
 * and to be thread safe on WIN32 */
const struct tm *
dcc_localtime(time_t secs, struct tm *result)
{
#ifdef HAVE_LOCALTIME_R
	localtime_r(&secs, result);
#else
	dcc_localtime_lock();
	result = localtime(&secs);
	dcc_localtime_unlock();
#endif
	return result;
}



int
dcc_get_secs(char *s, char **end, int min, int max, int def)
{
	static time_t epoch = 0;
	char *p;
	int secs;

	if (*s == '\0' || *s == ',') {
		secs = def;
		p = s;
	} else if (min > 0
		   && !CSTRCMP(s, "never")) {
		p = s+STRZ("never");
		if (!epoch)
			epoch = time(0);
		secs = epoch;
	} else if ((secs = strtoul(s, &p, 0)) >= DCC_MAX_SECS/60) {
		return -1;
	} else if (*p == '\0' || *p == ',') {
		;
	} else if (!CSTRCMP(p, "seconds")) {
		p += STRZ("seconds");
	} else if (!CSTRCMP(p, "s")) {
		p += STRZ("s");

	} else if (!CSTRCMP(p, "minutes")) {
		p += STRZ("minutes");
		secs *= 60;
	} else if (!CSTRCMP(p, "minute")) {
		p += STRZ("minute");
		secs *= 60;
	} else if (!CSTRCMP(p, "m")) {
		p += STRZ("m");
		secs *= 60;

	} else if (secs >= DCC_MAX_SECS/(60*60)) {
		return -1;
	} else if (!CSTRCMP(p, "hours")) {
		p += STRZ("hours");
		secs *= 60*60;
	} else if (!CSTRCMP(p, "hour")) {
		p += STRZ("hour");
		secs *= 60*60;
	} else if (!CSTRCMP(p, "h")) {
		p += STRZ("h");
		secs *= 60*60;

	} else if (secs >= DCC_MAX_SECS/(24*60*60)) {
		return -1;
	} else if (!CSTRCMP(p, "days")) {
		p += STRZ("days");
		secs *= 24*60*60;
	} else if (!CSTRCMP(p, "day")) {
		p += STRZ("day");
		secs *= 24*60*60;
	} else if (!CSTRCMP(p, "d")) {
		p += STRZ("d");
		secs *= 24*60*60;

	} else if (secs >= DCC_MAX_SECS/(7*24*60*60)) {
		return -1;
	} else if (!CSTRCMP(p, "weeks")) {
		p += STRZ("weeks");
		secs *= 7*24*60*60;
	} else if (!CSTRCMP(p, "week")) {
		p += STRZ("week");
		secs *= 7*24*60*60;
	} else if (!CSTRCMP(p, "w")) {
		p += STRZ("w");
		secs *= 7*24*60*60;

	} else {
		return -1;
	}

	if (secs > max)
		return -1;
	if (secs < min && secs != 0)
		return -1;

	if (*p != '\0') {
		if (*p != ',' || !end)
			return -1;
		++p;
	}
	if (end)
		*end = p;
	return secs;
}



time_t
tv_diff2us(const struct timeval *tv1, const struct timeval *tv2)
{
	time_t us;

	/* prevent overflow */
	us = tv1->tv_sec - tv2->tv_sec;
	if (us <= -FOREVER_SECS)
		return -FOREVER_USECS;
	if (us >= FOREVER_SECS)
		return FOREVER_USECS;
	us = us*DCC_USECS + (tv1->tv_usec - tv2->tv_usec);
	return us;
}



void
tvs_diff(struct timeval *tgt,
	 const struct timeval *a,
	 const struct timeval *b)
{
	tgt->tv_sec = a->tv_sec - b->tv_sec;
	tgt->tv_usec = a->tv_usec - b->tv_usec;
	if (tgt->tv_usec < 0) {
		--tgt->tv_sec;
		tgt->tv_usec += DCC_USECS;
	}
}



void
tvs_add(struct timeval *tgt,
	const struct timeval *a,
	const struct timeval *b)
{
	tgt->tv_sec = a->tv_sec + b->tv_sec;
	tgt->tv_usec = a->tv_usec + b->tv_usec;
	tgt->tv_sec += tgt->tv_usec / DCC_USECS;
	tgt->tv_usec %= DCC_USECS;
}



void
tvs_add_usec(struct timeval *tgt,
	     time_t usec)
{
	tgt->tv_usec += usec;
	tgt->tv_sec += tgt->tv_usec / DCC_USECS;
	tgt->tv_usec %= DCC_USECS;
}
