/* Distributed Checksum Clearinghouse
 *
 * threaded version of client locking
 *
 * Copyright (c) 2005 by Rhyolite Software, LLC
 *
 * This agreement is not applicable to any entity which sells anti-spam
 * solutions to others or provides an anti-spam solution as part of a
 * security solution sold to other entities, or to a private network
 * which employs the DCC or uses data provided by operation of the DCC
 * but does not provide corresponding data to other users.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * Parties not eligible to receive a license under this agreement can
 * obtain a commercial license to use DCC and permission to use
 * U.S. Patent 6,330,590 by contacting Commtouch at http://www.commtouch.com/
 * or by email to nospam@commtouch.com.
 *
 * A commercial license would be for Distributed Checksum and Reputation
 * Clearinghouse software.  That software includes additional features.  This
 * free license for Distributed ChecksumClearinghouse Software does not in any
 * way grant permision to use Distributed Checksum and Reputation Clearinghouse
 * software
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND RHYOLITE SOFTWARE, LLC DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL RHYOLITE SOFTWARE, LLC
 * BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Rhyolite Software DCC 1.3.42-1.59 $Revision$
 */

#include "dcc_ck.h"
#include "dcc_clnt.h"
#ifdef HAVE_PTHREAD_H
#include <pthread.h>
#else
#include <sys/pthread.h>
#endif
#include <signal.h>

u_char grey_on;
u_char grey_query_only;

DCC_WF cmn_wf, cmn_tmp_wf;


/* many POSIX thread implementations have unexpected side effects on
 * ordinary system calls, so don't use the threaded version unless
 * necessary */

/* protect the links among contexts and the miscellaneous global
 * variables in the DCC client library */
static pthread_mutex_t ctxts_mutex;
#ifdef DCC_DEBUG_CLNT_LOCK
static pthread_t ctxts_owner;
#endif

/* make syslog() thread-safe */
static pthread_mutex_t syslog_mutex;
static u_char syslog_threaded;

#ifdef DCC_DEBUG_HEAP
static pthread_mutex_t malloc_mutex;
static u_char malloc_threaded;
#endif

/* make gethostbyname() thread-safe */
static pthread_mutex_t host_mutex;

/* Only one DCC client process (not just one thread in every process) can
 * resolve server host names or send a burst of NOPs to measure RTTs.
 * These locks only keep more than one thread in a process from trying to
 * resolve server names to addresses.  A lock on the map file is used
 * to resolve the lock among processes. */
static pthread_mutex_t clnt_resolve_mutex;
#ifdef DCC_DEBUG_CLNT_LOCK
static pthread_t clnt_resolve_owner;
#endif
static pthread_t clnt_resolve_tid;
static pthread_cond_t clnt_resolve_cond;
static u_char clnt_resolve_stopping;

/* The threaded DNS blacklist and external filter support uses fork() to
 * create helper processes to wait for the typical single-threaded DNS
 * resolver library and who knows what sort of external filter. */
static pthread_mutex_t helper_cnt_mutex;
static pthread_mutex_t helper_recv_mutex;


void
dcc_ctxts_lock(void)
{
	int error = pthread_mutex_lock(&ctxts_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE, "pthread_mutex_lock(ctxts): %s",
			   ERROR_STR1(error));
#ifdef DCC_DEBUG_CLNT_LOCK
	ctxts_owner = pthread_self();
#endif
}



void
dcc_ctxts_unlock(void)
{
	int error;

#ifdef DCC_DEBUG_CLNT_LOCK
	ctxts_owner = 0;
#endif
	error = pthread_mutex_unlock(&ctxts_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE, "pthread_mutex_unlock(ctxts): %s",
			   ERROR_STR1(error));
}



#ifdef DCC_DEBUG_CLNT_LOCK
void
have_ctxts_lock(void)
{
	if (ctxts_owner != pthread_self())
		dcc_logbad(EX_SOFTWARE, "don't have ctxts lock");
}
#endif



void
dcc_syslog_lock(void)
{
	int error;

	if (!syslog_threaded)
		return;
	error = pthread_mutex_lock(&syslog_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE, "pthread_mutex_lock(syslog): %s",
			   ERROR_STR1(error));
}



void
dcc_syslog_unlock(void)
{
	int error;

	if (!syslog_threaded)
		return;
	error = pthread_mutex_unlock(&syslog_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE, "pthread_mutex_unlock(syslog): %s",
			   ERROR_STR1(error));
}



/* gethostbyname() etc. are usually not reentrant */
u_char dcc_host_locked = 1;

/* do not worry about locking gethostbyname() until the locks have
 * been initialized */
static u_char dcc_host_threaded = 0;

void
dcc_host_lock(void)
{
	int error;

	if (!dcc_host_threaded)
		return;

	error = pthread_mutex_lock(&host_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE, "pthread_mutex_lock(host): %s",
			   ERROR_STR1(error));
	dcc_host_locked = 1;
}



void
dcc_host_unlock(void)
{
	int error;

	if (!dcc_host_threaded)
		return;

	dcc_host_locked = 0;
	error = pthread_mutex_unlock(&host_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE, "pthread_mutex_unlock(host): %s",
			   ERROR_STR1(error));
}



#ifdef DCC_DEBUG_HEAP
void
dcc_malloc_lock(void)
{
	int error;

	if (!malloc_threaded)
		return;
	error = pthread_mutex_lock(&malloc_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE, "pthread_mutex_lock(malloc): %s",
			   ERROR_STR1(error));
}


void
dcc_malloc_unlock(void)
{
	int error;

	if (!malloc_threaded)
		return;
	error = pthread_mutex_unlock(&malloc_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE, "pthread_mutex_unlock(malloc): %s",
			   ERROR_STR1(error));
}
#endif /* DCC_DEBUG_HEAP */



#ifndef HAVE_LOCALTIME_R
/* make localtime() thread safe */
static pthread_mutex_t syslog_mutex;
static u_char localtime_threaded;

void
dcc_localtime_lock(void)
{
	int error;

	if (!localtime_threaded)
		return;
	error = pthread_mutex_lock(&localtime_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE, "pthread_mutex_lock(localtime): %s",
			   ERROR_STR1(error));
}



void
dcc_localtime_unlock(void)
{
	int error;

	if (!localtime_threaded)
		return;
	error = pthread_mutex_unlock(&localtime_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE, "pthread_mutex_unlock(localtime): %s",
			   ERROR_STR1(error));
}
#endif /* HAVE_LOCALTIME_R */



void
dcc_clnt_resolve_mutex_lock(void)
{
	int error;

	error = pthread_mutex_lock(&clnt_resolve_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE,
			   "pthread_mutex_lock(resolve): %s",
			   ERROR_STR1(error));
#ifdef DCC_DEBUG_CLNT_LOCK
	clnt_resolve_owner = pthread_self();
#endif
}



void
dcc_clnt_resolve_mutex_unlock(void)
{
	int error;

#ifdef DCC_DEBUG_CLNT_LOCK
	clnt_resolve_owner = 0;
#endif
	error = pthread_mutex_unlock(&clnt_resolve_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE,
			   "pthread_mutex_unlock(resolve): %s",
			   ERROR_STR1(error));
}



#ifdef DCC_DEBUG_CLNT_LOCK
void
have_clnt_resolve_mutex(void)
{
	if (clnt_resolve_owner != pthread_self())
		dcc_logbad(EX_SOFTWARE, "don't have resolve mutex");
}
#endif



const char *main_white_nm;

static void * NRATTRIB
clnt_resolve_thread(void *arg UATTRIB)
{
	DCC_WF wf, tmp_wf;
	DCC_EMSG emsg;
	DCC_CLNT_CTXT *ctxt;
	sigset_t sigsoff;
	int error;

	/* let the thread in charge of signals deal with them */
	sigemptyset(&sigsoff);
	sigaddset(&sigsoff, SIGHUP);
	sigaddset(&sigsoff, SIGINT);
	sigaddset(&sigsoff, SIGTERM);
	error = pthread_sigmask(SIG_BLOCK, &sigsoff, 0);
	if (error)
		dcc_logbad(EX_SOFTWARE, "pthread_sigmask(resolve): %s",
			   ERROR_STR1(error));

	dcc_wf_init(&wf, 0);
	if (main_white_nm)
		dcc_new_white_nm(emsg, &wf, main_white_nm);

	dcc_ctxts_lock();
	ctxt = dcc_alloc_ctxt();
	dcc_ctxts_unlock();
	for (;;) {
		dcc_ctxts_lock();
		error = pthread_cond_wait(&clnt_resolve_cond, &ctxts_mutex);
		if (error != 0)
			dcc_logbad(EX_SOFTWARE,
				   "pthread_cond_wait(resolve): %s",
				   ERROR_STR1(error));
#ifdef DCC_DEBUG_CLNT_LOCK
		ctxts_owner = pthread_self();
#endif

		if (clnt_resolve_stopping) {
			dcc_ctxts_unlock();
			pthread_exit(0);
		}

		if (!dcc_clnt_rdy(emsg, ctxt, DCC_CLNT_FG_NO_FAIL))
			dcc_error_msg("%s", emsg);
		else if (!dcc_info_unlock(emsg))
			dcc_logbad(dcc_ex_code, "%s", emsg);

		if (grey_on) {
			if (!dcc_clnt_rdy(emsg, ctxt, (DCC_CLNT_FG_GREY
						       | DCC_CLNT_FG_NO_FAIL)))
				dcc_error_msg("%s", emsg);
			else if (!dcc_info_unlock(emsg))
				dcc_logbad(dcc_ex_code, "%s", emsg);
		}

		dcc_ctxts_unlock();

		if (wf.ascii_nm[0] != '\0') {
			switch (dcc_rdy_white(emsg, &wf, &tmp_wf)) {
			case DCC_WHITE_OK:
			case DCC_WHITE_NOFILE:
			case DCC_WHITE_SILENT:
				break;
			case DCC_WHITE_CONTINUE:
			case DCC_WHITE_COMPLAIN:
				dcc_error_msg("%s", emsg);
				break;
			}
			cmn_wf.changed = 1;
		}
	}
}



u_char					/* 1=awoke the resolver thread */
dcc_clnt_wake_resolve(void)
{
	int error;

	/* we cannot awaken ourself */
	if (pthread_equal(pthread_self(), clnt_resolve_tid))
		return 0;

	error = pthread_cond_signal(&clnt_resolve_cond);
	if (error != 0)
		dcc_logbad(EX_SOFTWARE, "pthread_cond_signal(resolve): %s",
			   ERROR_STR1(error));
	return 1;
}



void
dcc_clnt_stop_resolve(void)
{
	if (clnt_resolve_stopping++)
		return;
	if (pthread_equal(pthread_self(), clnt_resolve_tid))
		return;
	pthread_cond_signal(&clnt_resolve_cond);
}



/* some pthreads implementations (e.g. AIX) don't like static
 * initializations */
static void
dcc_mutex_init(void *mutex, const char *nm)
{
	int error = pthread_mutex_init(mutex, 0);
	if (error)
		dcc_logbad(EX_SOFTWARE, "pthread_mutex_init(%s): %s",
			   nm, ERROR_STR1(error));
}



static pthread_mutex_t work_mutex;
#ifdef DCC_DEBUG_CLNT_LOCK
static pthread_t cwf_owner;
#endif

void
lock_work(void)
{
	int result = pthread_mutex_lock(&work_mutex);
	if (result)
		dcc_logbad(EX_SOFTWARE, "pthread_mutex_lock(work_free): %s",
			   ERROR_STR1(result));
}



void
unlock_work(void)
{
	int result = pthread_mutex_unlock(&work_mutex);
	if (result)
		dcc_logbad(EX_SOFTWARE, "pthread_mutex_unlock(work_free): %s",
			   ERROR_STR1(result));
}



/* lock all CWF structures as well as the cmn_wf structure */
static pthread_mutex_t cwf_mutex;

void
lock_wf(void)
{
	int error = pthread_mutex_lock(&cwf_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE, "pthread_mutex_lock(cwf): %s",
			   ERROR_STR1(error));
#ifdef DCC_DEBUG_CLNT_LOCK
	cwf_owner = pthread_self();
#endif
}



void
unlock_wf(void)
{
	int error;
#ifdef DCC_DEBUG_CLNT_LOCK
	cwf_owner = 0;
#endif
	error = pthread_mutex_unlock(&cwf_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE, "pthread_mutex_unlock(cwf): %s",
			   ERROR_STR1(error));
}



#ifdef DCC_DEBUG_CLNT_LOCK
void
have_cwf_lock(void)
{
	if (cwf_owner != pthread_self())
		dcc_logbad(EX_SOFTWARE, "don't have cwf lock");
}
#endif



void
dcc_clnt_thread_init(void)
{
	int error;

#ifdef DCC_WIN32
	win32_init();
#endif

	dcc_mutex_init(&ctxts_mutex, 0);
	dcc_mutex_init(&syslog_mutex, 0);
	syslog_threaded = 1;
#ifdef DCC_DEBUG_HEAP
	dcc_mutex_init(&malloc_mutex, 0);
	malloc_threaded = 1;
#endif
#ifndef HAVE_LOCALTIME_R
	dcc_mutex_init(&localtime_mutex, 0);
	localtime_threaded = 1;
#endif
	dcc_mutex_init(&host_mutex, 0);
	dcc_host_threaded = 1;
	dcc_host_locked = 0;

	/* Some pthreads implementations (e.g. AIX) don't like static
	 * POSIX thread initializations */

	dcc_mutex_init(&work_mutex, "wf_mutex");
	dcc_mutex_init(&cwf_mutex, "cwf_mutex");

	dcc_mutex_init(&clnt_resolve_mutex, 0);
	error = pthread_cond_init(&clnt_resolve_cond, 0);
	if (error)
		dcc_logbad(EX_SOFTWARE, "phtread_cond_init(resolve): %s",
			   ERROR_STR1(error));
	error = pthread_create(&clnt_resolve_tid, 0, clnt_resolve_thread, 0);
	if (error)
		dcc_logbad(EX_SOFTWARE, "pthread_create(resolve): %s",
			   ERROR_STR1(error));
	error = pthread_detach(clnt_resolve_tid);
	if (error)
		dcc_logbad(EX_SOFTWARE, "pthread_detach(resolve): %s",
			   ERROR_STR1(error));
}



/* protect DNS blacklist helper channels */
u_char
helper_cnt_lock_init(void)
{
	dcc_mutex_init(&helper_cnt_mutex, 0);
	return 1;
}



void
helper_cnt_lock(void)
{
	int error;

	error =  pthread_mutex_lock(&helper_cnt_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE,
			   "pthread_mutex_lock(helper counter): %s",
			   ERROR_STR1(error));
}



void
helper_cnt_unlock(void)
{
	int error;

	error = pthread_mutex_unlock(&helper_cnt_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE,
			   "pthread_mutex_unlock(helper counter): %s",
			   ERROR_STR1(error));
}



u_char
helper_recv_lock_init(void)
{
	dcc_mutex_init(&helper_recv_mutex, 0);
	return 1;
}



void
helper_recv_lock(void)
{
	int error;

	error =  pthread_mutex_lock(&helper_recv_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE,
			   "pthread_mutex_lock(helper recv): %s",
			   ERROR_STR1(error));
}



void
helper_recv_unlock(void)
{
	int error;

	error = pthread_mutex_unlock(&helper_recv_mutex);
	if (error)
		dcc_logbad(EX_SOFTWARE,
			   "pthread_mutex_unlock(helper recv): %s",
			   ERROR_STR1(error));
}
