# Debconf.pm - utility functions for DebAux scripts

# Copyright (C) 2002 Stefan Hornburg  (Racke) <racke@linuxia.de>

# This file is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any
# later version.

# This file is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied warranty
# of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this file; see the file COPYING.  If not, write to the Free
# Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

package DebAux::Debconf;
use strict;
use Debconf::Client::ConfModule;

=head1 NAME

  DebAux::Debconf - easy interface to Debconf::Client::ConfModule

=head1 SYNOPSIS

  use DebAux::Debconf;
  DebAux::Util::get('package/name');
  DebAux::Util::set('package/name','value');
  DebAux::Util::ask('medium','package/name');

=head1 DESCRIPTION

  This module provides an easy interface to Debconf::Client::Module
  which does the error handling for you.

=head1 FUNCTIONS

=over 4

=item get I<name>

  Get the value for I<name>. Dies on any error.

=back

=cut

sub get {
	my ($name) = @_;
	my ($status, $value) = Debconf::Client::ConfModule::get($name);

	if ($status) {
		die ("Couldn't get debconf value for $name (Status: $status)\n");
	}

	$value;
}

=over 4

=item set I<name> I<value>

  Set the value for I<name>. Dies on any error.

=back

=cut

sub set {
	my ($name, $value) = @_;
	my $status = Debconf::Client::ConfModule::set($name, $value);

	if ($status) {
		die ("Couldn't set debconf value $value for $name (Status: $status)\n");
	}

	$value;
}
	
=over 4

=item ask I<priority> I<name>

  Asks for the value for I<name> with the priority I<priority>.
  Combines the debconf commands INPUT and GO.
  Dies on any error.

=back

=cut

sub ask {
	my ($priority, $name) = @_;
	my ($status, $value);

	($status, $value) = Debconf::Client::ConfModule::input($priority, $name);
	if ($status && $status != 30) {
		die "Configuration of $name failed with status $status ($value)\n";
	}
	Debconf::Client::ConfModule::go();
}

1;
