# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Athropos@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import cgi, gobject, gtk.glade, modules, os.path

from gtk        import gdk
from gui        import infoMsgBox, errorMsgBox
from tools      import consts, prefs
from gettext    import gettext as _
from gui.window import Window


(
    ROW_ICON,       # An icon indicating whether the module is configurable
    ROW_TEXT,       # Name and description of the module
    ROW_NAME,       # Name of the module
    ROW_ENABLED,    # True if the module is currently enabled
    ROW_UNLOADABLE  # True if the module can be disabled
) = range(5)


class Preferences:
    """ Allow the user to load/unload/configure the modules """


    def __init__(self):
        """ Constructor """
        self.window = None


    def show(self):
        """ Show the dialog box """
        if self.window is None:
            self.store  = gtk.ListStore(gdk.Pixbuf, gobject.TYPE_STRING, gobject.TYPE_STRING, gobject.TYPE_BOOLEAN, gobject.TYPE_BOOLEAN)
            self.window = Window('Preferences.glade', 'vbox1', __name__, _('Preferences'), 390, 350)
            # Get widgets
            self.tree      = self.window.getWidget('tree-modules')
            self.btnPrefs  = self.window.getWidget('btn-prefs')
            self.btnClose  = self.window.getWidget('btn-close')
            # Configure the tree
            toggleRenderer = gtk.CellRendererToggle()
            self.tree.insert_column_with_attributes(-1, '', toggleRenderer,           active=ROW_ENABLED, activatable=ROW_UNLOADABLE)
            self.tree.insert_column_with_attributes(-1, '', gtk.CellRendererText(),   markup=ROW_TEXT)
            self.tree.insert_column_with_attributes(-1, '', gtk.CellRendererPixbuf(), pixbuf=ROW_ICON)
            self.tree.get_column(1).set_expand(True)
            toggleRenderer.connect('toggled', self.onModuleToggled)
            self.fillList()
            self.tree.set_model(self.store)
            # GTK handlers
            self.btnPrefs.connect('clicked', self.onPreferences)
            self.tree.connect('cursor-changed', self.onSelectionChanged)
            self.window.getWidget('btn-help').connect('clicked', self.onHelp)
            self.btnClose.connect('clicked', lambda btn: self.window.hide())

        if not self.window.isVisible():
            self.btnPrefs.set_sensitive(False)
            self.btnClose.grab_focus()
            self.tree.get_selection().unselect_all()

        self.window.show()


    def fillList(self):
        """ Fill the list of modules """
        self.tree.freeze_child_notify()
        self.store.clear()
        modList = [m for m in modules.getModules() if m[1][modules.MOD_CONFIGURABLE] or not m[1][modules.MOD_MANDATORY]]
        modList.sort()
        for (name, data) in modList:
            text = '<b>%s</b>\n<small>%s</small>' % (cgi.escape(name), cgi.escape(data[modules.MOD_DESC]))

            if data[modules.MOD_CONFIGURABLE] and data[modules.MOD_INSTANCE]: icon = consts.icoBtnPrefs
            else:                                                             icon = None

            self.store.append([icon, text, name, data[modules.MOD_INSTANCE] is not None, not data[modules.MOD_MANDATORY]])
        self.tree.thaw_child_notify()


    # --== GTK handlers ==--


    def onModuleToggled(self, renderer, path):
        """ A module has been enabled/disabled """
        iter = self.store.get_iter(path)
        name = self.store.get_value(iter, ROW_NAME)

        if self.store.get_value(iter, ROW_ENABLED):
            modules.unload(name)
        else:
            errMsg = modules.load(name)
            if errMsg is not None:
                errorMsgBox(self.window, _('Unable to load this module.'), errMsg)

        self.fillList()


    def onHelp(self, btn):
        """ Show a small help message box """
        helpMsg  = _('This is the list of available modules. You can enable/disable them by checking/unchecking the checkbox in front of them. Some modules cannot be disabled.')
        helpMsg += '\n\n'
        helpMsg += _('If an icon is displayed on the right of the module, it means that you can configure that module. Note that this is only possible if the module is enabled.')
        infoMsgBox(self.window, helpMsg)


    def onSelectionChanged(self, tree):
        """ Decide whether the new selection may be configured """
        selectedRow = tree.get_selection().get_selected()
        if selectedRow is not None:
            model, iter  = selectedRow
            configurable = model.get_value(iter, ROW_ICON) is not None
            enabled      = model.get_value(iter, ROW_ENABLED)
            self.btnPrefs.set_sensitive(enabled and configurable)


    def onPreferences(self, btn):
        """ Configure the selected module """
        selectedRow = self.tree.get_selection().get_selected()
        if selectedRow is None:
            self.btnPrefs.set_sensitive(False)
        else:
            model, iter = selectedRow
            modules.configure(model.get_value(iter, ROW_NAME), self.window)


# --== Global functions ==--


__instance  = None

def show():
    """ Show the preferences dialog box """
    global __instance

    if __instance is None:
        __instance = Preferences()
    __instance.show()
