# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Athropos@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import cgi, string

from tools   import consts, sec2str
from gettext import gettext as _


# Special fields that may be used to call format()
FIELDS = (
            ( 'track'       , _('Track number')                          ),
            ( 'title'       , _('Track title')                           ),
            ( 'artist'      , _('Track artist')                          ),
            ( 'album'       , _('Track album')                           ),
            ( 'duration_sec', _('Duration in seconds (e.g., 194)')       ),
            ( 'duration_str', _('Duration as a string (e.g., 3:14)')     ),
            ( 'playlist_pos', _('Position of the track in the playlist') ),
            ( 'playlist_len', _('Number of tracks in the playlist')      ),
            ( 'path'        , _('Full path to the file')                 )
         )


def getFormatSpecialFields(usePango=True):
    """
        Return a string in plain English (or whatever language being used) giving the special fields that may be used to call Track.format()
        If usePango is True, the returned string uses Pango formatting for better presentation
    """
    if usePango: return '\n'.join(['<tt><b>%s</b></tt> %s' % (string.ljust(field, 15), desc) for (field, desc) in FIELDS])
    else:        return '\n'.join(['%s %s' % (string.ljust(field, 15), desc) for (field, desc) in FIELDS])


class Track:
    """ Store information about a track """


    def __init__(self, (number, title, artist, album, length, path), plPos, plLen):
        """ Constructor """
        self.path   = path
        self.plPos  = plPos
        self.plLen  = plLen
        self.title  = title
        self.album  = album
        self.artist = artist
        self.length = length
        self.number = number


    def hasTitle(self):  return self.title  != consts.UNKNOWN_TITLE
    def hasAlbum(self):  return self.album  != consts.UNKNOWN_ALBUM
    def hasArtist(self): return self.album  != consts.UNKNOWN_ARTIST
    def hasLength(self): return self.album  != consts.UNKNOWN_LENGTH
    def hasNumber(self): return self.number != consts.UNKNOWN_TRACKNUMBER


    def getPath(self):   return self.path
    def getPlPos(self):  return self.plPos
    def getPlLen(self):  return self.plLen
    def getTitle(self):  return self.title
    def getAlbum(self):  return self.album
    def getArtist(self): return self.artist
    def getNumber(self): return self.number


    def getLengthSec(self):
        """ Return the length of the track in seconds (e.g., 194) """
        return self.length


    def getLengthStr(self):
        """ Return the length of the track as a formatted string (e.g., 3:14) """
        return sec2str(self.length)


    def getSafeTitle(self):
        """ Return the title, or an empty string if not known """
        if self.hasTitle(): return self.title
        else:               return ''


    def getSafeAlbum(self):
        """ Return the album, or an empty string if not known """
        if self.hasAlbum(): return self.album
        else:               return ''


    def getSafeArtist(self):
        """ Return the artist, or an empty string if not known """
        if self.hasArtist(): return self.artist
        else:                return ''


    def getSafeLength(self):
        """
            Return the length casted into a string (e.g., '194')
            Return an empty string if not known
        """
        if self.hasLength(): return str(self.length)
        else:                return ''


    def getSafeNumber(self):
        """
            Return the number casted into a string (e.g., '3')
            Return an empty string if not known
        """
        if self.hasNumber(): return str(self.number)
        else:                return ''




    def format(self, fmtString):
        """ Replace the special fields in the given string by their corresponding value """
        result = fmtString

        result = result.replace( '{path}',         self.getPath()           )
        result = result.replace( '{album}',        self.getAlbum()          )
        result = result.replace( '{track}',        str(self.getNumber())    )
        result = result.replace( '{title}',        self.getTitle()          )
        result = result.replace( '{artist}',       self.getArtist()         )
        result = result.replace( '{duration_sec}', str(self.getLengthSec()) )
        result = result.replace( '{duration_str}', self.getLengthStr()      )
        result = result.replace( '{playlist_pos}', str(self.getPlPos())     )
        result = result.replace( '{playlist_len}', str(self.getPlLen())     )

        return result


    def formatHTMLSafe(self, fmtString):
        """
            Replace the special fields in the given string by their corresponding value
            Also ensure that the fields don't contain HTML special characters (&, <, >)
        """
        result = fmtString

        result = result.replace( '{path}',         cgi.escape(self.getPath())   )
        result = result.replace( '{album}',        cgi.escape(self.getAlbum())  )
        result = result.replace( '{track}',        str(self.getNumber())        )
        result = result.replace( '{title}',        cgi.escape(self.getTitle())  )
        result = result.replace( '{artist}',       cgi.escape(self.getArtist()) )
        result = result.replace( '{duration_sec}', str(self.getLengthSec())     )
        result = result.replace( '{duration_str}', self.getLengthStr()          )
        result = result.replace( '{playlist_pos}', str(self.getPlPos())         )
        result = result.replace( '{playlist_len}', str(self.getPlLen())         )

        return result
