# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Athropos@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import gtk, gtk.glade, modules, os.path, tools

from tools   import consts
from modules import Module
from gettext import gettext as _


MOD_NAME            = _('Status Icon')
MOD_DESC            = _('Add an icon to the notification area')
MOD_IS_MANDATORY    = False
MOD_IS_CONFIGURABLE = False


class StatusIcon(Module):
    """
        Put an icon in the notification area and allow the user to directly interact with the program
    """


    def __init__(self, wTree):
        """ Constructor """
        Module.__init__(self)
        self.tooltip             = consts.appName
        self.isPlaying           = False
        self.isPaused            = False
        self.popupMenu           = None
        self.statusIcon          = gtk.StatusIcon()
        self.mainWindow          = wTree.get_widget('win-main')
        self.trackHasNext        = False
        self.emptyTracklist      = True
        self.trackHasPrevious    = False
        self.isMainWindowVisible = True
        modules.register(self, [modules.MSG_EVT_MOD_LOADED,    modules.MSG_EVT_MOD_UNLOADED, modules.MSG_EVT_APP_STARTED, modules.MSG_EVT_APP_QUIT,
                                modules.MSG_EVT_NEW_TRACK,     modules.MSG_EVT_PAUSED,       modules.MSG_EVT_UNPAUSED,    modules.MSG_EVT_STOPPED,
                                modules.MSG_EVT_NEW_TRACKLIST, modules.MSG_EVT_TRACK_MOVED])
        # GTK events
        self.statusIcon.connect('activate',     self.onActivate)
        self.statusIcon.connect('popup-menu',   self.onPopupMenu)
        self.statusIcon.connect('size-changed', lambda icon, size: self.setImage(size))


    def setImage(self, size):
        """ Change the image used by the status icon based on the available size """
        if   size >= 48: file = 'decibel-audio-player.png'
        elif size >= 32: file = 'decibel-audio-player-32.png'
        elif size >= 24: file = 'decibel-audio-player-24.png'
        else:            file = 'decibel-audio-player-16.png'

        self.statusIcon.set_from_file(os.path.join(consts.dirPix, file))


    def install(self):
        """ Install the Status icon """
        self.setImage(self.statusIcon.get_size())
        self.statusIcon.set_tooltip(consts.appName)
        self.onNewTrack(None)
        self.statusIcon.set_visible(True)


    def uninstall(self):
        """ Uninstall the Status icon """
        self.statusIcon.set_visible(False)


    def onNewTrack(self, track):
        """ A new track is being played, None if none """
        if track is None: self.tooltip = consts.appName
        else:             self.tooltip  = '%s - %s' % (track.getArtist(), track.getTitle())

        self.isPaused  = False
        self.isPlaying = track is not None
        self.statusIcon.set_tooltip(self.tooltip)


    def onPause(self):
        """ The current track has been paused """
        self.isPaused = True
        self.statusIcon.set_tooltip(self.tooltip + ' ' + _('[paused]'))


    def onUnpause(self):
        """ The current track has been unpaused """
        self.isPaused = False
        self.statusIcon.set_tooltip(self.tooltip)


    def __setVisible(self, widget, show):
        """ Show/hide a widget """
        if show: widget.show()
        else:    widget.hide()


    # --== Message handler ==--


    def handleMsg(self, msg, params):
        """ Handle messages sent to this module """
        if   msg == modules.MSG_EVT_APP_QUIT:      self.uninstall()
        elif msg == modules.MSG_EVT_MOD_LOADED:    self.install()
        elif msg == modules.MSG_EVT_APP_STARTED:   self.install()
        elif msg == modules.MSG_EVT_MOD_UNLOADED:  self.uninstall()
        elif msg == modules.MSG_EVT_NEW_TRACK:     self.onNewTrack(params['track'])
        elif msg == modules.MSG_EVT_STOPPED:       self.onNewTrack(None)
        elif msg == modules.MSG_EVT_PAUSED:        self.onPause()
        elif msg == modules.MSG_EVT_UNPAUSED:      self.onUnpause()
        elif msg == modules.MSG_EVT_TRACK_MOVED:   self.trackHasNext, self.trackHasPrevious = params['hasNext'], params['hasPrevious']
        elif msg == modules.MSG_EVT_NEW_TRACKLIST: self.emptyTracklist = (len(params['files']) == 0)


    # --== GTK handlers ==--


    def onActivate(self, statusIcon):
        """ The user has clicked on the status icon """
        if self.isMainWindowVisible: self.mainWindow.hide()
        else:                        self.mainWindow.show()

        self.isMainWindowVisible = not self.isMainWindowVisible


    def onPopupMenu(self, statusIcon, button, time):
        """ The user asks for the popup menu """
        if self.popupMenu is None:
            wTree              = tools.loadGladeFile('StatusIconMenu.glade')
            self.menuPlay      = wTree.get_widget('item-play')
            self.menuStop      = wTree.get_widget('item-stop')
            self.menuNext      = wTree.get_widget('item-next')
            self.popupMenu     = wTree.get_widget('menu-popup')
            self.menuPause     = wTree.get_widget('item-pause')
            self.menuUnpause   = wTree.get_widget('item-unpause')
            self.menuPrevious  = wTree.get_widget('item-previous')
            self.menuSeparator = wTree.get_widget('item-separator')
            # Connect handlers
            wTree.get_widget('item-quit').connect('activate', lambda btn: modules.postQuitMsg())
            self.menuPlay.connect('activate',     lambda btn: modules.postMsg(modules.MSG_CMD_TOGGLE_PAUSE))
            self.menuStop.connect('activate',     lambda btn: modules.postMsg(modules.MSG_CMD_STOP))
            self.menuNext.connect('activate',     lambda btn: modules.postMsg(modules.MSG_CMD_NEXT))
            self.menuPrevious.connect('activate', lambda btn: modules.postMsg(modules.MSG_CMD_PREVIOUS))
            self.menuPause.connect('activate',    lambda btn: modules.postMsg(modules.MSG_CMD_TOGGLE_PAUSE))
            self.menuUnpause.connect('activate',  lambda btn: modules.postMsg(modules.MSG_CMD_TOGGLE_PAUSE))

        # Show only relevant menu entries
        self.__setVisible(self.menuPlay,      not self.isPlaying and not self.emptyTracklist)
        self.__setVisible(self.menuStop,      self.isPlaying)
        self.__setVisible(self.menuNext,      self.trackHasNext)
        self.__setVisible(self.menuPause,     self.isPlaying and not self.isPaused)
        self.__setVisible(self.menuUnpause,   self.isPlaying and self.isPaused)
        self.__setVisible(self.menuPrevious,  self.trackHasPrevious)
        self.__setVisible(self.menuSeparator, self.isPlaying or not self.emptyTracklist)

        self.popupMenu.popup(None, None, None, button, time)
