/*
 * A Decibel demo
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <QtCore/QStringList>
#include <QtCore/QDebug>
#include <QtCore/QCoreApplication>
#include <QtCore/QTimer>
#include <QtDBus/QDBusConnection>

#include <QtTapioca/Connection>
#include <QtTapioca/Channel>
#include <QtTapioca/TextChannel>
#include <QtTapioca/Handle>

#include <Decibel/ContactManager>
#include <Decibel/DBusNames>

// Simplistic demo to "chat up" some contact. All it does is to
// send a ping message...
// Usage: $0 accounthandle contact_id

int main(int argc, char ** argv)
{
    QCoreApplication app(argc, argv);

    Q_ASSERT(argc == 3);

    int account_handle = QString(argv[1]).toInt();
    uint contact_id = QString(argv[2]).toInt();

    if (account_handle <= 0)
    {
        qDebug("Invalid account handle given (not a number?).");
        return 10;
    }
    if (contact_id == 0)
    {
        qDebug("Invalid contact ID given (not a number?).");
        return 11;
    }

    // Connect to session bus:
    if (!QDBusConnection::sessionBus().isConnected())
    {
        qDebug("Cannot connect to the D-BUS session bus.\n"
               "To start it, run:\n"
               "\teval `dbus-launch --auto-syntax`\n");
        return 1;
    }

    // Register custom types:
    Decibel::registerTypes();

    qDebug() << "D-Bus set up.";

    // connect to the Decibel daemon:
    qDebug() << "contacting:" << Decibel::daemon_service
             << Decibel::daemon_contactmanager_path;
    de::basyskom::Decibel::ContactManager
    contact_mgr(Decibel::daemon_service, Decibel::daemon_contactmanager_path,
                QDBusConnection::sessionBus(), &app);

    qDebug() << "ContactManager acquired.";

    QDBusReply<Decibel::ChannelInfo>
    reply = contact_mgr.contactContactUsingAccount(contact_id, account_handle,
                                                   int(QtTapioca::Channel::Text),
                                                   true);

    qDebug() << "Got my reply.";

    if(!reply.isValid())
    {
        QDBusError error = reply.error();
        qDebug() << "DBus Error while setting presence:"
                 << error.type() << ":"
                 << error.message();
        return 3;
    }
    if(reply.value().connection_service.isEmpty())
    {
        qDebug() << "Some error occurred when creating connection.";
        return 4;
    }

    qDebug() << "Reply is valid.";

    // Create QtTapioca Connection and Channel:
    QtTapioca::Connection * connection(new QtTapioca::Connection(reply.value().connection_service,
                                                                 reply.value().connection_path.path(),
                                                                 &app));

    QList<QtTapioca::Channel *> channels = connection->openChannels();
    QtTapioca::Channel * channel;
    foreach(channel, channels)
    {
        if (channel->serviceName() == reply.value().channel_service &&
            channel->objectPath() == reply.value().channel_path.path() &&
            channel->target()->handle()->type() == reply.value().target_handle_type &&
            channel->target()->handle()->id() == reply.value().target_handle)
        {
            break;
        }
        channel = 0;
    }
    QtTapioca::TextChannel * text_channel = dynamic_cast<QtTapioca::TextChannel *>(channel);
    if (0 == text_channel) { return 1; }

    qDebug() << "Got my textchannel.";

    // Send a ping!
    text_channel->sendMessage("ping from Decibel!");

    qDebug() << "Ping send!";

    return app.exec();
}
