/* dia-handle_.c
 * Copyright (C) 2001  Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "unit-test.h"
#include "dia-handle.h"

DiaCanvas	*canvas = NULL;
DiaCanvasItem	*item = NULL;
DiaHandle	*handle = NULL;

void
test_dia_handle_setup (void)
{
	canvas = dia_canvas_new (NULL);
	TEST (canvas != NULL);
	TEST (DIA_IS_CANVAS (canvas));
	TEST (DIA_IS_CANVAS_GROUP (canvas->root));

	item = dia_canvas_item_new (DIA_CANVAS_GROUP (canvas->root),
				    dia_canvas_item_get_type (), NULL);
	TEST (item != NULL);
	TEST (DIA_IS_CANVAS_ITEM (item));
	TEST (item->parent == (DiaCanvasGroup*) canvas->root);
	TEST (g_list_length (item->handles) == 0);

	handle = dia_handle_new (item);
	TEST (handle != NULL);
	TEST (DIA_IS_HANDLE (handle));
	TEST (handle->owner == item);
	TEST (g_list_length (item->handles) == 1);
	TEST (g_list_nth_data (item->handles, 0) == handle);
}

void
test_dia_handle_teardown (void)
{
	gtk_object_unref (GTK_OBJECT (canvas));
	canvas = NULL;
	item = NULL;
	handle = NULL;
}

TEST_BEGIN ("DiaHandle", test_dia_handle_setup, test_dia_handle_teardown)

NEW_TEST (dia_handle_new_with_pos)
{
	DiaHandle *h;

	h = dia_handle_new_with_pos (item, 4.0, 4.0);
	dia_handle_update_now (h);
	TEST (h != NULL);
	TEST (DIA_IS_HANDLE (h));
	TESTFL (h->pos_i.x, 4.0);
	TESTFL (h->pos_i.y, 4.0);
	TESTFL (gcl_variable_get_value (h->pos_w.x), 4.0);
	TESTFL (gcl_variable_get_value (h->pos_w.y), 4.0);
	TEST (g_list_last (item->handles)->data == h);

	h = dia_handle_new_with_pos (item, 5.5, 5.5);
	dia_handle_update_now (h);
	TEST (h != NULL);
	TEST (DIA_IS_HANDLE (h));
	TESTFL (h->pos_i.x, 5.5);
	TESTFL (h->pos_i.y, 5.5);
	TESTFL (gcl_variable_get_value (h->pos_w.x), 5.5);
	TESTFL (gcl_variable_get_value (h->pos_w.y), 5.5);
	TEST (g_list_last (item->handles)->data == h);
}

NEW_TEST (dia_handle_get/set_pos_i)
{
	gdouble x, y;

	dia_handle_get_pos_i (handle, &x, &y);
	TESTFL (x, 0.0);
	TESTFL (y, 0.0);

	dia_handle_set_pos_i (handle, 1.0, 2.0);
	TEST (GTK_OBJECT_FLAGS (handle) & DIA_HANDLE_NEED_UPDATE_I2W);
	dia_handle_get_pos_i (handle, &x, &y);
	TESTFL (x, 1.0);
	TESTFL (y, 2.0);

	dia_handle_get_pos_w (handle, &x, &y);
	TESTFL (x, 1.0);
	TESTFL (y, 2.0);
}

NEW_TEST (dia_handle_get/set_pos_w)
{
	gdouble x, y;

	dia_handle_get_pos_w (handle, &x, &y);
	TESTFL (x, 0.0);
	TESTFL (y, 0.0);

	dia_handle_set_pos_w (handle, 1.0, 2.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.x), 1.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.y), 2.0);
	TEST (GTK_OBJECT_FLAGS (handle) & DIA_HANDLE_NEED_UPDATE_W2I);
	dia_handle_get_pos_w (handle, &x, &y);
	TESTFL (x, 1.0);
	TESTFL (y, 2.0);

	dia_handle_get_pos_i (handle, &x, &y);
	TESTFL (x, 1.0);
	TESTFL (y, 2.0);
}

NEW_TEST (dia_handle_distance_i)
{
	gdouble dist = dia_handle_distance_i (handle, 5.0, 5.0);
	TESTFL (dist, 10.0);
}

NEW_TEST (dia_handle_distance_w)
{
	gdouble dist = dia_handle_distance_w (handle, 5.0, 5.0);
	TESTFL (dist, 10.0);
}

NEW_TEST (dia_handle_request_update_i2w)
{
	dia_handle_update_now (handle);
	TEST ((GTK_OBJECT_FLAGS (handle) & (DIA_HANDLE_NEED_UPDATE_W2I | DIA_HANDLE_NEED_UPDATE_W2I)) == 0);
	
	dia_handle_request_update_i2w (handle);
	TEST ((GTK_OBJECT_FLAGS (handle) & DIA_HANDLE_NEED_UPDATE_I2W) != 0);
	TEST ((GTK_OBJECT_FLAGS (handle) & DIA_HANDLE_NEED_UPDATE_W2I) == 0);
}

NEW_TEST (dia_handle_request_update_w2i)
{
	dia_handle_update_now (handle);
	TEST ((GTK_OBJECT_FLAGS (handle) & (DIA_HANDLE_NEED_UPDATE_W2I | DIA_HANDLE_NEED_UPDATE_W2I)) == 0);
	
	dia_handle_request_update_w2i (handle);
	TEST ((GTK_OBJECT_FLAGS (handle) & DIA_HANDLE_NEED_UPDATE_I2W) == 0);
	TEST ((GTK_OBJECT_FLAGS (handle) & DIA_HANDLE_NEED_UPDATE_W2I) != 0);
}


NEW_TEST (dia_handle_update_now)
{
	gdouble x, y;

	dia_handle_set_pos_i (handle, 1.0, 2.0);

	dia_handle_request_update_i2w (handle);
	TEST ((GTK_OBJECT_FLAGS (handle) & DIA_HANDLE_NEED_UPDATE_I2W) != 0);
	dia_handle_update_now (handle);
	
	dia_handle_get_pos_w (handle, &x, &y);
	TESTFL (x, 1.0);
	TESTFL (y, 2.0);
}

#if 0
NEW_TEST (dia_handle_move_1)
{
	/* Offset of the DiaCanvasItem is 0.0! */
	TESTFL (handle->pos_i.x, 0.0);
	TESTFL (handle->pos_i.y, 0.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.x), 0.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.y), 0.0);

	dia_handle_move_i (handle, 10.0, 10.0);
	dia_handle_update_now (handle);
	TESTFL (handle->pos_i.x, 10.0);
	TESTFL (handle->pos_i.y, 10.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.x), 10.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.y), 10.0);

	dia_handle_move_i (handle, 10.0, 10.0);
	dia_handle_update_now (handle);
	TESTFL (handle->pos_i.x, 20.0);
	TESTFL (handle->pos_i.y, 20.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.x), 20.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.y), 20.0);
}

NEW_TEST (dia_handle_move_2)
{
	item->affine[0] = 2.0;
	item->affine[1] = 2.0;
	item->affine[2] = 2.0;
	item->affine[3] = 2.0;
	item->affine[4] = 2.0;
	item->affine[5] = 2.0;

	TESTFL (handle->pos_i.x, 0.0);
	TESTFL (handle->pos_i.y, 0.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.x), 0.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.y), 0.0);

	dia_handle_move_i (handle, 10.0, 10.0);
	TESTFL (handle->pos_i.x, 10.0);
	TESTFL (handle->pos_i.y, 10.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.x), 0.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.y), 0.0);

	dia_handle_move_i (handle, 10.0, 10.0);
	dia_handle_update_now (handle);
	TESTFL (handle->pos_i.x, 20.0);
	TESTFL (handle->pos_i.y, 20.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.x), 0.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.y), 0.0);
}

NEW_TEST (dia_handle_move_affine)
{
	TESTFL (handle->pos_i.x, 0.0);
	TESTFL (handle->pos_i.y, 0.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.x), 0.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.y), 0.0);

	dia_handle_move_i (handle, 10.0, 10.0);
	dia_handle_update_i2w (handle);
	TESTFL (handle->pos_i.x, 10.0);
	TESTFL (handle->pos_i.y, 10.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.x), 10.0 * 2.0 + 10.0 * 2.0 + 2.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.y), 10.0 * 2.0 + 10.0 * 2.0 + 2.0);
}

NEW_TEST (dia_handle_move_pos_w.1)
{
	/* Offset of the DiaCanvasItem is 0.0! */
	TESTFL (handle->pos_i.x, 0.0);
	TESTFL (handle->pos_i.y, 0.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.x), 0.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.y), 0.0);

	dia_handle_move_w (handle, 10.0, 10.0);
	dia_handle_update_now (handle);
	TESTFL (handle->pos_i.x, 10.0);
	TESTFL (handle->pos_i.y, 10.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.x), 10.0);
	TESTFL (gcl_variable_get_value (handle->pos_w.y), 10.0);
}
#endif

TEST_END ()
