#!/usr/bin/env python
# -*- coding: UTF-8 -*-
###########################################################################
# displayconfig.py - description                                          #
# ------------------------------                                          #
# begin     : Fri Mar 02 2007                                             #
# copyright : (C) 2007 Canonical                                          #
# email     : Michael Vogt <michael.vogt@ubuntu.com>                      #
#             Sebastian Heinlein <sebi@glatzor.de>                        #
#                                                                         #
###########################################################################
#                                                                         #
#   This program is free software; you can redistribute it and/or modify  #
#   it under the terms of the GNU General Public License as published by  #
#   the Free Software Foundation; either version 2 of the License, or     #
#   (at your option) any later version.                                   #
#                                                                         #
###########################################################################

import string
import os
import select
import sys
import csv
import time
import signal
import shutil

import pygtk
pygtk.require("2.0")
import gtk
import gtk.glade
import gobject
from SimpleGladeApp import SimpleGladeApp
from widgets import *

from displayconfig.displayconfigcommon import _findXorgConfig, testX
from displayconfig.displayconfigabstraction import *
from displayconfig.displayconfigabstraction import Screen, GfxCard, XSetup

from gettext import gettext as _

# Map secondary screen orientation to the index of available options
# in the combobox
ORIENTATION_INDEX = {XSetup.RIGHTOF : 0,
                     XSetup.LEFTOF :1,
                     XSetup.ABOVE : 2,
                     XSetup.UNDER :3}

class DisplayConfig(SimpleGladeApp):

    def __init__(self, gladefile):
        """Provide a dialog that allows to configure your screens and 
           graphics cards including resolution, refresh rate, orientation,
           drivers and finally dual screen mode"""
        SimpleGladeApp.__init__(self, gladefile)
       
        self.store_screens = gtk.ListStore(gobject.TYPE_STRING,
                                           gtk.gdk.Pixbuf,
                                           gobject.TYPE_PYOBJECT)
        self.treeview_screens.set_model(self.store_screens)
        column_screens = gtk.TreeViewColumn(_("Screens"))
        renderer_icon = gtk.CellRendererPixbuf()
        renderer_screen = gtk.CellRendererText()
        column_screens.pack_start(renderer_icon, True)
        column_screens.pack_start(renderer_screen, True)
        column_screens.add_attribute(renderer_screen, "markup", 0)
        column_screens.add_attribute(renderer_icon, "pixbuf", 1)
        self.treeview_screens.append_column(column_screens)

        self.icon_theme = gtk.icon_theme_get_default()

        self.store_screen_resolutions = gtk.ListStore(gobject.TYPE_STRING)
        renderer = gtk.CellRendererText()
        self.combobox_resolution.set_model(self.store_screen_resolutions)
        self.combobox_resolution.pack_start(renderer)
        self.combobox_resolution.add_attribute(renderer, "markup", 0)

        self.store_screen_rates = gtk.ListStore(gobject.TYPE_STRING)
        renderer = gtk.CellRendererText()
        self.combobox_rate.set_model(self.store_screen_rates)
        self.combobox_rate.pack_start(renderer)
        self.combobox_rate.add_attribute(renderer, "markup", 0)

        self.store_screen_rotations = gtk.ListStore(gobject.TYPE_STRING,
                                                    gobject.TYPE_PYOBJECT)
        renderer = gtk.CellRendererText()
        self.combobox_rotation.set_model(self.store_screen_rotations)
        self.combobox_rotation.pack_start(renderer)
        self.combobox_rotation.add_attribute(renderer, "markup", 0)

        #FIXME: Add support to configure colour depth and gama correction
        #self.store_screen_colours = gtk.ListStore(gobject.TYPE_STRING)
        #renderer = gtk.CellRendererText()
        #self.combobox_colours.set_model(self.store_screen_colours)
        #self.combobox_colours.pack_start(renderer)
        #self.combobox_colours.add_attribute(renderer, "markup", 0)

        self.store_screens_primary = gtk.ListStore(gobject.TYPE_STRING,
                                                   gobject.TYPE_PYOBJECT)
        renderer = gtk.CellRendererText()
        self.combobox_primary.set_model(self.store_screens_primary)
        self.combobox_primary.pack_start(renderer)
        self.combobox_primary.add_attribute(renderer, "markup", 0)

        self.store_screens_secondary = gtk.ListStore(gobject.TYPE_STRING,
                                                     gobject.TYPE_PYOBJECT)
        renderer = gtk.CellRendererText()
        self.combobox_secondary.set_model(self.store_screens_secondary)
        self.combobox_secondary.pack_start(renderer)
        self.combobox_secondary.add_attribute(renderer, "markup", 0)

        self.xconfigpath = _findXorgConfig()
        SetDataFileDir("/usr/share/displayconfig-gtk/")

        self.xsetup = XSetup(self.xconfigpath)
        self._populateMonitorDialog()
        self._populateGfxcardDialog()

        # setup all handlers manually, since we need to inhibt them
        # during sync

        self.handlers = []
        for (widget, signal, handler) in \
            [(self.combobox_resolution, "changed", 
              self.on_combobox_resolution_changed),
             (self.combobox_rate, "changed", 
              self.on_combobox_rate_changed),
             (self.combobox_rotation, "changed",
              self.on_combobox_rotation_changed),
             (self.combobox_primary, "changed",
              self.on_combobox_primary_changed),
             (self.combobox_secondary, "changed",
              self.on_combobox_secondary_changed),
             (self.combobox_orientation, "changed", 
              self.on_combobox_orientation_changed),
             (self.radiobutton_extend, "toggled", 
              self.on_radiobutton_extend_toggled),
             (self.radiobutton_clone, "toggled", 
              self.on_radiobutton_clone_toggled)]:
            self.handlers.append((widget, widget.connect(signal, handler)))

        # we can only write if we run as root
        if os.getuid()!=0:
            self.button_apply.set_sensitive(False)

        self._syncGUI()
        
    def _populateMonitorDialog(self):
        """Initialize the dialog that allows to configure screens"""
        self.monitordb = monitordb = GetMonitorModelDB()

        # Store the vendor names and a ListStore with the corresponding
        # monitor models
        store_screen_vendors = gtk.ListStore(gobject.TYPE_STRING, 
                                             gobject.TYPE_PYOBJECT)

        # Setup the user interface
        renderer = gtk.CellRendererText()
        #FIXME: At&T is a problem using markup
        column = gtk.TreeViewColumn(_("Manufacturer"), renderer, markup=0)
        self.treeview_screen_vendors.append_column(column)
        self.treeview_screen_vendors.set_model(store_screen_vendors)

        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn(_("Model"), renderer, text=0)
        self.treeview_screen_models.append_column(column)

        # Load predefined monitors with model information
        vendors = monitordb.vendordb.keys()
        vendors.sort()
        for vendor in vendors:
            monitorkeys = self.monitordb.vendordb[vendor].keys()
            store_monitors = gtk.ListStore(gobject.TYPE_STRING, 
                                           gobject.TYPE_PYOBJECT)
            store_screen_vendors.append([gobject.markup_escape_text(vendor),
                                         store_monitors])
            monitorkeys.sort()
            for monitorkey in monitorkeys:
                store_monitors.append([monitorkey,
                                       self.monitordb.vendordb[vendor][monitorkey]])
        # Load generic monitors
        store_generic_monitors = gtk.ListStore(gobject.TYPE_STRING, 
                                               gobject.TYPE_PYOBJECT)
        generics = monitordb.genericdb.keys()
        generics.sort()
        store_screen_vendors.insert(0, ["<b>%s</b>" % _("Generic"),
                                        store_generic_monitors])
        for generic in generics:
            store_generic_monitors.append([generic,
                                           monitordb.genericdb[generic]])
        # Load custom monitors if available
        customs = monitordb.getCustomMonitors().keys()
        if len(customs) > 0:
            store_custom_monitors = gtk.ListStore(gobject.TYPE_STRING, 
                                                  gobject.TYPE_PYOBJECT)
            store_screen_vendors.insert(1, ["<b>%s</b>" % _("Custom"),
                                            store_custom_monitors])
            customs.sort()
            for custom in customs:
                store_custom_monitors.append([custom,
                                              monitordb.getCustomMonitors()[custom]])

    def on_treeview_screen_vendors_cursor_changed(self, treeview):
        """Attach the ListStore that contains the models of the selected
           vendor to the model treeview."""
        model = treeview.get_model()
        cursor_path = treeview.get_cursor()[0]
        store_monitors = model[cursor_path][1]
        self.treeview_screen_models.set_model(store_monitors)
        self.treeview_screen_models.set_cursor(0)

    def on_treeview_monitors_cursor_changed(self, treeview):
        """Update the frequence range information of the selected monitor."""
        model = treeview.get_model()
        cursor_path = treeview.get_cursor()[0]
        mon = model[cursor_path][1]
        if mon is not None:
            self.label_hrange.set_text(mon.getVerticalSync())
            self.label_vrange.set_text(mon.getHorizontalSync())

    def _populateGfxcardDialog(self):
        """Setup the dialog that allows to select the driver for a gfx card."""
        self.gfxcarddb = gfxcarddb = GfxCardModelDB()
        #Store the vendor name and a ListStore with corresponding models
        self.store_gfxcard_vendors = gtk.ListStore(gobject.TYPE_STRING,
                                                   gobject.TYPE_PYOBJECT)
        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn(_("Model"), renderer, text=0)
        self.treeview_gfxcards.append_column(column)

        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn(_("Manufacturer"), renderer, text=0)
        self.treeview_gfxcard_vendors.append_column(column)

        renderer = gtk.CellRendererText()
        self.combobox_gfxcard_drivers.pack_start(renderer)
        self.combobox_gfxcard_drivers.add_attribute(renderer, "markup", 0)

        vendors = gfxcarddb.vendordb.keys()
        vendors.sort()
        for vendor in vendors:
            store_vendor_cards = gtk.ListStore(gobject.TYPE_STRING,
                                               gobject.TYPE_PYOBJECT)
            self.store_gfxcard_vendors.append([vendor, store_vendor_cards])
            cardkeys = self.gfxcarddb.vendordb[vendor].keys()
            cardkeys.sort()
            for cardkey in cardkeys:
                store_vendor_cards.append([cardkey,
                                           self.gfxcarddb.vendordb[vendor][cardkey]])

        # Store all available drivers
        self.store_gfxcard_drivers = gtk.ListStore(gobject.TYPE_STRING,
                                                   gobject.TYPE_PYOBJECT)
        drivers = gfxcarddb.driverdb.keys()
        drivers.sort()
        for driver in drivers:
            self.store_gfxcard_drivers.append([driver,
                                               gfxcarddb.driverdb[driver]])

        self.treeview_gfxcard_vendors.set_model(self.store_gfxcard_vendors)
        self.combobox_gfxcard_drivers.set_model(self.store_gfxcard_drivers)

    def on_treeview_gfxcards_cursor_changed(self, treeview):
        """Check if a proprietary driver is available for the selected
           model and if so allow to select it. Furthermore update the 
           choose-driver-by-name combobox."""
        model = treeview.get_model()
        cursor_path = treeview.get_cursor()[0]
        card = model[cursor_path][1]
        # Default to the open source driver
        self.combobox_gfxcard_freedom.set_active(0)
        self.combobox_gfxcard_freedom.set_sensitive(False)
        # Allow to choose a proprietary driver if available
        if card is not None:
            s = card.getProprietaryDriver()
            self.combobox_gfxcard_freedom.set_sensitive(s is not None)
        # select the driver of the current model in the driver chooser
        self._select_in_combobox(self.combobox_gfxcard_drivers,
                                 card.getDriver(),
                                 0)

    def on_treeview_gfxcard_vendors_cursor_changed(self, treeview):
        """Attach the ListStore that contains the models of the selected
           vendor to the model treeview."""
        path = treeview.get_cursor()[0]
        if path is None:
            return
        model =  treeview.get_model()
        iter = model.get_iter(path)
        (name, store_cards) = model[iter]
        self.treeview_gfxcards.set_model(store_cards)
        self.treeview_gfxcards.set_cursor(0)

    def on_radiobutton_gfxcard_model_toggled(self, button):
        """Only enable widgets that are used by the selected way of 
           choosing a gfx card driver"""
        self.hbox_gfxcard_model.set_sensitive(button.get_active())
        self.combobox_gfxcard_drivers.set_sensitive(not button.get_active())
        # update the information
        self.on_treeview_gfxcards_cursor_changed(self.treeview_gfxcards)

    def block_handlers(self):
        """Block signals of all configuration widgets"""
        for (widget, handler) in self.handlers:
            widget.handler_block(handler)
    def unblock_handlers(self):
        """Unblock signals of all configuration widgets"""
        for (widget, handler) in self.handlers:
            widget.handler_unblock(handler)

    def _syncGUI(self):
        """Update the user interface to the latest configuration"""
        # Avoid tiggering of configuration changes
        self.block_handlers()
        # Show a frame for each gfx card on the hardware tab
        self.store_screens.clear()
        self.vbox_cards.foreach(lambda w: self.vbox_cards.remove(w))
        for gfxcard in self.xsetup.getGfxCards():
            self.vbox_cards.add(GfxCardFrame(self, gfxcard, self.on_configure_device_clicked))
            for screen in gfxcard.getScreens():
                screen_name = get_screen_name(screen)
                role = self.xsetup.getScreenRole(screen)
                if role == XSetup.ROLE_PRIMARY:
                    screen_name = "<b>%s</b>" % screen_name
                if not screen.isLive():
                    screen_name = "<i>%s</i>" % screen_name
                screen_icon = self.icon_theme.load_icon("video-display", 
                                                        32, 
                                                        gtk.ICON_LOOKUP_NO_SVG)
                #FIXME: Usind different icons here would be nice
                #if device.getMonitorAspect()==ModeLine.ASPECT_16_9:
                #screen_icon = self.icon_theme.load_icon("video-display", 
                #                                        32, 
                #                                        gtk.ICON_LOOKUP_NO_SVG)
                self.store_screens.append([screen_name,
                                           screen_icon, 
                                           screen])
        self._setup_dual_screen()
        self.treeview_screens.set_cursor(0)
        self.unblock_handlers()

    def _setup_dual_screen(self):
        """Update the widgets of the dual screen configuration tab to the
           latest configuration."""
        available_layouts = self.xsetup.getAvailableLayouts()
        self.frame_secondary.set_sensitive(available_layouts != \
                                           XSetup.LAYOUT_SINGLE)
        self.combobox_secondary.set_sensitive(available_layouts != \
                                              XSetup.LAYOUT_SINGLE)
        self.radiobutton_extend.set_inconsistent(available_layouts == \
                                                 XSetup.LAYOUT_SINGLE)
        self.radiobutton_clone.set_inconsistent(available_layouts == \
                                                XSetup.LAYOUT_SINGLE)
        self.radiobutton_extend.set_sensitive(available_layouts & \
                                              XSetup.LAYOUT_DUAL)
        self.radiobutton_clone.set_sensitive(available_layouts & \
                                             XSetup.LAYOUT_CLONE)

        # Select the current layout
        current_layout = self.xsetup.getLayout()
        self.radiobutton_extend.set_active(current_layout == XSetup.LAYOUT_DUAL)
        self.radiobutton_clone.set_active(current_layout == XSetup.LAYOUT_CLONE)

        # Select the current orientation
        orient = self.xsetup.getDualheadOrientation()
        self.combobox_orientation.set_active(ORIENTATION_INDEX[orient])

        self.store_screens_primary.clear()
        self.store_screens_secondary.clear()
        self.store_screens_secondary.append([_("None"), None])
        for screen in self.xsetup.getAllScreens():
            self.store_screens_primary.append([get_screen_name(screen),
                                               screen])
            if screen == self.xsetup.getPrimaryScreen():
                self.combobox_primary.set_active(len(self.store_screens_primary) - 1)
            elif screen == self.xsetup.getSecondaryScreen():
                self.combobox_secondary.set_active(len(self.store_screens_secondary) - 1)
                self.store_screens_secondary.append([get_screen_name(screen),
                                                    screen])
            else:
                self.store_screens_secondary.append([get_screen_name(screen),
                                                    screen])
        if available_layouts != XSetup.LAYOUT_SINGLE:
            self.combobox_secondary.set_active(0)

    def on_combobox_primary_changed(self, box):
        """Set the primary display to the selected one."""
        model = box.get_model()
        screen = model[box.get_active()][1]
        self.xsetup.setScreenRole(screen, XSetup.ROLE_PRIMARY)

    def on_combobox_secondary_changed(self, box):
        """Set the secondary screen to the selected one and disable the 
           dual head support and related widgets if no 
           secondary screen is selected."""
        active = box.get_active()
        self.frame_secondary.set_sensitive(active != 0)
        if active == 0:
            self.xsetup.setLayout(XSetup.LAYOUT_SINGLE)
            second = self.xsetup.getSecondaryScreen()
            self.xsetup.setScreenRole(second, XSetup.ROLE_UNUSED)
        else:
            model = box.get_model()
            screen = model[active][1]
            self.xsetup.setScreenRole(screen, XSetup.ROLE_SECONDARY)

    def on_combobox_orientation_changed(self, box):
        """Change the orientation of the secondary screen."""
        active = box.get_active()
        orient = [XSetup.RIGHTOF, XSetup.LEFTOF, XSetup.ABOVE, 
                  XSetup.UNDER][active]
        self.xsetup.setDualheadOrientation(XSetup.RIGHTOF)

    def on_radiobutton_extend_toggled(self, radio):
        """Toggle the dual mode of the secondary screen."""
        state = radio.get_active()
        self.combobox_orientation.set_sensitive(state)
        if radio.get_active() == True:
            self.xsetup.setLayout(XSetup.LAYOUT_DUAL)

    def on_radiobutton_clone_toggled(self, radio):
        """Toggle the clone mode of the secondary screen."""
        if radio.get_active() == True:
            self.xsetup.setLayout(XSetup.LAYOUT_CLONE)

    def _select_in_combobox(self, combobox, key, col):
        """Select the row with value key in the column col in a given
           combobox"""
        def find(model, path, iter):
            if model[path][col] == key:
                combobox.set_active_iter(iter)
                return True
            return False
        model = combobox.get_model()
        if model:
            model.foreach(find)

    def _select_in_treeview(self, treeview, key, col):
        """Select the row with value key in the column col in a given
           treeview"""
        def find(model, path, iter):
            if model[path][col] == key:
                treeview.expand_to_path(path)
                treeview.scroll_to_cell(path, use_align=True, row_align=0.5)
                treeview.set_cursor(path)
                return True
            return False
        model = treeview.get_model()
        if model:
            model.foreach(find)

    def on_configure_device_clicked(self, button, device):
        """Show a dialog that allows to configure the given device. Device
           can be a Screen or a GfxCard."""
        # Configure a screen device
        if isinstance(device, Screen):
            # Try to detect the current model. otherwise use the model
            # of the currently selected monitor
            detected_model = self.monitordb.detect()
            if detected_model != None:
                model = detected_model
            else:
                model = device.getMonitorModel()
            # Select the manfuctorer of the monitor. handle custom and
            # generic devices separately
            model_type = model.getType()
            if model_type == MonitorModel.TYPE_NORMAL:
                self._select_in_treeview(self.treeview_screen_vendors,
                                         device.getMonitorModel().getManufacturer(),
                                         0)
            elif  model_type == MonitorModel.TYPE_PLUGNPLAY:
                self._select_in_treeview(self.treeview_screen_vendors,
                                         "<b>%s</b>" % _("Custom"),
                                         0)
            else: 
                self._select_in_treeview(self.treeview_screen_vendors,
                                         "<b>%s</b>" % _("Generic"),
                                         0)
            # Select the model of the monitor
            self._select_in_treeview(self.treeview_screen_models,
                                     model.name, 0)
            # Run the dialog
            res = self.dialog_monitor.run()
            self.dialog_monitor.hide()
            if res == gtk.RESPONSE_OK:
                model = self.treeview_screen_models.get_model()
                cursor_path = self.treeview_screen_models.get_cursor()[0]
                mon = model[cursor_path][1]
                if mon is not None:
                    device.setMonitorModel(mon)
                    self._syncGUI()
        # Configure a gfx card device
        elif isinstance(device, GfxCard):
            # Automatically select the detected model
            self._select_in_treeview(self.treeview_gfxcard_vendors,
                                     device.getDetectedGfxCardModel().getVendor(), 0)
            self._select_in_treeview(self.treeview_gfxcards,
                                     device.getDetectedGfxCardModel().name, 0)
            self._select_in_combobox(self.combobox_gfxcard_drivers,
                                     device.getDetectedGfxCardModel().getDriver(),
                                     0)
            # by default select the model browser
            self.radiobutton_gfxcard_model.set_active(True)
            # Run the dialog
            res = self.dialog_gfxcard.run()
            self.dialog_gfxcard.hide()
            if res == gtk.RESPONSE_OK:
                if self.radiobutton_gfxcard_driver.get_active() == True:
                    # The user specified a driver
                    driver_index = self.combobox_gfxcard_drivers.get_active()
                    driver = self.store_gfxcard_drivers[driver_index][1]
                    device.getGfxCardModel().setDriver(driver)
                else:
                    # The user has chosen a model from the list
                    model = self.treeview_gfxcards.get_model()
                    cursor_path = self.treeview_gfxcards.get_cursor()[0]
                    card = model[cursor_path][1]
                    if card is not None:
                        device.setGfxCardModel(card)
                        # Check if the proprietary driver was chosen
                        isprop = (self.combobox_gfxcard_freedom.get_active()==1)
                        device.setProprietaryDriver(isprop)
                self._syncGUI()

    def on_treeview_screens_cursor_changed(self, treeview):
        """Update the configuration widgets of the selected screen"""
        # Get the currently selected screen device
        path = treeview.get_cursor()[0]
        if path is None:
            return
        model =  treeview.get_model()
        iter = model.get_iter(path)
        (name, pixbuf, device) = model[iter]

        # Show the resolution configuration of the selected screen
        self.store_screen_resolutions.clear()
        for res in device.getAvailableResolutions():
            res_string = "%s x %s" % (res[0], res[1])
            self.store_screen_resolutions.append([res_string])
        self.combobox_resolution.set_active(device.getResolutionIndex())

        # Show the refresh rate configuration of the selected screen
        #FIXME: Should depend on the currently selected resolution
        for rate in device.getAvailableRefreshRates():
            self.store_screen_rates.append(["%s Hz" % rate])
        self.combobox_rate.set_active(device.getRefreshRateIndex())

        # Show the rotation configuration of the selected screen
        self.store_screen_rotations.clear()
        rotations = device.getAvailableRotations()
        rotation_index = {}
        if (rotations & Screen.RR_Rotate_0):
            self.store_screen_rotations.append([_("Normal"),
                                               Screen.RR_Rotate_0])
            rotation_index[Screen.RR_Rotate_0] = len(self.store_screen_rotations)-1
        if (rotations & Screen.RR_Rotate_90):
            self.store_screen_rotations.append([_("Right"),
                                               Screen.RR_Rotate_90])
            rotation_index[Screen.RR_Rotate_90] = len(self.store_screen_rotations)-1
        if (rotations & Screen.RR_Rotate_180):
            self.store_screen_rotations.append([_("Flipped"),
                                               Screen.RR_Rotate_180])
            rotation_index[Screen.RR_Rotate_180] = len(self.store_screen_rotations)-1
        if (rotations & Screen.RR_Rotate_270):
            self.store_screen_rotations.append([_("Left"),
                                               Screen.RR_Rotate_270])
            rotation_index[Screen.RR_Rotate_270] = len(self.store_screen_rotations)-1
        self.combobox_rotation.set_active(rotation_index[device.getRotation()])


    def on_combobox_rotation_changed(self, combobox):
        """Set the screen rotation to the selected value. Changes will be
           applied later."""
        # Get current screen
        path = self.treeview_screens.get_cursor()[0]
        screen = self.store_screens[path][2]
        # Get index of active rotation
        index = combobox.get_active()
        # Set rotation
        screen.setRotation([Screen.RR_Rotate_0, Screen.RR_Rotate_90,
                            Screen.RR_Rotate_180, Screen.RR_Rotate_270][index])

    def on_combobox_rate_changed(self, combobox):
        pass

    def on_combobox_resolution_changed(self, combobox):
        """Update the available refresh rates"""
        # get current screen
        path = self.treeview_screens.get_cursor()[0]
        screen = self.store_screens[path][2]

        # Set the resolution
        index_res = self.combobox_resolution.get_active()
        screen.setResolutionIndex(index_res)

    def on_button_cancel_clicked(self, widget):
        gtk.main_quit()

    def on_button_apply_clicked(self, widget):
        # show a confirmation dialog
        if self.xsetup.isLiveResolutionConfigChanged():
            if self.xsetup.applyLiveResolutionChanges():
                # show a dialog to revert the changes
                timer = gobject.timeout_add(15000, 
                                            self.dialog_confirmation.response,
                                            gtk.RESPONSE_CANCEL)
                res = self.dialog_confirmation.run()
                self.dialog_confirmation.hide()
                gobject.source_remove(timer)
                if res == gtk.RESPONSE_OK:
                    self.xsetup.acceptLiveResolutionChanges()
                else:
                    self.xsetup.rejectLiveResolutionChanges()
                    return False
            else:
                # Nothing changed
                self.xsetup.applyLiveResolutionChanges()
        self.xsetup.acceptLiveResolutionChanges()
        self.apply_changes()
        gtk.main_quit()

    def on_button_test_config_clicked(self, widget):
        print "on_button_test_config_clicked()"
        (res, msg) = testX(self.xsetup,"/usr/share/displayconfig-gtk/servertestdialog-gtk")
        print res
        print msg
        if (res == False):
            dia = gtk.MessageDialog(parent=self.window_main,
                                    type=gtk.MESSAGE_INFO,
                                    buttons=gtk.BUTTONS_CLOSE,
                                    message_format=_("X server failed"))
            dia.format_secondary_text(_("The configuration was not successful"))
            dia.run()
            dia.destroy()
            
    def apply_changes(self):
        print "applying changes"
        # Backup up the current config file.
        i = 1
        while os.path.exists("%s.%i" % (self.xconfigpath,i)):
            i += 1
        shutil.copyfile(self.xconfigpath,"%s.%i" % (self.xconfigpath,i))
        # Write out the new config
        tmpfilename = self.xconfigpath + ".tmp"
        self.xsetup.writeXorgConfig(tmpfilename)
        os.rename(tmpfilename, self.xconfigpath)

# vim:ts=4:sw=4:et
