/******************************************************************************
 *
 * Beats-per-minute (BPM) detection routine
 *
 * Author        : Copyright (c) Olli Parviainen
 * Author e-mail : oparviai @ iki.fi
 * File created  : 11-Jan-2003
 *
 * Last changed  : $Date: 2003/10/24 07:08:39 $
 * File revision : $Revision: 1.7 $
 *
 * $Id: BPMDetect.cpp,v 1.7 2003/10/24 07:08:39 melanie_t Exp $
 *
 * License :
 *
 *  SoundTouch sound processing library
 *  Copyright (c) Olli Parviainen
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *****************************************************************************/

#include <math.h>
#include <assert.h>
#include <string.h>
#include "FIFOSampleBuffer.h"
#include "PeakFinder.h"
#include "BPMDetect.h"
#include "jack.h"

#define INPUT_BLOCK_SAMPLES       4096
#define DECIMATED_BLOCK_SAMPLES   2048

typedef unsigned short ushort;
typedef unsigned long ulong;

/// decay constant for calculating RMS volume sliding average approximation 
/// (time constant is about 10 sec)
const float avgdecay = 0.99986f;

/// Normalization coefficient for calculating RMS sliding average approximation.
const float avgnorm = (1 - avgdecay);

QBPMEvent::QBPMEvent(QBPMEvent::Type t) : QEvent((QEvent::Type)t)
{
	s_bpm=0.0f;
}

QBPMEvent::~QBPMEvent()
{
}

void QBPMEvent::setBpm(float i_bpm)
{
	s_bpm=i_bpm;
}

float QBPMEvent::bpm()
{
	return s_bpm;
}

BPMDetect::BPMDetect(QWidget *parent, StreamSource *source)
{
	s_parent=parent;
	s_source=source;
	s_abort=false;
    xcorr = NULL;

    buffer = new FIFOSampleBuffer();

    decimateSum = 0;
    decimateCount = 0;
    decimateBy = 0;

    envelopeAccu = 0;

    // Initialize RMS volume accumulator to RMS level of 3000, which is a pretty typical RMS 
    // signal level value for songdata. This value is then adapted to the actual level during
    // processing.
    RMSVolumeAccu = (3000 * 3000) / avgnorm;
}



BPMDetect::~BPMDetect()
{
    delete[] xcorr;
    delete buffer;
	delete s_source;
}


/// low-pass filter & decimate to about 500 Hz. return number of outputted samples.
///
/// Decimation is used to remove the unnecessary frequencies and thus to reduce 
/// the amount of data needed to be processed as calculating autocorrelation 
/// function is a very-very heavy operation.
///
/// Anti-alias filtering is done simply by averaging the samples. This is really a 
/// poor-man's anti-alias filtering, but it's not so critical in this kind of application
/// (it'd also be difficult to design a high-quality filter with steep cut-off at very 
/// narrow band)
int BPMDetect::decimate(short *dest, const short *src, int numsamples)
{
    int count, outcount, iout;

    assert(decimateBy != 0);
    outcount = 0;
    for (count = 0; count < numsamples; count ++) 
    {
        decimateSum += src[count];

        decimateCount ++;
        if (decimateCount >= decimateBy) 
        {
            // Store every Nth sample only
            iout = decimateSum / decimateBy;
            decimateSum = 0;
            decimateCount = 0;
            // check ranges for sure (shouldn't actually be necessary)
            if (iout > 32767) 
            {
                iout = 32767;
            } 
            else if (iout < -32768) 
            {
                iout = -32768;
            }
            dest[outcount] = iout;
            outcount ++;
        }
    }
    return outcount;
}



// Calculates autocorrelation function of the sample history buffer
void BPMDetect::updateXCorr(const int process_samples)
{
    int offs;
    short *pBuffer;
    
    assert(buffer->numSamples() >= (uint)(process_samples + windowLen));

    pBuffer = buffer->ptrBegin();
    for (offs = windowStart; offs < windowLen; offs ++) 
    {
        ulong sum;
        int i;

        sum = 0;
        for (i = 0; i < process_samples; i ++) 
        {
            sum += pBuffer[i] * pBuffer[i + offs];    // scaling the sub-result shouldn't be necessary
        }
//        xcorr[offs] *= xcorr_decay;   // decay 'xcorr' here with suitable coefficients 
                                        // if it's desired that the system adapts automatically to
                                        // various bpms, e.g. in processing continouos music stream.
                                        // The 'xcorr_decay' should be a value that's smaller than but 
                                        // close to one, and should also depend on 'process_samples' value.

        xcorr[offs] += sum;
    }
}



// Calculates envelope of the sample data
void BPMDetect::calcEnvelope(short *samples, int numsamples) 
{
    const float decay = 0.7f;               // decay constant for smoothing the envelope
    const float norm = (1 - decay);

    int i;
    int out;
    float val;

    for (i = 0; i < numsamples; i ++) {

        // calc average RMS volume
        RMSVolumeAccu *= avgdecay;
        val = (float)(abs(samples[i]));
        RMSVolumeAccu += val * val;

        // cut amplitudes that are below 2 times average RMS volume
        // (we're interested in peak values, not the silent moments)
        val -= 2 * (float)sqrt(RMSVolumeAccu * avgnorm);
        val = (val > 0) ? val : 0;

        // smooth amplitude envelope
        envelopeAccu *= decay;
        envelopeAccu += val;
        out = (int)(envelopeAccu * norm);

        // cut peaks (shouldn't be necessary though)
        if (out > 32767) out = 32767;
        samples[i] = out;
    }
}



// reads a block of samples from the file, anti-alias filter and decimate.
// returns number of samples outputted.
int BPMDetect::inputSamples(StreamSource *source, short *outBuffer)
{
    int numChannels;
    short samples[INPUT_BLOCK_SAMPLES];
    int numSamples;

    // read a block of samples
    numSamples = source->get_buffer((char *)samples, INPUT_BLOCK_SAMPLES/2);
    numChannels = 2;

    // convert from stereo to mono if necessary
    if (numChannels == 2)
    {
        int i;

        for (i = 0; i < numSamples; i ++)
        {
            samples[i] = (samples[i * 2] + samples[i * 2 + 1]) / 2;
        }
    }
    
    // decimate
    return decimate(outBuffer, samples, numSamples);
}


void BPMDetect::processBlock(short *samples, int numSamples)
{
    // envelope new samples and add them to buffer
    calcEnvelope(samples, numSamples);
    buffer->putSamples(samples, numSamples);

    // when the buffer has enought samples for processing...
    if ((int)buffer->numSamples() > windowLen) 
    {
        int processLength;

        // how many samples are processed
        processLength = buffer->numSamples() - windowLen;

        // ... calculate autocorrelations for oldest samples...
        updateXCorr(processLength);
        // ... and remove them from the buffer
        buffer->receiveSamples(processLength);
    }
}



void BPMDetect::init(StreamSource *source)
{
    int sampleRate;

    sampleRate = source->sample_rate;

//printf("Source sample rate %d\n", sampleRate);
    // choose decimation factor so that result is approx. 500 Hz
    decimateBy = sampleRate / 500;
    assert(decimateBy > 0);

    // Calculate window length & starting item according to desired min & max bpms
    windowLen = (60 * sampleRate) / (decimateBy * MIN_BPM);
    windowStart = (60 * sampleRate) / (decimateBy * MAX_BPM);

    assert(windowLen > windowStart);

    // allocate new working objects
    xcorr = new float[windowLen];
    memset(xcorr, 0, windowLen * sizeof(float));

    // we do processing in mono mode
    buffer->setChannels(1);
    buffer->clear();
}


float BPMDetect::detectBpm(StreamSource *source)
{
    short decimated[DECIMATED_BLOCK_SAMPLES];
    int dsamples;
    int sampleRate;
	int yield_counter=0;
    float peakPos;
    PeakFinder peakFinder;

    // Initialize processing constants etc.
    init(source);
	source->play(0, 0);

    // process the file
    while (source->is_playing())
    {
        // input a block of samples
        dsamples = inputSamples(source, decimated);
        assert(dsamples <= DECIMATED_BLOCK_SAMPLES);
        // process block
        processBlock(decimated, dsamples);
		if((++yield_counter) == 20)
		{
			usleep(10000);
			yield_counter=0;
		}
		if(s_abort)
			return 0.0f;
    }

    // find peak position
    peakPos = peakFinder.detectPeak(xcorr, windowStart, windowLen);
	int newWin=(int)(peakPos*0.75f);
    float peakPos2 = peakFinder.detectPeak(xcorr, windowStart, newWin);
	int newStart=(int)(peakPos+(peakPos-(float)windowStart)*1.5f);
    float peakPos3;
	if(newStart < windowLen)
		peakPos3 = peakFinder.detectPeak(xcorr, newStart, windowLen);
	else
		peakPos3=0.0;

//	printf("Peak %f %f %f\n", peakPos, peakPos2, peakPos3);
	float pv1, pv2, pv3;
	pv1=xcorr[(int)peakPos];
	pv2=xcorr[(int)peakPos2];
	pv3=xcorr[(int)peakPos3];

	printf("%s\n", (const char *)s_source->filename);
	printf("%f %f %f %f %f\n", pv1, pv2, pv3, pv2/pv1, pv3/pv1);

    // delete xcorr array
    delete[] xcorr;
    xcorr = NULL;

    assert(decimateBy != 0);
    if (peakPos < 1e-6) return 0.0; // detection failed.

    // calculate BPM
    sampleRate = source->sample_rate;
    float b1=60.0f * (((float)sampleRate / (float)decimateBy) / peakPos);
    float b2=60.0f * (((float)sampleRate / (float)decimateBy) / peakPos2);
    float b3=60.0f * (((float)sampleRate / (float)decimateBy) / peakPos3);

	printf("%f %f %f\n", b1, b2, b3);

	float b=b1;
	if(pv2/pv1 > 0.4f)
		b=b2;
	if(b < 60.0)
		b*=2;
	if(b > 200)
		b/=2;
    return b;
}

void BPMDetect::run(void)
{
	sleep(2); // Be nice, the initial read is going on in another thread at
			  // this time
	s_res=detectBpm(s_source);
	QBPMEvent *e=new QBPMEvent;
	e->setBpm(s_res);
	QThread::postEvent(s_parent, e);
}

float BPMDetect::result()
{
	return s_res;
}

void BPMDetect::abort()
{
	s_abort=true;
	while(!finished())
		usleep(50000);
}
