/*
 *  dmachinemon / a distributed machine monitor by dancer.
 *  Copyright (C) 2001 Junichi Uekawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * command-line parser
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "config.h"
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif
#include "dmachinemon/dmachinemon-commandlineparser.h"

/**
   Print some common command-line information
 */
void
dmachinemon_print_commandline_help (void)
{
  printf(
	 "-c   --clientport               Specify the client port\n"
	 "-n   --nodeport                 Specify the node port\n"
	 "-p   --parent                   Specify the parent host name\n"
	 "-s   --sleep                    Specify the interval to sleep between actions\n"
	 "-m   --maxreconnect             Number of times it will try to connect to uplink before giving up\n"
	 "-l   --layers                   Number of layers it allows for message to pass (Message TTL)\n"
	 "-d   --dieonload                Die on load average of specified value\n"
	 "-D   --Downlinknum              Number of downlinks to have for each servent\n"
	 "-h   --help                     Give out this message\n"
	 "-L d --static-link d            Use static link for downlink\n"
	 "-L u --static-link u            Use static link for uplink\n"
	 "-q   --query-opts               Query the current list of options\n"
	 "-t   --threshold                Change the clientseencount threshold\n"
	 "-r   --rsh                      Use rsh/ssh to connect\n"
 );
}

static void 
initialize_cdat (dm_commandoption * cdat)
{
  cdat->port_client = "3007" ;
  cdat->port_nodes = "3008" ;
  cdat->parenthostname = NULL ;
  cdat->sleeptime = 15 ;
  cdat->max_num_reconnect = 3;
  cdat->number_of_hosts_of_seenby = 0; /* default is none, or no checking. */
  cdat->dieonload_flag= 0;
  cdat->dieonload=0.0;
  cdat->downlinknum_threshold=4;
  cdat->downlink_static=0;
  cdat->uplink_static=0;
  cdat->debuglevel=1;
  cdat->clientseencount_threshold=50;
  cdat->rsh=NULL;
}

static void
query_cdat (const dm_commandoption * cdat)
{
  fprintf (stderr, "List of current options\n");
#define PRINTIT(A,S) fprintf (stderr, "%s: "S"\n", #A, cdat->A);
  PRINTIT(port_client, "%s");
  PRINTIT(port_nodes, "%s");
  PRINTIT(parenthostname, "%s");
  PRINTIT(number_of_hosts_of_seenby, "%i");
  PRINTIT(max_num_reconnect, "%i");
  PRINTIT(sleeptime, "%i");
  PRINTIT(dieonload, "%f");
  PRINTIT(downlinknum_threshold, "%i");
  PRINTIT(downlink_static, "%i");
  PRINTIT(uplink_static, "%i");
  PRINTIT(debuglevel, "%i");
  PRINTIT(clientseencount_threshold, "%i");
  PRINTIT(rsh, "%s");
}


/**
   Parse command-line options for dmachinemon
 */
void
dmachinemon_parse_options (int ac /** argc from main.*/
			   , char ** av, /** argv from main. */
			   dm_commandoption * cdat /** pointer to the option */
			   )
{
  int index_point;  
  int c;			/* option */

#ifdef HAVE_GETOPT_LONG
  static struct option long_options[]=
    {
      {"clientport", required_argument, 0, 'c'}, 
      {"nodeport", required_argument, 0, 'n'}, 
      {"parent", required_argument, 0, 'p'}, 
      {"sleep", required_argument, 0, 's'}, 
      {"layers", required_argument, 0, 'l'}, 
      {"maxreconnect", required_argument, 0, 'm'}, 
      {"dieonload", required_argument, 0, 'd'}, 
      {"Downlinknum", required_argument, 0, 'D'}, 
      {"static-link", required_argument, 0, 'L'}, 
      {"query-opts", no_argument, 0, 'q'}, 
      {"verbose", required_argument, 0, 'v'}, 
      {"help", no_argument, 0, 'h'}, 
      {"threshold", required_argument, 0, 't'}, 
      {"rsh", required_argument, 0, 'r'}, 
      {0,0,0,0}
    };
#else
#define getopt_long(a,b,c,d,e) getopt(a,b,c)
#endif

  initialize_cdat (cdat);
  
  while((c = getopt_long (ac, av, "c:n:p:s:hl:m:d:D:qv:L:t:r:", long_options, &index_point)) != -1)
    {
      switch (c)
	{
	case 'c':		/* client port */
	  cdat->port_client = strdup(optarg);
	  break;
/*MANUAL([.TP
.BI "--clientport|-c [" "port number for clients" "]"
The port number a 
.B "dmachinemon-gtkiface"
or a 
.B "dmachinemon-htmloutput"
program may connect to.
])*/
	case 'n':		/* nodes port */
	  cdat->port_nodes = strdup(optarg);
	  break;
/*MANUAL([.TP
.BI "--nodeport [" "upstream port number" "]"
The port number for the upstream. 
.B "servent"
connects using this port, and 
other 
.B "servent" 
will connect to the system through this port.
This value needs to be consistent throughout the cluster system.
])*/
	case 'p':		/* parent hostname */
	  cdat->parenthostname = strdup(optarg);
	  break;
/*MANUAL([.TP
.BI "--parent [" "parent host name" "]"
The name of the master node, or an intermediate servent system.
Not required for the master node.
])*/
	case 's':		/* sleep time */
	  cdat->sleeptime = atoi(optarg);
	  break;
/*MANUAL([.TP
.BI "--sleep|-s [" "wait time" "]"
The amount of time to wait for individual access in seconds.
])*/
	case 'l':		/* relaying layer number */
	  cdat->number_of_hosts_of_seenby = atoi(optarg);
	  break;
/*MANUAL([.TP
.BI "--layers|-l [" "number of layers" "]"
The number of layers the information available at this node will
be passed on to.
Default value is 
.B "0"
and it means that information is always relayed upwards.

This parameter limits that behavior, so that 
it is possible to create large networks without having 
concerns about overloaded uplink.
])*/
	case 'm':		/* max number of reconnects */
	  cdat->max_num_reconnect = atoi(optarg);
	  break;
/*MANUAL([.TP
.BI "--maxreconnect|-m [" "number of reconnect to allow" "]"
The number of times servent will try to reconnect before giving up
])*/
	case 'd':		/* die on load */
	  cdat->dieonload_flag = 1;
	  cdat->dieonload = atof(optarg);
	  break;
/*MANUAL([.TP
.BI "--dieonload|-d [" "max load average" "]"
The node will die when load average exceeds the specified value.
])*/
	case 'D':		/* Number of downlinks */
	  cdat->downlinknum_threshold = atoi(optarg);
	  break;
/*MANUAL([.TP
.BI "--Downlinknum|-D [" "number of downlink" "]"
The number of downlinks a host will serve until it will start trying
to relink.
])*/
	case 'q':
	  query_cdat(cdat);
	  break;
/*MANUAL([.TP
.BI "--query-opts|-q"
dumps the current list of options configured by the 
command-line. This exists for diagnostic purposes.
])*/
	case 'v':
	  cdat->debuglevel = atoi (optarg);
	  break;
/*MANUAL([.TP
.BI "--verbose|-v [" "verbosity level" "]"
sets the verbosity level, default is 1.
Increase to get more debug information.

Setting this value to 0 will disable all output,
and make the application daemonize.
])*/
	case 'L':		/* static links for */
	  switch (*optarg)
	    {
	    case 'u': case 'U':
	      cdat -> uplink_static = 1;
	      break;
	    case 'd': case 'D':
	      cdat -> downlink_static = 1;
	      break;
	    default:
	      dmachinemon_print_commandline_help();
	      exit (1);
	    }
	  break;
/*MANUAL([.TP
.BI "--static-link|L [" "u|d" "]"
The uplink, or the downlink, will be handled to be static link, and 
they will not be dynamically rearranged.
.B "u"
prohibits relinking when uplink dies.
.B "d"
prohibits sending messages to relink when there are too many
downlinks.

.B "d" 
is the only option useful to master node.
])*/
	case 't':
	  cdat->clientseencount_threshold = atoi(optarg);
	  break;
/*MANUAL([.TP
.BI "--threshold|-t [" "clientseencount threshold" "]"
sets the threshold for how many times a data to be seen to be considered obsolete data.
A client may request data from this servent before a new data comes, and sometimes
re-sending the data every time may be a bad idea.
])*/
	case 'r':
	  cdat->rsh = strdup(optarg);
	  break;
/*MANUAL([.TP
.BI "--rsh|-r [" "rsh-application" "]"
Uses remote shell to connect to uplinks instead of using TCP/IP.
ssh or rsh or anything compatible can be used.
servents will connect to uplink using the option.
])*/
	case 'h':
	  dmachinemon_print_commandline_help();
	  exit(1);
	  break;	  
/*MANUAL([.TP
.BI "--help|-h"
Gives out a help message.
])*/
	default:
	  dmachinemon_print_commandline_help();
	  exit(1);
	  break;	  
	}
    }
}

