/*
 * sunlite_base.c
 *
 * Copyright (C) Michael Stickel <michael@cubic.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
/*
 * This file contains some low-level functions to access the USBDMX interface.
 *
 */

#include "dmxsunlite.h"
#include "dmxsunlite_firmware.h"


/*-----------------------------------------------------------
 *-- usberror2string
 *-- Returns the usb error in a human readable form.
 *---------------------------------------------------------*/
char *usberror2string (int e)
{
  switch (e)
    {
    case USB_ST_NOERROR:           return "USB_ST_NOERROR";
    case USB_ST_CRC:               return "USB_ST_CRC";
#if 0
    case USB_ST_BITSTUFF:          return "USB_ST_BITSTUFF";     /* == USB_ST_INTERNALERROR */
    case USB_ST_NORESPONSE:        return "USB_ST_NORESPONSE";   /* == USB_ST_TIMEOUT */
    case USB_ST_DATAUNDERRUN:      return "USB_ST_DATAUNDERRUN"; /* == USB_ST_SHORT_PACKET */
#endif
    case USB_ST_DATAOVERRUN:       return "USB_ST_DATAOVERRUN";
    case USB_ST_BUFFEROVERRUN:     return "USB_ST_BUFFEROVERRUN";
    case USB_ST_BUFFERUNDERRUN:    return "USB_ST_BUFFERUNDERRUN";
    case USB_ST_INTERNALERROR:     return "USB_ST_INTERNALERROR: unknown error";
    case USB_ST_SHORT_PACKET:      return "USB_ST_SHORT_PACKET";
    case USB_ST_PARTIAL_ERROR:     return "USB_ST_PARTIAL_ERROR: ISO transfer only partially completed";
    case USB_ST_URB_KILLED:        return "USB_ST_URB_KILLED: URB canceled by user";
    case USB_ST_URB_PENDING:       return "USB_ST_URB_PENDING";
    case USB_ST_REMOVED:           return "USB_ST_REMOVED: device not existing or removed";
    case USB_ST_TIMEOUT:           return "USB_ST_TIMEOUT: communication timed out, also in urb->status";
    case USB_ST_NOTSUPPORTED:      return "USB_ST_NOTSUPPORTED";
    case USB_ST_BANDWIDTH_ERROR:   return "USB_ST_BANDWIDTH_ERROR: too much bandwidth used";
    case USB_ST_URB_INVALID_ERROR: return "USB_ST_URB_INVALID_ERROR: invalid value/transfer type";
    case USB_ST_URB_REQUEST_ERROR: return "USB_ST_URB_REQUEST_ERROR: invalid endpoint";
    case USB_ST_STALL:             return "USB_ST_STALL: pipe stalled, also in urb->status";
    }
    return "unknown error";
}


/*-----------------------------------------------------------
 *-- dmxusb_receive_bulk
 *--
 *-- Requests data using a bulk message.
 *---------------------------------------------------------*/
int dmxusb_receive_bulk (struct dmxusb_interface *u2d_if, int pipe_id, char *buffer, int size)
{
  int actual_length = 0;
  int status = usb_bulk_msg (u2d_if->dmx_dev,
			     usb_rcvbulkpipe(u2d_if->dmx_dev, pipe_id),
			     buffer, size, &actual_length,
			     DMXSUNLITE_TIMEOUT);
  if (status < 0)
    {
      printk ("dmxusb_receive_bulk(pipe=%d) USB-Error: %s\n", pipe_id, usberror2string (status));
      return -1;
    }

  return actual_length;
}


/*-----------------------------------------------------------
 *-- dmxusb_send_bulk
 *--
 *-- Sends data to the interface using a bulk message.
 *---------------------------------------------------------*/
int dmxusb_send_bulk (struct dmxusb_interface *u2d_if, int pipe_id, char *buffer, int size)
{
  int actual_length = 0;
  int status = usb_bulk_msg (u2d_if->dmx_dev,
			     usb_sndbulkpipe(u2d_if->dmx_dev, 1 /* pipe_id */),
			     buffer, size, &actual_length,
			     DMXSUNLITE_TIMEOUT);
  if (status < 0)
    {
      printk ("dmxusb_send_bulk(pipe=%d) USB-Error: %s\n", pipe_id, usberror2string (status));
      return -1;
    }
  return actual_length;
}




/*-----------------------------------------------------------
 *-- udmx_read_ports
 *--
 *-- Reads the 10 digital input-ports from the USBDMX1 interface.
 *---------------------------------------------------------*/
int udmx_read_ports (struct dmxusb_interface *usbdmx_if)
{
  unsigned char buf[0x20];
  int status;

  printk ("udmx_read_ports ()\n");

  status = dmxusb_receive_bulk (usbdmx_if, 0, buf, sizeof(buf));
  if (status > 0)
    {
      info("udmx_read_ports: got %d bytes\n", status);
    }
  else if (status < 0)
    {
      /* some error */
      return status;
    }
  return 0;
}



/*-----------------------------------------------------------
 *-- udmx_init_1
 *--
 *-- Sends the first init sequence to the USBDMX1 interface.
 *---------------------------------------------------------*/
int udmx_init_1 (struct dmxusb_interface *usbdmx_if)
{
  static unsigned char init1[] = {
    0x90, 0x51, 0x02, 0x00, 0x00, 0x4b, 0xe7, 0x77
  };
  static unsigned char init2[] = {
    0xa0, 0x51, 0x01, 0x00, 0x00, 0x4b, 0xe7, 0x77
  };
  int status;

  printk("udmx_init_1: 1\n");

  status = dmxusb_send_bulk (usbdmx_if, 0, init1, sizeof(init1));
  if (status > 0)
    info ("udmx_init_1: wrote %d bytes from first packet\n", status);
  else if (status < 0)
    return status;

  printk("udmx_init_1: 2\n");


  status = dmxusb_send_bulk (usbdmx_if, 0, init2, sizeof(init2));
  if (status > 0)
    info ("udmx_init_1: wrote %d bytes from second packet\n", status);
  else if (status < 0)
    return status;

  printk("udmx_init_1: 3\n");

  return 0;
}



/*-----------------------------------------------------------
 *-- udmx_init_2
 *--
 *-- Sends the second init sequence to the USBDMX1 interface.
 *---------------------------------------------------------*/
int udmx_init_2(struct dmxusb_interface *usbdmx_if)
{
  static unsigned char init1[] = {
    0x90, 0x51, 0x02, 0x00, 0x06, 0x9a, 0xee, 0x33
  };
  static unsigned char init2[] = {
    0xa0, 0x51, 0x01, 0x00, 0x06, 0x9a, 0xee, 0x33
  };
  int status;

  printk("udmx_init_2: 1\n");

  status = dmxusb_send_bulk (usbdmx_if, 0, init1, sizeof(init1));
  if (status > 0)
    info ("udmx_init_2: wrote %d bytes from first packet\n", status);
  else if (status < 0)
    return status;

  printk("udmx_init_2: 2\n");

  status = dmxusb_send_bulk (usbdmx_if, 0, init2, sizeof(init2));
  if (status > 0)
    info ("udmx_init_2: wrote %d bytes from second packet\n", status);
  else if (status < 0)
    return status;

  printk("udmx_init_2: 3\n");

  return 0;
}



/*-----------------------------------------------------------
 *-- udmx_init_3
 *--
 *-- Sends the third init sequence to the USBDMX1 interface.
 *---------------------------------------------------------*/
int udmx_init_3(struct dmxusb_interface *usbdmx_if)
{
  int status = 0;

  static unsigned char init[] = {
    0xb0, 0x00, 0x00, 0xce, 0x9c, 0x0a, 0xb2, 0x10,
    0x6a, 0x2a, 0xe1, 0x71, 0x69, 0xbf, 0x3b, 0x5b,
    0xce, 0x15, 0x00, 0x10, 0x06, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00
  };

  printk("udmx_init_3: 1\n");

  status = dmxusb_send_bulk (usbdmx_if, 0, init, sizeof(init));
  if (status > 0)
    info ("udmx_init_3: wrote %d bytes\n", status);
  else if (status < 0)
    return status;

  printk("udmx_init_3: 2\n");

  return 0;
}


/*-----------------------------------------------------------
 *-- udmx_init
 *--
 *-- Initializes the USBDMX1 interface.
 *---------------------------------------------------------*/
int udmx_init (struct dmxusb_interface *usbdmx_if)
{
  int error;

  printk("udmx_init: 1\n");

  if ((error = udmx_init_1(usbdmx_if)))
    return error;

  printk("udmx_init: 2\n");

  if ((error = udmx_read_ports(usbdmx_if)))
    return error;

  printk("udmx_init: 3\n");

  if ((error = udmx_init_2(usbdmx_if)))
    return error;

  printk("udmx_init: 4\n");

  if ((error = udmx_read_ports(usbdmx_if)))
    return error;

  printk("udmx_init: 5\n");

  if ((error = udmx_init_3(usbdmx_if)))
    return error;

  printk("udmx_init: 6\n");

  return 0;
}


/*-----------------------------------------------------------
 *-- udmx_in_init
 *--
 *-- Initializes the USBDMX-IN interface.
 *---------------------------------------------------------*/
int udmx_in_init (struct dmxusb_interface *usbdmx_if)
{
  int rc;
  unsigned char inbuffer[512];

  /*
    rc = usb_get_device_descriptor(usbdmx_if->dmx_dev);
    printk("bf: usb_get_device_descriptor %d\n", rc);
  */

  rc = usb_control_msg(usbdmx_if->dmx_dev,
		       usb_rcvctrlpipe(usbdmx_if->dmx_dev, 0),
		       0x10,
		       USB_DIR_IN | USB_TYPE_VENDOR,
		       0, 0, inbuffer, 1, USBDMX2_TIMEOUT);

  printk("usb_control_msg (request 0x10) sent: rc=%d\n", rc);

  return 0;
}




/*-----------------------------------------------------------
 *-- udmx_init_packet
 *--
 *-- Initializes the USBDMX bulkout buffer.
 *---------------------------------------------------------*/
void udmx_init_packet (unsigned char packet[])
{
	int p, a, s, i;

	for (i = 0; i < PACKET_SIZE; ++i)
		packet[i] = 0;

	for (p = 0; p < 26; ++p) {
		a = p * 32;
		s = p * 20;

		packet[a] = 0x80;
		packet[a+1] = s / 2;
		packet[a+2] = 0x84;
		packet[a+7] = s / 2 + 2;
		packet[a+8] = 0x84;
		packet[a+13] = s / 2 + 4;
		if (p < 25) {
			packet[a+14] = 0x84;
			packet[a+19] = s / 2 + 6;
			packet[a+20] = 0x84;
			packet[a+25] = s / 2 + 8;
			packet[a+26] = 0x04;
			packet[a+31] = 0x00;
		} else /* the last packet ist short */
			packet[a+14] = 0x04;
	}
}




/*-----------------------------------------------------------
 *-- UDMX_SET_SLOT
 *--
 *-- Sets one channel in the bulkout buffer.
 *---------------------------------------------------------*/
#define UDMX_SET_SLOT(packet, slot, value) \
  (packet)[((slot) / 20) * 32		\
	   + (((slot) / 4) % 5) * 6	\
	   + 3				\
	   + ((slot) % 4)] = (value);


/*-----------------------------------------------------------
 *-- udmx_do_write
 *--
 *-- Writes <size> channels with offset <offs> from <dmx_values>
 *-- to the bulkout-buffer of <usbdmx_if>.
 *---------------------------------------------------------*/
int udmx_do_write (struct dmxusb_interface *usbdmx_if, int offs, int size, unsigned char *dmx_values)
{
  int           i;

  for (i = 0; i < size; ++i)
    UDMX_SET_SLOT((char *) usbdmx_if->bulkout_buffer, offs+i, dmx_values[i]);
  return 0;
}



/*-----------------------------------------------------------
 *-- udmx_update
 *--
 *-- Sends the current dmx-values to
 *-- the interface.
 *---------------------------------------------------------*/
int udmx_update (struct dmxusb_interface *usbdmx_if)
{
  int status = dmxusb_send_bulk (usbdmx_if, 0, usbdmx_if->bulkout_buffer, PACKET_SIZE);
  if (status > 0)
    ; /* info ("udmx_init_3: wrote %d bytes\n", status); */
  else if (status < 0)
    return status;

  return 0;
}



/*-----------------------------------------------------------
 *-- ezusb_writememory
 *--
 *-- Writes <length> bytes from the buffer <data> at <address>
 *-- into the memory of the EzUSB device.
 *---------------------------------------------------------*/
/* EZ-USB Control and Status Register.  Bit 0 controls 8051 reset */
#define CPUCS_REG    0x7F92

static int ezusb_writememory (struct usb_device *dev, int address, unsigned char *data, int length, __u8 bRequest)
{
  int result;
  unsigned char *transfer_buffer =  kmalloc (length, GFP_KERNEL);

  if (!transfer_buffer)
    {
      printk( "ezusb_writememory - kmalloc(%d) failed.", length);
      return -ENOMEM;
    }
  memcpy (transfer_buffer, data, length);
  result = usb_control_msg (dev, usb_sndctrlpipe(dev, 0), bRequest, 0x40, address, 0, transfer_buffer, length, 300);
  kfree (transfer_buffer);
  return result;
}



/*-----------------------------------------------------------
 *-- USBDMX2_startup
 *--
 *-- Uploads the firmware to the USBDMX2 interface.
 *---------------------------------------------------------*/
int  USBDMX2_startup (struct usb_device *dev, __u16 idProduct, const struct sunlite_hex_record *firmware)
{
  int response;
  const struct sunlite_hex_record *record = firmware;

  dbg("command_port_read_callback");

  printk ("sunlite_startup()\n");

#if 0
  if (idProduct == USB_PRODUCT_EZ_USB2)
    record = &sunlite_firmware[0];
  else if (idProduct == USB_PRODUCT_EZ_USB3)
    record = &sunlite_firmware_in[0];
  else {
    printk ("no firmware for device found\n");
    return 0;
  }
#endif
  if (!record)
    {
      printk ("no firmware for device available\n");
      return 0;
    }

  while (record->address != 0xffff)
    {
      response = ezusb_writememory (dev, record->address,
					(unsigned char *)record->data, record->data_size, 0xa0);
      if (response < 0)
        {
          printk("sunlite_startup - ezusb_writememory failed for loader (%d %04X %p %d)",
		     response, record->address, record->data, record->data_size);
          return -1;
        }
      ++record;
    }

  printk ("sunlite_startup - device is ready\n");
      /* we want this device to fail to have a driver assigned to it. */
  return 0;
}

void bulk_urb_complete (struct urb *purb) {

  unsigned char *ptmp_buffer;
  struct dmxusb_universe *u2d_u;
  struct dmxusb_interface *u2d_if = (struct dmxusb_interface *)purb->context;

  if (purb->status)
    info("purb->status: %d", purb->status);
  else if (purb->actual_length != USBDMX_IN_BULK_BUF_SIZE)
    ; /*info("purb->actual_length 0x%x != 0x%x", purb->actual_length, USBDMX_IN_BULK_BUF_SIZE);*/
  else if (!purb->context)
    info("no purb->context");
  else {
    ptmp_buffer = (unsigned char *)purb->transfer_buffer + 3; /* offset 3 */
    u2d_u = u2d_if->universe;
    if (u2d_u  && u2d_u->universe) {
      if (memcmp(u2d_u->buffer, ptmp_buffer, USBDMX_IN_BULK_BUF_SIZE-3)) {
	/*printk("received {$%02X,$%02X,$%02X,$%02X,$%02X}\n", ptmp_buffer[0],ptmp_buffer[1],ptmp_buffer[2],ptmp_buffer[3], ptmp_buffer[4]);*/

	memcpy(u2d_u->buffer, ptmp_buffer, USBDMX_IN_BULK_BUF_SIZE-3);

	u2d_u->data_avail=1;
	u2d_u->universe->signal_changed (u2d_u->universe, 0, USBDMX_IN_BULK_BUF_SIZE-3);

      }
    } else
      info("no universe");
  }

  atomic_set(&u2d_if->urb_submit_pending, 0);

}
