/* main.c - Main routines for dnotify
 *
 * Copyright (C) 2002  Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <stdarg.h>
#include <errno.h>
#include <getopt.h>
#include <dirent.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "dnotify.h"
#include "gettext.h"
#define _(s) gettext(s)
#define N_(s) gettext_noop(s)

#define HELP_MESSAGE \

typedef struct {
    int fd;
    char *name;
} MonitorFD;

typedef struct {
    void *next;
    int fd;
} QueueFD;

static QueueFD *first_queue_fd = NULL;
static QueueFD *last_queue_fd = NULL;
static char *prepend_dir = NULL;

static const char *const short_opts = "-ABCDMRabef:op:q:rst:";
static struct option long_opts[] = {
    { "access",             no_argument,        NULL, 'A' },
    { "modify",             no_argument,        NULL, 'M' },
    { "create",             no_argument,        NULL, 'C' },
    { "delete",             no_argument,        NULL, 'D' },
    { "rename",             no_argument,        NULL, 'R' },
    { "attrib",             no_argument,        NULL, 'B' },
    { "all",                no_argument,        NULL, 'a' },
    { "execute",            no_argument,        NULL, 'e' },
    { "file",               required_argument,  NULL, 'f' },
    { "processes",          required_argument,  NULL, 'p' },
    { "queue",              required_argument,  NULL, 'q' },
    { "times",              required_argument,  NULL, 't' },
    { "once",               no_argument,        NULL, 'o' },
    { "recursive",          no_argument,        NULL, 'r' },
    { "background",         no_argument,        NULL, 'b' },
    { "silent",             no_argument,        NULL, 's' },
    { "quiet",              no_argument,        NULL, 's' },
    { "version",            no_argument,        NULL, 'V' },
    { "help",               no_argument,        NULL, 'H' },
    { 0,                    0,                  0,    0,  },
};

static void
sigrt_handler(int sig, siginfo_t *si, void *data)
{
    /* No operation. */
}

static void
sigchld_handler(int sig)
{
    /* No operation. */
}

/* errno must be set to zero prior to calling readdir, otherwise
 * there is no way to distinguish between EOF and an error.
 * GNU libc does not modify errno on EOF or success.
 */
static struct dirent *
safe_readdir(DIR *dir)
{
    errno = 0;
    return readdir(dir);
}

static char *
catfiles(const char *path1, const char *path2)
{
    char *out;
    char *sep;

    if (path1[0] == '\0' && path2[0] == '\0')
    	return strdup("/");
    if (path2[0] == '\0')
    	return strdup(path1);
    if (path1[0] == '\0')
    	return strdup(path2);

    sep = (path1[strlen(path1)-1] == '/' ? "" : "/");
    if (asprintf(&out, "%s%s%s", path1, sep, path2) < 0)
    	return NULL;

    return out;
}

static void
add_directory(const char *dirname, MonitorFD **monv, int *monc, int *monvsize, int recursive)
{
    int fd;

    fd = open(dirname, O_RDONLY);
    if (fd < 0)
	die(_("cannot open directory `%s' - %s"), dirname, errstr);

    if ((*monc)+1 >= *monvsize) {
	*monvsize = (*monvsize == 0 ? 8 : (*monvsize) * 2);
	*monv = realloc(*monv, (*monvsize) * sizeof(MonitorFD));
    }
    (*monv)[*monc].fd = fd;

    if (prepend_dir != NULL && dirname[0] != '/') {
    	(*monv)[*monc].name = catfiles(prepend_dir, dirname);
    } else {
    	(*monv)[*monc].name = strdup(dirname);
    }
    if ((*monv)[*monc].name == NULL)
	die("%s", errstr);
	
    (*monc)++;

    if (recursive) {
	DIR *dir;
	struct dirent *ep;

	dir = opendir(dirname);
	if (dir == NULL)
	    die(_("cannot open directory `%s' - %s"), dirname, errstr);

	while ((ep = safe_readdir(dir)) != NULL) {
	    if ((strcmp(ep->d_name, ".") == 0 || strcmp(ep->d_name, "..") == 0))
		continue;

	    if (ep->d_type == DT_UNKNOWN) {
		char *subdirname;
		struct stat statbuf;

		/* I admit this is slow but modern libc's don't return DT_UNKNOWN */
		subdirname = concat_dirname(dirname, ep->d_name);

		if (lstat(subdirname, &statbuf) < 0)
			die(_("cannot stat file `%s' - %s"), subdirname, errstr);
		if (S_ISDIR(statbuf.st_mode))
			add_directory(subdirname, monv, monc, monvsize, recursive);
		free(subdirname);
	    }
	    else if (ep->d_type == DT_DIR) {
		char *subdirname;

		subdirname = concat_dirname(dirname, ep->d_name);
		add_directory(subdirname, monv, monc, monvsize, recursive);
		free(subdirname);
	    }
	    /* subdirname will be freed when the MonitorFD is freed! */
	}

	if (errno != 0)	/* assume readdir sets errno to 0 if all is ok */
	    die(_("cannot read directory `%s' - %s"), dirname, errstr);
	if (closedir(dir) < 0)
	    die(_("cannot read directory `%s' - %s"), dirname, errstr);
    }
}

/* This function returns non-zero if a child had terminated. */
static int
wait_process(void)
{
    int status;
    int rc;

    rc = waitpid(-1, &status, WNOHANG);
    if (rc < 0)
	die(_("waitpid on child failed - %s"), errstr);
    if (rc > 0) {
	if (!WIFEXITED(status))
	    warn(_("child terminated abnormally"));
	if (WEXITSTATUS(status) != 0)
	    warn(_("child terminated with non-zero status"));
	return 1;
    }

    return 0;
}

static void
run_child(int fd, char **argv, int argcount, MonitorFD *monv, int monc)
{
    pid_t child;

    child = fork();
    if (child < 0)
	die(_("cannot fork child - %s"), errstr);
    if (child == 0) {
	char *child_argv[argcount == 0 ? 3 : argcount+1];
	const char *trigfile = NULL;
	int c;

	/* Try a smart guess to find monitor filename, otherwise do O(n) search */
	c = fd-monv[0].fd;
	if (c >= 0 && c < monc && monv[c].fd == fd) {
	    trigfile = monv[c].name;
	} else {
	    for (c = 0; c < monc; c++) {
		if (monv[c].fd == fd) {
		    trigfile = monv[c].name;
		    break;
		}
	    }
	    if (trigfile == NULL)
		exit(EXIT_FAILURE);
	}

	if (argcount == 0) {
	    puts(trigfile);
	    exit(EXIT_SUCCESS);
	}

	for (c = 0; c < monc; c++)
	    close(monv[c].fd); /* ignore errors */

	child_argv[0] = argv[0];
	for (c = 1; c < argcount; c++)
	    child_argv[c] = expand_braces(argv[c], trigfile);
	child_argv[argcount] = NULL;

	if (execvp(child_argv[0], child_argv) < 0)
	    die(_("cannot execute `%s' - %s"), child_argv[0], errstr);
	exit(EXIT_FAILURE);
    }
}

static int
unqueue_fd(void)
{
    QueueFD *qfd = first_queue_fd;
    int ret = qfd->fd;

    first_queue_fd = qfd->next;
    if (qfd->next == NULL)
	last_queue_fd = NULL;

    free(qfd);

    return ret;
}

static void
queue_fd(int fd)
{
    QueueFD *qfd = malloc(sizeof(QueueFD));
    if (qfd == NULL)
	die("%s", errstr);
    qfd->fd = fd;

    if (last_queue_fd != NULL)
	last_queue_fd->next = qfd;
    qfd->next = NULL;
    last_queue_fd = qfd;
    if (first_queue_fd == NULL)
	first_queue_fd = qfd;
}

int main(int argc, char **argv)
{
    MonitorFD *monv = NULL;
    int monc = 0;
    int monvsize = 0;
    char *dirv[argc-1];
    char *listv[argc-1];
    int listc = 0;
    int dirc = 0;
    struct sigaction act;
    long notifyarg = 0;
    int c;
    int background = 0;		/* corresponds to --background */
    int recursive = 0;          /* corresponds to --recursive */
    int32_t times = -1;         /* corresponds to --times */
    int32_t max_processes = 1;  /* corresponds to --processes */
    int32_t max_queue = -1;     /* corresponds to --queue */
    int32_t cur_processes = 0;
    int32_t cur_queue = 0;
    sigset_t signalset;
    char *line = NULL;
    size_t line_size = 0;
    int done = 0;

#ifdef ENABLE_NLS
    setlocale(LC_ALL, "");
    bindtextdomain(PACKAGE, LOCALEDIR);
    textdomain(PACKAGE);
#endif

    while (1) {
	c = getopt_long(argc, argv, short_opts, long_opts, NULL);
	if (c == -1 || c == 'e')
	    break;

	switch (c) {
	case 1:	/* non-option argument */
	    /*dirv[dirc] = strdup(optarg);
	    if (dirv[dirc] == NULL)
		die("%s", errstr);
	    dirc++;*/
	    dirv[dirc++] = optarg;
	    break;
	case 'f':
	    listv[listc++] = optarg;
	    break;
	case 'A':
	    notifyarg |= DN_ACCESS;
	    break;
	case 'M':
	    notifyarg |= DN_MODIFY;
	    break;
	case 'C':
	    notifyarg |= DN_CREATE;
	    break;
	case 'D':
	    notifyarg |= DN_DELETE;
	    break;
	case 'R':
	    notifyarg |= DN_RENAME;
	    break;
	case 'B':
	    notifyarg |= DN_ATTRIB;
	    break;
	case 'a':
	    notifyarg |= DN_ACCESS|DN_MODIFY|DN_CREATE|DN_DELETE|DN_RENAME|DN_ATTRIB;
	    break;
	case 's':
	    quiet = 1;
	    break;
        case 'b':
            background = 1;
            break;
	case 'r':
	    recursive = 1;
	    break;
	case 'p':
	    if (!parse_int32(optarg, &max_processes))
		die(_("invalid processes value: %s"), optarg);
	    if (max_processes <= 0)
		max_processes = -1;
	    break;
	case 'q':
	    if (!parse_int32(optarg, &max_queue))
		die(_("invalid queue value: %s"), optarg);
	    break;
	case 't':
	    if (!parse_int32(optarg, &times))
		die(_("invalid times value: %s"), optarg);
	    if (times <= 0)
		times = -1;
	    break;
	case 'o':
	    times = 1;
	    break;
	case 'V': /* --version */
            printf("%s %s\n", PACKAGE, VERSION);
            printf(_("Written by %s.\n"), AUTHOR);
            puts("\nCopyright (C) 2002-2004 Oskar Liljeblad.");
            puts(_("\
This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE."));
	    exit(EXIT_SUCCESS);
	case 'H': /* --help */
	    printf(_("Usage: %s [OPTION]... DIRECTORY... [-e COMMAND...]\n"), argv[0]);
            puts(_("Execute a command every time the contents of a directory change."));
            puts(_("\nEvents:"));
            puts(_("  -A, --access             trigger when a file in the directory was accessed"));
            puts(_("  -M, --modify             trigger when a file in the directory was modified"));
            puts(_("  -C, --create             trigger when a file was created in the directory"));
            puts(_("  -D, --delete             trigger whan a file was unlinked from the directory"));
            puts(_("  -R, --rename             trigger when a file in the directory was renamed"));
            puts(_("\
  -B, --attrib             trigger when the directory had its attributes\n\
                           changed (after chmod, chown)"));
            puts(_("  -a, --all                all of the above"));
            puts(_("\nGeneral:"));
            puts(_("\
  -e, --execute=COMMAND..  command to execute when an event is triggered\n\
                           (all remaining args are treated as command args)"));
            puts(_("  -f, --file=FILE          read directories to monitor from FILE, one per line"));
            puts(_("  -p, --processes=COUNT    max number of commands to run at a time"));
            puts(_("  -q, --queue=DEPTH        max depth of queue holding commands to be run"));
            puts(_("  -t, --times=COUNT        exit after running the command COUNT times"));
            puts(_("  -o, --once               same as `--times 1'"));
            puts(_("  -r, --recursive          monitor subdirectories too (recursively)"));
            puts(_("  -b, --background         run in background (detach)"));
            puts(_("  -s, --silent             don't print warnings about non-zero child exits"));
            puts(_("      --quiet              same as `--silent'"));
            puts(_("      --help               display this help and exit"));
            puts(_("      --version            output version information and exit"));
            puts(_("\nUnless events have been specified, the events are set to create and delete.\n"));
            puts(_("\
Specifying -1 for queue means unlimited depth. The default value for --queue\n\
is -1, and 1 for --processes.\n"));
            puts(_("\
The default command (unless specified with -e) is to print the name of the\n\
of the directory updated. The string `{}' in the command specification is\n\
replaced by the name of the directory that was updated.\n"));
            printf(_("Report bugs to <%s>.\n"), PACKAGE_BUGREPORT);
	    exit(EXIT_SUCCESS);
	case '?':
	    exit(EXIT_FAILURE);
	}
    }

    if (dirc == 0 && listc == 0)
	die(_("missing directory argument\nTry `%s --help' for more information."), argv[0]);

    if (notifyarg == 0)
	notifyarg = DN_CREATE|DN_DELETE;

    if (background) {
        pid_t child;

    	prepend_dir = get_current_dir_name();
	if (prepend_dir == NULL)
	    die(_("cannot get current directory name: %s"), errstr);

        child = fork();
        if (child < 0)
            die(_("fork failed: %s"), errstr);
        if (child > 0)
            exit(EXIT_SUCCESS);

        if (setsid() < 0)
            die(_("setsid failed: %s"), errstr);

        close(STDIN_FILENO); /* ignore errors */
        close(STDOUT_FILENO); /* ignore errors */
        close(STDERR_FILENO); /* ignore errors */
    }

    sigemptyset(&signalset);
    sigaddset(&signalset, SIGCHLD);
    sigaddset(&signalset, SIGRTMIN);

    /* Work around lpthread bug (libc/4927). */
    if (sigprocmask(SIG_UNBLOCK, &signalset, NULL) < 0)
	die(_("sigprocmask failed - %s"), errstr);

    /* Register SIGRTMIN and SIGCHLD signal handlers. */
    act.sa_sigaction = sigrt_handler;
    sigemptyset(&act.sa_mask);
    act.sa_flags = SA_SIGINFO | SA_RESTART | SA_NODEFER;
    if (sigaction(SIGRTMIN, &act, NULL) < 0)
	die(_("sigaction failed - %s"), errstr);

    act.sa_sigaction = (void (*)(int, siginfo_t *, void *)) sigchld_handler;
    act.sa_flags = SA_RESTART | SA_NODEFER | SA_NOCLDSTOP;
    if (sigaction(SIGCHLD, &act, NULL) < 0)
	die(_("sigaction failed - %s"), errstr);

    /* Block SIGRTMIN and SIGCHLD. */
    if (sigprocmask(SIG_BLOCK, &signalset, NULL) < 0)
	die(_("sigprocmask failed - %s"), errstr);

    for (c = 0; c < listc; c++) {
	FILE *file;
	ssize_t len;

	file = fopen(listv[c], "r");
	if (file == NULL)
	    die(_("%s: cannot open for reading - %s"), listv[c], errstr);
	while ((len = getline(&line, &line_size, file)) >= 0) {
	    if (len != 0) {
		if (line[len-1] == '\n')
		    line[len-1] = '\0';
		add_directory(line, &monv, &monc, &monvsize, recursive);
	    }
	}
	if (ferror(file))
	    die(_("%s: cannot read - %s"), listv[c], errstr);
	fclose(file);
    }
    for (c = 0; c < dirc; c++)
	add_directory(dirv[c], &monv, &monc, &monvsize, recursive);

    for (c = 0; c < monc; c++) {
	if (fcntl(monv[c].fd, F_SETSIG, SIGRTMIN) < 0)
	    die(_("fcntl F_SETSIG on `%s' failed - %s"), monv[c].name, errstr);
	if (fcntl(monv[c].fd, F_NOTIFY, notifyarg|DN_MULTISHOT) < 0)
	    die(_("fcntl F_NOTIFY on `%s' failed - %s"), monv[c].name, errstr);
    }

    if (background && chdir("/") < 0)
    	die(_("cannot change current directory to `/': %s"), errstr);

    while (!done) {
	siginfo_t signalinfo;
	int signal;

	signal = TEMP_FAILURE_RETRY(sigwaitinfo(&signalset, &signalinfo));
	if (signal < 0)
	    die(_("sigwaitinfo failed - %s"), errstr);

	if (signal == SIGCHLD) {
	    while (cur_processes > 0 && wait_process()) {
		cur_processes--;
		if (times == 0 && cur_processes == 0)
		    done = 1; /* Terminate program */
		if (cur_queue > 0) {
		    cur_queue--;
		    run_child(unqueue_fd(), argv+optind, argc-optind, monv, monc);
		    cur_processes++;
		    if (times > 0)
			times--;
		}
	    }
	}
	else if (signal == SIGRTMIN && (times < 0 || times > 0)) {
	    int fd = signalinfo.si_fd;

	    if (max_processes < 0 || cur_processes < max_processes) {
		run_child(fd, argv+optind, argc-optind, monv, monc);
		cur_processes++;
		if (times > 0)
		    times--;
	    } else if (max_queue < 0 || cur_queue < max_queue) {
		queue_fd(fd);
		cur_queue++;
	    }
	}
    }

    for (c = 0; c < monc; c++) {
	close(monv[c].fd); /* ignore errors */
	free(monv[c].name);
    }
    free(monv);
    free(prepend_dir);

    exit(EXIT_SUCCESS);
}
