import java.io.*;
import java.util.*;

/**
 * Class responsible for controlling data regarding the files being indexed.
 * This will construct a linked list of IGFiles, for each file to be processed,
 * call the relevant FileParser class and finally sort the data. 
 *
 * @version $Id: FileList.java,v 1.33 2002/09/22 23:13:28 blsecres Exp $
 */
class FileList {
    /** The base level for log messages */
    private static final int LOGLEVEL = 9;

    /** Log file */
    IGLog log;

    /** Project settings to use */
    Project project;

    /** List object storing the data */
    LinkedList list;


    /**
     * Set up the project and log objects.  The list may be made at a later
     * time.
     * @param useLog The object to use for log messages
     * @param useProj The project settings to use
     */
    public FileList(IGLog useLog, Project useProj) {
        log = useLog;
        project = useProj;
        list = new LinkedList();
    }


    /**
     * Populate the list using the settings from the project file as given when
     * the object was created.
     * @param wishlist The index items which are requested.
     */
    public void makeList(IGKeySet wishlist) throws IOException,
	    IllegalVariableException {
        ParserChooser parserChooser = new ParserChooser(log, wishlist);
		FileListBuilder scanner = null;
		IGLog.ui.startScan();

	log.add(4, "Generating file listings");
	if (project.getEnumSetting("GET_METHOD").equals(
		    "GET_METHOD_FILESYSTEM")) {
	    log.add(6, "Starting Filesystem get method");
	    if (project.getEnumSetting("FILE_SYSTEM_SEARCH").equals(
			"FILE_SYSTEM_SEARCH_DIR_SCAN")) {
		scanner = new DirScanner(log, parserChooser,
			project.getArraySetting("INDEX_INCLUDE_FILTERS"),
			project.getArraySetting("INDEX_EXCLUDE_FILTERS"),
			this);
	    } else if (project.getEnumSetting("FILE_SYSTEM_SEARCH").equals(
			"FILE_SYSTEM_SEARCH_LINK_FOLLOW")) {
		scanner = new LinkScanner(log, parserChooser,
			project.getArraySetting("INDEX_INCLUDE_FILTERS"),
			project.getArraySetting("INDEX_EXCLUDE_FILTERS"),
			project.getArraySetting("SCAN_INCLUDE_FILTERS"),
			project.getArraySetting("SCAN_EXCLUDE_FILTERS"),
			false, this);
	    } else {
		/* TODO */
	    }
	} else if (project.getEnumSetting("GET_METHOD").equals(
		    "GET_METHOD_SERVER")) {
	    log.add(6, "Starting server get method");
		scanner = new LinkScanner(log, parserChooser,
			project.getArraySetting("INDEX_INCLUDE_FILTERS"),
			project.getArraySetting("INDEX_EXCLUDE_FILTERS"),
			project.getArraySetting("SCAN_INCLUDE_FILTERS"),
			project.getArraySetting("SCAN_EXCLUDE_FILTERS"),
			true, this);
		parserChooser.setHTTPHeaders(
			project.getArraySetting("HTTP_HEADERS"));
	} else {
	    log.add(1, "Unknown get method: "
		    + project.getEnumSetting("GET_METHOD"));
	    /* TODO */
	}

	if (scanner != null)
	    scanner.scan(project.getArraySetting("SCAN_ROOT"));

	log.add(5, "Generated list with " + count() + " items");
    }


    /**
     * Adds an individual file to the list
     * @param file The IGFile object to add to the list
     */
    protected void addFile(IGFile file) throws IllegalVariableException {
	log.add(6, "Adding file "+file.getLocation());
	// todo: only do this if sitemap type index?
	file.put(IGKey.FILE_PARTS, IGMisc.stringToArray(file.getLocation(),
		    System.getProperty("file.separator")));
	file.put(IGKey.REL_LOCATION, IGMisc.relative(file.getLocation(),
		    project.getStringSetting("OUTPUT_FILENAME")));

	// make sure all strings have a valid String value
	file.fillStrings();

        list.add(file);
    }


    /**
     * Access the size of the file list
     * @return The number of files in the list
     */
    public int count() {
        return list.size();
    }


    /**
     * Access the LinkedList used to store IGFile objects
     * @return A LinkedList of IGFile's
     */
    public LinkedList getList() {
        return list;
    }
}
/*
 * $Log: FileList.java,v $
 * Revision 1.33  2002/09/22 23:13:28  blsecres
 * Propagated exceptions that were unnecessarily caught.
 *
 * Revision 1.32  2002/09/15 20:23:06  blsecres
 * Added code to fill in all meta information strings for an IGFile object.
 *
 * Revision 1.31  2002/09/04 20:08:48  howama
 * user interface work
 *
 * Revision 1.30  2002/09/01 10:59:29  howama
 * More work and random bug fixes.
 *
 * Revision 1.29  2002/08/31 22:00:20  blsecres
 * Provided fallbacks for missing ordered list sort criteria.
 * Implemented insignificant article removal for sorting purposes.
 * Modified parsers to only provide information asked for in IGKeySet.
 *
 * Revision 1.28  2002/08/31 15:27:56  howama
 * various changes
 *
 * Revision 1.27  2002/08/30 02:27:06  blsecres
 * Initial ordered list support.
 * User configurable sorting.
 *
 * Revision 1.26  2002/08/29 18:29:08  blsecres
 * Added functionality for setting HTTP request headers.
 *
 * Revision 1.25  2002/08/27 13:32:07  blsecres
 * Renamed ParserFactory to ParserChooser.
 *
 * Revision 1.24  2002/08/27 00:07:26  blsecres
 * Rearranged IGMisc and calling classes.
 *
 * Revision 1.23  2002/08/23 13:58:01  howama
 * propogated illegalvariable exceptions rather than catching them
 *
 * Revision 1.22  2002/08/23 03:40:23  blsecres
 * Integrated LinkScanner into FileList.makeList().
 * Started coding support for 'linkfollow' based file system search
 *     in LinkScanner.
 *
 * Revision 1.21  2002/08/22 19:47:13  blsecres
 * Switched code to use IGKey values.
 *
 * Revision 1.20  2002/08/22 13:42:37  howama
 * more work
 *
 * Revision 1.19  2002/08/20 13:24:37  howama
 * modified exeption handling
 *
 * Revision 1.18  2002/08/19 18:45:34  blsecres
 * Fixed bugs in and added features to LinkScanner and calling classes.
 *
 * Revision 1.17  2002/08/16 10:34:47  howama
 * new IGFile
 *
 * Revision 1.16  2002/08/11 19:52:20  howama
 * Removed things which shouldn't have been here. This now does not deal with any
 * particular index generator implementation. Also changed to using a LinkedList
 * object at some point.
 *
 * Revision 1.15  2002/08/10 15:01:24  howama
 * before rewriting. Implemented sort method - really badly. About to convert the
 * whole thing to use a LinkedList object.
 *
 * Revision 1.14  2002/08/10 12:41:49  howama
 * Directory sort routines
 *
 * Revision 1.13  2002/08/09 17:43:10  howama
 * removed unused RE import
 *
 * Revision 1.12  2002/08/09 16:17:15  howama
 * exception handling changes
 *
 * Revision 1.11  2002/08/09 13:04:14  howama
 * added getfirst method
 *
 * Revision 1.10  2002/08/09 10:30:44  howama
 * updates for changed Project class
 *
 * Revision 1.9  2002/08/08 20:10:57  howama
 * updates
 *
 * Revision 1.8  2002/08/08 17:19:19  howama
 * more work + fixes
 *
 * Revision 1.7  2002/07/30 12:57:51  divzero
 * Now non-existant directories are caught.
 *
 * Will.
 *
 * Revision 1.6  2002/07/24 09:24:21  howama
 * updated javadoc docs
 *
 * Revision 1.5  2002/07/17 15:07:45  howama
 * various work
 *
 * Revision 1.4  2002/07/12 14:07:31  howama
 * updates
 *
 * Revision 1.3  2002/07/11 18:25:02  howama
 * starting work
 *
 * Revision 1.2  2002/07/05 10:52:01  howama
 * Added log tag
 *
 */
