import java.io.*;
import java.util.Collections;
import java.util.Comparator;
import gnu.regexp.*;

/**
 * Generates an ordered-list style index.
 *
 * @version $Id: OrderedListGen.java,v 1.25 2002/09/24 14:32:49 howama Exp $
 * @author Mark Howard, Ben Secrest &lt;blsecres@users.sourceforge.net&gt;
 */
public class OrderedListGen extends AnyIndexGen {
    /** The highest level of log messages generated */
    private static final int LOGLEVEL = 9;

    /** The log file */
    private IGLog log;

    /** Project data */
    private Project project;

    /** File List object */
    private FileList list;

    /** Name of the output file */
    private String outputFilename;

    /** Replacer for embedded strings */
    private IGReplace replacer;

    /** The chosen sort method */
    private IGKey sortKey;

    /** Articles to trim off strings */
    private String[] articles;


    /**
     * Constructor will set up links to various objects.
     *
     * Assumptions: 
     *   - Project settings have been loaded
     *   - Log has been started
     *   
     * @param iglog An instantiated log object
     * @param projData Project data object
     */
    public OrderedListGen(IGLog iglog, Project projData)
	    throws IllegalVariableException, REException {
        log = iglog;
        project = projData;
        log.add(9, "OrderedListGen Object Created");

	outputFilename = project.getStringSetting("OUTPUT_FILENAME");

	sortKey = getSortKey(project.getEnumSetting("OL_SORT_KEY"));

	articles = project.getArraySetting("OL_ARTICLES");

	replacer = new IGReplace(log, project);
    }


    /**
     * Begins the generation process.
     */
    public void generate() throws java.io.IOException,
	    IllegalVariableException, REException, org.apache.oro.text.regex.MalformedPatternException {
	if (LOGLEVEL >= IGLog.SECTION)
	    log.add(IGLog.SECTION, "Starting Ordered List Generation");

        list = new FileList(log, project);

        IGKeySet wishlist = IGReplace.getWanted(log, project);
		list.makeList(wishlist);

	Collections.sort(list.getList(), new IGFileComparator(sortKey,
		    articles));
	
	writeOutput();
    }


    /**
     * Write out the html file
     */
    private void writeOutput() throws IllegalVariableException, IOException,
	    REException {
        if (! outFileAcceptable(log, outputFilename))
	    // throw new IOException
	    return;

	// object used to write to the output file
	BufferedWriter fileWriter = new BufferedWriter(
		new OutputStreamWriter(
		    new FileOutputStream(outputFilename), "UTF8"));
	// object used to collect the navigation body
	StringWriter navWriter = null;
	// object used to collect the ordered list body
	StringWriter olWriter = new StringWriter();

	fileWriter.write(replacer.makeCode("OL_START_TXT")
		+ IGMisc.lineSeparator);

	if (! (sortKey.equals(IGKey.FILE_SIZE)
		    || sortKey.equals(IGKey.FILE_TYPE))) {
	    navWriter = new StringWriter();
	    navWriter.write(replacer.makeCode("OL_NAV_START")
		    + IGMisc.lineSeparator);
	}

	// object to replace ordered list links
	IGReplace linkReplacer = new IGReplace(log, project,
		project.getStringSetting("OL_LINK"),
		IGKey.REPLACE_TYPE_FILE,
		    "OL_LINK");
	// object to replace characters for ordered list navigation
	IGReplace navReplacer = new IGReplace(log, project,
		project.getStringSetting("OL_NAV_LINK"),
		IGKey.REPLACE_TYPE_SECTION, 
		    "OL_NAV_LINK");
	// object to replace section headers in ordered list
	IGReplace sectionReplacer = new IGReplace(log, project,
		project.getStringSetting("OL_SEC_START"),
		IGKey.REPLACE_TYPE_SECTION,
		    "OL_SEC_START");

	char currentChar = '\0';

	for (java.util.Iterator i = list.getList().iterator(); i.hasNext();
		) {
	    IGFile file = (IGFile) i.next();
	    String curString = trimArticles(file.getString(sortKey),
		    articles);

	    // provide section characters for Alphabetical sort methods
	    if (! (sortKey.equals(IGKey.FILE_SIZE)
			|| sortKey.equals(IGKey.FILE_TYPE))
		    && Character.toUpperCase(curString.charAt(0))
		    != currentChar) {
		if (currentChar != '\0')
		    olWriter.write(replacer.makeCode("OL_SEC_END"));

		currentChar = Character.toUpperCase(curString.charAt(0));
		navWriter.write(navReplacer.makeCode(currentChar)
			+ IGMisc.lineSeparator);
		olWriter.write(sectionReplacer.makeCode(currentChar)
			+ IGMisc.lineSeparator);
	    }

	    olWriter.write(linkReplacer.makeCode(file)
		    + IGMisc.lineSeparator);
	}

	if (! (sortKey.equals(IGKey.FILE_SIZE)
		    || sortKey.equals(IGKey.FILE_TYPE))) {
	    navWriter.write(replacer.makeCode("OL_NAV_END")
		+ IGMisc.lineSeparator);
	    olWriter.write(replacer.makeCode("OL_NAV_END")
		+ IGMisc.lineSeparator);

	    fileWriter.write(navWriter.toString());
	}

	fileWriter.write(olWriter.toString());
	fileWriter.write(replacer.makeCode("OL_END_TXT"));
	fileWriter.close();
    }


    /**
     * Determine which IGKey object is referenced by a given value of
     * OL_SORT_KEY.
     * @param sortKeyValue The internal String representation of the value of
     * 	OL_SORT_KEY
     * @return The IGKey object or an acceptable default, <tt>FILE_NAME</tt>
     */
    private static IGKey getSortKey(String sortKeyValue) {
	// default to something safe
	if (sortKeyValue == null)
	    return IGKey.FILE_NAME;
	else if (sortKeyValue.equals("OL_SORT_KEY_TITLE"))
	    return IGKey.TITLE;
	else if (sortKeyValue.equals("OL_SORT_KEY_AUTHOR"))
	    return IGKey.AUTHOR;
	else if (sortKeyValue.equals("OL_SORT_KEY_DESCRIPTION"))
	    return IGKey.DESCRIPTION;
	else if (sortKeyValue.equals("OL_SORT_KEY_KEYWORDS"))
	    return IGKey.DESCRIPTION;
	else if (sortKeyValue.equals("OL_SORT_KEY_FILE_SIZE"))
	    return IGKey.FILE_SIZE;
	else if (sortKeyValue.equals("OL_SORT_KEY_FILE_TYPE"))
	    return IGKey.FILE_TYPE;
	else /* key.equals("OL_SORT_KEY_FILE_NAME") */
	    return IGKey.FILE_NAME;
    }


    /**
     * Strip articles off strings for sorting
     * @param str The input string
     * @param arts The articles to trim off a string
     * @return The input string with a leading article possibly trimmed off
     */
    public static String trimArticles(String str, String[] arts) {
	int index = str.indexOf(' ');

	if (index != -1) {
	    String firstWord = str.substring(0, index);


	    for (int i = 0; i < arts.length; i++)
		if (firstWord.equalsIgnoreCase(arts[i])
			&& index + 1 < str.length())
		    return str.substring(index + 1);
	}

	return str;
    }
}


/**
 * Class to compare to IGFiles based on a given IGKey.
 */
class IGFileComparator implements Comparator {
    /** Key to use for sorting */
    private IGKey sortKey;

    /** Articles to trim off the begining of sort strings */
    private String[] articles;


    /**
     * Construct a comparator that uses the value referenced by a given key for
     * comparison.
     * @param key The IGKey to extract from an IGFile for comparison
     * @param arts The articles to trim off a string
     */
    public IGFileComparator(IGKey key, String[] arts) {
	sortKey = key;
	articles = arts;
    }


    /**
     * Implement the compare method for the Comparator interface
     * @param o1 The first IGFile to compare
     * @param o2 The second IGFile to compare
     * @return An integer greater than, equal to or less than zero based on a
     * 	comparison of two objects
     */
    public int compare(Object o1, Object o2) {
	IGFile file1 = (IGFile) o1;
	IGFile file2 = (IGFile) o2;

	if (sortKey.equals(IGKey.FILE_SIZE)) {
	    long size1 = Long.parseLong(file1.getString(IGKey.FILE_SIZE));
	    long size2 = Long.parseLong(file2.getString(IGKey.FILE_SIZE));

	    return (size1 > size2 ? +1 : (size1 == size2 ? 0 : -1));
	} else {
	    String string1 = OrderedListGen.trimArticles(
			file1.getString(sortKey), articles);
	    String string2 = OrderedListGen.trimArticles(
			file2.getString(sortKey), articles);

	    return (string1.compareToIgnoreCase(string2));
	}
    }
}
